﻿using System.ComponentModel;
using System.Web.UI;
using System.Web.UI.WebControls;

namespace Framework.Web.UI
{
    /// <summary>
    /// 値の比較を行う検証コントロール
    /// </summary>
    [DefaultProperty("Text")]
    [ToolboxData("<{0}:CompareValidator runat=server></{0}:CompareValidator>")]
    public class CompareValidator : WebControl, IInputControl
    {
        /// <summary>
        /// 比較元コントロールのID。指定できるのはIInputControlだけです。
        /// </summary>
        [Category("Custom")]
        [DefaultValue("")]
        public string ControlToValidate { get; set; }

        /// <summary>
        /// 比較先コントロールのID。指定できるのはIInputControlだけです。
        /// </summary>
        [Category("Custom")]
        [DefaultValue("")]
        public string ControlToCompare { get; set; }

        /// <summary>
        /// 比較方法。(Equal, GreaterThanなど）
        /// </summary>
        [Category("Custom")]
        [DefaultValue("GreaterThan")]
        public ValidationCompareOperator Operator
        {
            get { return _operator; }
            set { _operator = value; }
        }
        private ValidationCompareOperator _operator = ValidationCompareOperator.GreaterThan;

        /// <summary>
        /// 比較するときのデータ型。いまのところIntegerとCurrencyしか実装していません。
        /// </summary>
        [Category("Custom")]
        [DefaultValue("Integer")]
        public ValidationDataType Type
        {
            get { return _type; }
            set { _type = value; }
        }
        ValidationDataType _type = ValidationDataType.Integer;

        #region IInputControl メンバ

        public bool IsValid
        {
            get { return true; }
            set { }
        }
        
        public System.Web.UI.Control GetErrorControl()
        {
            return this;
        }

        public bool IsInputed()
        {
            return false;
        }

        public bool IsBlank()
        {
            return false;
        }

        public bool ValidateInput(bool dataTypeOnly)
        {
            if (dataTypeOnly == true) return true;

            if (this.ControlToValidate.IsEmpty() || this.ControlToCompare.IsEmpty())
            {
                return true;
            }

            var cv = (IInputControl)base.NamingContainer.FindControl(this.ControlToValidate);
            var cc = (IInputControl)base.NamingContainer.FindControl(this.ControlToCompare);

            bool ret = true;
            if (cv.IsBlank() || cc.IsBlank())
            {
                //どちらかが空の場合は比較しない。
                ret = true;
            }
            else
            {
                //入力チェック
                if (this.Type == ValidationDataType.Integer)
                {
                    //Intの場合

                    int cvVal;
                    int ccVal;
                    if (int.TryParse(cv.GetInputValue(), out cvVal) && int.TryParse(cc.GetInputValue(), out ccVal))
                    {
                        ret = this.Compare(ccVal, cvVal);
                    }
                }
                else if (this.Type == ValidationDataType.Currency)
                {
                    //Decimalの場合

                    decimal cvVal;
                    decimal ccVal;
                    if (decimal.TryParse(cv.GetInputValue(), out cvVal) && decimal.TryParse(cc.GetInputValue(), out ccVal))
                    {
                        ret = this.Compare(ccVal, cvVal);
                    }
                }
                else if (this.Type == ValidationDataType.Double)
                {
                    //Doubleの場合

                    double cvVal;
                    double ccVal;
                    if (double.TryParse(cv.GetInputValue(), out cvVal) && double.TryParse(cc.GetInputValue(), out ccVal))
                    {
                        ret = this.Compare(ccVal, cvVal);
                    }
                }
            }

            cv.SetValidateResult(ret);
            cc.SetValidateResult(ret);

            return ret;
        }

        /// <summary>
        /// 比較した結果がOperatorを満たす場合はtrueを返します。
        /// </summary>
        /// <param name="cc">比較先</param>
        /// <param name="cv">比較元</param>
        /// <returns></returns>
        private bool Compare(System.IComparable cc, System.IComparable cv)
        {
            bool ret = true;
            switch (this.Operator)
            {
                case ValidationCompareOperator.GreaterThan:
                    ret = (cv.CompareTo(cc) > 0);
                    break;
                case ValidationCompareOperator.GreaterThanEqual:
                    ret = (cv.CompareTo(cc) >= 0);
                    break;
                case ValidationCompareOperator.LessThan:
                    ret = (cv.CompareTo(cc) < 0);
                    break;
                case ValidationCompareOperator.LessThanEqual:
                    ret = (cv.CompareTo(cc) <= 0);
                    break;
                case ValidationCompareOperator.NotEqual:
                    ret = (cv.CompareTo(cc) != 0);
                    break;
                case ValidationCompareOperator.Equal:
                    ret = (cv.CompareTo(cc) == 0);
                    break;
                default:
                    ret = true;
                    break;
            }

            return ret;
        }

        public void SetValidateResult(bool isSuccess)
        {
            //何もしない
        }

        public void ClearInput()
        {
            //何もしない
        }

        public string GetInputValue()
        {
            //何もしない
            return null;
        }

        #endregion
    }
}
