﻿using System;
using System.Linq;
using System.ComponentModel;
using System.Text.RegularExpressions;
using System.Web.UI;

namespace Framework.Web.UI
{
    /// <summary>
    /// マッピング機能を持ったテキストボックスです
    /// </summary>
    [DefaultProperty("MappingName")]
    [ToolboxData("<{0}:MTextBox runat=server></{0}:MTextBox>")]
    public class MTextBox : System.Web.UI.WebControls.TextBox, IMappingControl, IInputControl
    {
        [Category("Custom")]
        [DefaultValue("")]
        public string MappingName {get; set;}

        [Category("Custom")]
        [DefaultValue("")]
        public string DefaultValue
        {
            get { return (string)ViewState["DefaultValue"]; }
            set { ViewState["DefaultValue"] = value; }
        }

        [Category("Custom")]
        [DefaultValue("false")]
        public bool IsRequired { get; set; }

        [Category("Custom")]
        [DefaultValue("NotSet")]
        public InputFormat InputFormat
        {
            get { return _inputFormat; }
            set { _inputFormat = value; }
        }
        private InputFormat _inputFormat = InputFormat.NotSet;

        /// <summary>
        /// 数値の範囲や正規表現を指定します。
        /// [1...9] ⇒1以上9以下  ...9] ⇒ 9以下
        /// (1...9) ⇒1より大きく9より小さい　(1.. ⇒１より大きい
        /// </summary>
        [Category("Custom")]
        [DefaultValue("")]
        public string InputFormatOption { get; set; }

        /// <summary>
        /// Decimalの精度を"15,6"のように指定します。
        /// 
        /// </summary>
        [Category("Custom")]
        [DefaultValue("")]
        public string InputDecimalPrecision { get; set; }

        /// <summary>
        /// S-JISにした時のバイト数の最大値を指定します。
        /// </summary>
        [Category("Custom")]
        public int MaxBytesBySJIS 
        {
            get { return _maxBytesBySJIS; }
            set { _maxBytesBySJIS = value; }
        }
        private int _maxBytesBySJIS = -1;

        /// <summary>
        /// リストキー。指定した場合はマッピング時にTextに変換した値がセットされます。
        /// </summary>
        [Category("Custom")]
        [DefaultValue("")]
        public string ListKey { get; set; }

        /// <summary>
        /// リストソースリクエストイベント。ListKeyが設定されたときに呼び出されます。
        /// </summary>
        public event Framework.Data.ListSrcRequestEventHandler ListSrcRequest;

        /// <summary>
        /// テキスト値をキーにして、テキストをローカライズするかどうか？
        /// </summary>
        public bool LocalizeByText { get; set; }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public MTextBox()
        {
            this.LocalizeByText = false;
        }

        #region IMappingControl メンバ

        /// <summary>
        /// dataにマッピングのデータを追加します。
        /// Key = MappingNameプロパティの値
        /// Value = Textプロパティの値
        /// </summary>
        /// <param name="data"></param>
        public void RequestMappingData(Framework.Data.MappingData data)
        {
            data.Add(this.MappingName, base.Text);
        }

        /// <summary>
        /// dataにKeyがMappingNameのデータが登録されている場合は、Textに該当データのValue値をセットする。
        /// 該当データがない場合はTextにDefaultValue値をセットする。
        /// ListKeyプロパティが空でない場合は、Text値をリストソースのValueとみなして、リストソースのText値に変換する。
        /// LocalizeByTextがtrueの場合は、Text値をローカライズのKeyとみなしてローカライズする。
        /// </summary>
        /// <param name="data"></param>
        public void SetMappingData(Framework.Data.MappingData data)
        {
            string txt = null;

            //データバインド(<%# ･･･ %>)式を展開するためにここでDataBindする
            this.DataBind();

            string val = string.IsNullOrEmpty(this.MappingName) ? null : data[this.MappingName];
            if (val != null)
            {
                txt = val;
            }
            else if (this.DefaultValue != null)
            {
                txt = this.DefaultValue;
            }

            //ListKeyによる変換
            if (this.ListKey.IsNotEmpty() && txt != null)
            {
                if (this.ListSrcRequest != null)
                {
                    //リストソースのリクエスト
                    var e = new Framework.Data.ListSrcRequestEventArgs()
                    {
                        ListKey = this.ListKey,
                    };
                    this.ListSrcRequest(this, e);

                    var row = e.ListSrc.ListSrc.FirstOrDefault(t => t.Value == txt);
                    if (row != null)
                    {
                        txt = row.Text;
                    }
                }
                else
                {
                    txt = ListMgr.GetText(this.ListKey, txt);
                }
            }

            //テキストのローカライズ
            if (this.LocalizeByText == true && txt != null)
            {
                txt = LTextMgr.GetText(txt);
            }

            base.Text = txt;
        }

        #endregion

        #region IInputControl メンバ

        /// <summary>
        /// 入力値の検証結果。おもにUIControllerがこのプロパティを利用します。
        /// </summary>
        public bool IsValid 
        {
            get { return _isValid; }
            set { _isValid = value; }
        }
        private bool _isValid = true;

        /// <summary>
        /// 入力エラーのあるコントロールを返します。本コントロールの場合は自身を返します。
        /// ユーザコントロールなどの複合コントロールの場合は、最初のエラーコントロールを返します。
        /// </summary>
        /// <returns></returns>
        public System.Web.UI.Control GetErrorControl()
        {
            return this;
        }

        /// <summary>
        /// 値が入力されている場合はtrueを返します。
        /// DefaultValueが設定されている場合は、DefaultValueと異なっている場合にtrueを返します。
        /// </summary>
        /// <returns></returns>
        public bool IsInputed()
        {
            if (this.DefaultValue.IsNotEmpty())
            {
                return (this.DefaultValue != base.Text);
            }
            else
            {
                return !this.IsBlank();
            }
        }

        /// <summary>
        /// Textが空文字の場合にtrueを返します
        /// </summary>
        /// <returns></returns>
        public bool IsBlank()
        {
            return string.IsNullOrEmpty(base.Text);
        }

        /// <summary>
        /// 入力値の検証を行います。dataTypeOnlyがtrueの場合は必須や範囲などの検証はなしで、型書式だけを検証します。
        /// </summary>
        /// <param name="dataTypeOnly">trueの場合は型書式だけを検証する</param>
        /// <returns>OKの場合はtrueを返します</returns>
        public bool ValidateInput(bool dataTypeOnly)
        {
            string txt = base.Text;

            if (this.IsBlank())
            {
                //入力値が空の場合は、必須以外はOKとする

                if (dataTypeOnly == true) return true;

                return !(this.IsRequired);
            }

            //文字数チェック(SJISでのバイト数）
            if (this.MaxBytesBySJIS > 0)
            {
                if (System.Text.Encoding.GetEncoding(932).GetByteCount(txt) > this.MaxBytesBySJIS)
                {
                    return false;
                }
            }
            else if (this.MaxLength > 0)
            {
                if (txt.Length > this.MaxLength)
                {
                    return false;
                }
            }

            //書式チェック
            if (this.InputFormat == InputFormat.NotSet)
            {
                return true;
            }
            else if (InputFormat.IsInt())
            {
                //整数の場合

                int i;
                if (int.TryParse(txt, out i) == false)
                {
                    return false;
                }

                if (dataTypeOnly == true)
                {
                    return true;
                }

                if (InputFormat == InputFormat.PInt && i <= 0)
                {
                    return false;
                }

                if (InputFormat == InputFormat.UInt && i < 0)
                {
                    return false;
                }

                return Validate(i, str => str.ToIntInterval());
            }
            else if (InputFormat.IsDouble())
            {
                //実数の場合

                double d;
                if (double.TryParse(txt, out d) == false)
                {
                    return false;
                }

                if (dataTypeOnly == true)
                {
                    return true;
                }

                if (InputFormat == InputFormat.PDouble && d <= 0.0)
                {
                    return false;
                }

                if (InputFormat == InputFormat.UDouble && d < 0.0)
                {
                    return false;
                }

                return Validate(d, str => str.ToDoubleInterval());
            }
            else if (InputFormat.IsDecimal())
            {
                //Decimal の場合

                decimal d;
                if (decimal.TryParse(txt, out d) == false)
                {
                    return false;
                }

                if (dataTypeOnly == false)
                {
                    if (InputFormat == InputFormat.PDecimal && d <= 0m)
                    {
                        return false;
                    }

                    if (InputFormat == InputFormat.UDecimal && d < 0m)
                    {
                        return false;
                    }
                }

                //精度のチェック
                if (this.InputDecimalPrecision.IsNotEmpty() == true)
                {
                    var precStr = this.InputDecimalPrecision.SplitByConma();
                    int prec = precStr[0].ToInt();
                    int scale = precStr[1].ToInt();

                    //最大値のチェック
                    decimal maxValue = 10m.Pow(prec - scale) - 10m.Pow(-scale);
                    if (d > maxValue || d < maxValue.Negate())
                    {
                        return false;
                    }

                    //小数点以下の桁数のチェック
                    if (txt.SubstrAfter(".").Length > scale)
                    {
                        return false;
                    }
                }

                if (dataTypeOnly == true)
                {
                    return true;
                }

                return Validate(d, str => str.ToDecimalInterval());
            }
            else if (InputFormat.IsDate())
            {
                //日付の場合

                DateTime d;
                if (DateTime.TryParse(txt, out d) == false)
                {
                    return false;
                }

                if (dataTypeOnly == true)
                {
                    return true;
                }

                return Validate(d, str => str.ToDateTimeInterval());
            }
            else if (InputFormat.IsRegex())
            {
                if (InputFormatOption.IsEmpty())
                {
                    return true;
                }

                var regex = new Regex(InputFormatOption);
                return regex.IsMatch(txt);
            }

            return true;
        }

        bool Validate<T>(T t, Func<string, Interval<T>> toInterval) where T : IComparable<T>, IComparable
        {
            if (InputFormatOption.IsEmpty() == true)
            {
                return true;
            }

            var intervalStrs = InputFormatOption.SplitBy("or");
            foreach (var intervalStr in intervalStrs)
            {
                var interval = toInterval(intervalStr.Trim());
                if (interval.Contains(t) == true)
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// isSuccessがfalseの場合は背景色を黄色にします。
        /// </summary>
        /// <param name="isSuccess"></param>
        public void SetValidateResult(bool isSuccess)
        {
            this.IsValid = (this.IsValid && isSuccess);
            base.BackColor = this.IsValid ? UIController.ValidateSuccessColor : UIController.ValidateErrorColor;
        }

        /// <summary>
        /// 入力値を空にします。DefaultValueが設定されている場合は入力値をDefaultValueにします
        /// </summary>
        public void ClearInput()
        {
            if (this.DefaultValue != null)
            {
                base.Text = this.DefaultValue;
            }
            else
            {
                base.Text = string.Empty;
            }
        }

        /// <summary>
        /// Text値を返します。
        /// </summary>
        /// <returns></returns>
        public string GetInputValue()
        {
            return base.Text;
        }

        #endregion
    }
}
