﻿
namespace Framework.Data
{
    /// <summary>
    /// アプリケーションレベルの定数を管理/取得するためのクラスです。
    /// 通常の設定ファイル（Web.Config や App.config）以外にXMLファイルやデータベースなどを定数のデータソースとして利用できます。
    /// 定数のデータソースはキャッシュされるので高速に取得できます。
    /// 定数取得の優先順位は １．設定ファイル（Web.Config や App.config）、２．その他のデータソース です。
    /// </summary>
    public class AppSetting
    {
        /// <summary>
        /// XMLファイルをデータソースとして使用する場合は、本プロパティを設定する。
        /// </summary>
        public static string DataSrcXMLFilePath { get; set; }

        /// <summary>
        /// リスナ。
        /// </summary>
        private static IAppSettingRequestListener _listener;

        /// <summary>
        /// Webシステムでない場合のキャッシュオブジェクト
        /// </summary>
        private static AppSettingDataSet _cache;

        private const string CACHE_KEY = "Framework.Data.AppSetting.Cache";

        /// <summary>
        /// アプリ定数のデータソースを返します。
        /// </summary>
        private static AppSettingDataSet.AppSettingDataTable DataSrc
        {
            get
            {
                var ctx = System.Web.HttpContext.Current;

                //キャッシュからデータソースを取得
                var src = (ctx == null) ? _cache : ctx.Cache[CACHE_KEY] as AppSettingDataSet;

                if (src == null)
                {
                    //キャッシュされていない場合

                    src = new AppSettingDataSet();
                    System.Web.Caching.CacheDependency cd = null;

                    if (AppSetting.DataSrcXMLFilePath.IsNotEmpty())
                    {
                        //XMLファイルから読み込む

                        src.AppSetting.ReadXml(AppSetting.DataSrcXMLFilePath);
                        if (ctx != null)
                        {
                            cd = new System.Web.Caching.CacheDependency(AppSetting.DataSrcXMLFilePath);
                        }
                    }
                    else if (_listener != null)
                    {
                        //リスナーにリクエストする

                        var arg = new AppSettingRequestArgs();
                        arg.DataSrc = src;
                        arg.CacheDependency = null;
                        _listener.RequestDataSrc(arg);

                        src = arg.DataSrc;
                        cd = arg.CacheDependency;
                    }

                    //キャッシュに保存
                    if (ctx != null)
                    {
                        ctx.Cache.Insert(CACHE_KEY, src, cd);
                    }
                    else
                    {
                        _cache = src;
                    }
                }

                return src.AppSetting;
            }
        }

        /// <summary>
        /// 設定値を返します。
        /// </summary>
        /// <param name="key"></param>
        /// <returns></returns>
        public static string GetValue(string key)
        {
            //アプリケーション設定ファイル(Web.config や App.config)から探す
            string val = System.Configuration.ConfigurationManager.AppSettings[key];

            if (val == null)
            {
                //見つからない場合は、データソースから探す

                val = DataSrc.FindByKey(key).Value;
            }

            return val;
        }

        /// <summary>
        /// 指定したkeyの接続文字列を返します。
        /// </summary>
        /// <param name="key"></param>
        /// <returns></returns>
        public static string GetConnectionString(string key)
        {
            var conStrSettings = System.Configuration.ConfigurationManager.ConnectionStrings[key];
            if (conStrSettings != null)
            {
                return conStrSettings.ConnectionString;
            }
            return AppSetting.GetValue(key);
        }

        /// <summary>
        /// キャッシュをクリアします。データは次回アクセス時に再作成されます。
        /// </summary>
        public static void ClearCache()
        {
            var ctx = System.Web.HttpContext.Current;
            if (ctx != null)
            {
                ctx.Cache.Remove(CACHE_KEY);
            }
            else
            {
                _cache = null;
            }
        }

        /// <summary>
        /// データソースのリクエストリスナーを登録します。
        /// アプリケーション側で起動時などにIAppSettingRequestListenerを実装したクラスのインスタンスを登録しておくと、必要になったときに呼び出されます。
        /// </summary>
        /// <param name="listener"></param>
        public static void SetListener(IAppSettingRequestListener listener)
        {
            _listener = listener;
        }

    }
}
