﻿using System;
using System.Data;
using System.Data.SqlServerCe;
using Framework.Log;

namespace Framework.DB.SqlServerCe
{
    /// <summary>
    /// SQL Server Compactアクセスのベースクラスです。
    /// </summary>
    public class DbAccessBase
    {
        private readonly string ConnectionString = "";

        private SqlCeConnection _con = null;
        private SqlCeTransaction _tran = null;

        /// <summary>
        /// 現在時刻を返します。2回目以降は最初にの取得と同じ時刻を返します。トランザクション内で更新日付列の値を同じにしたい場合などに使用します。
        /// </summary>
        public DateTime Now
        {
            get
            {
                if (_now == DateTime.MinValue)
                {
                    _now = DateTime.Now;
                }
                return _now;
            }
        }
        private DateTime _now = DateTime.MinValue;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public DbAccessBase()
        {
            this.ConnectionString = DbMgr.GetConnectionString();
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="connectionString">接続文字列</param>
        public DbAccessBase(string connectionString)
        {
            this.ConnectionString = connectionString;
        }

        /// <summary>
        /// コンストラクタ。コネクションを引き継ぎたい場合に使用します。
        /// </summary>
        /// <param name="con">コネクションオブジェクト</param>
        public DbAccessBase(SqlCeConnection con)
        {
            _con = con;
        }

        /// <summary>
        /// コンストラクタ。トランザクションを引き継ぎたい場合に使用します。
        /// </summary>
        /// <param name="tran">トランザクション</param>
        public DbAccessBase(SqlCeTransaction tran)
        {
            //_con = tran.Connection;
            _tran = tran;
        }

        /// <summary>
        /// 自動でコネクション＆トランザクションを開始し、transactionProcedureを呼び出します。処理終了後に自動でコネクション＆トランザクションを終了します。
        /// </summary>
        /// <example>
        /// <code>
        /// ExecuteTransaction(() =>
        /// {
        ///     ExecuteNonQuery(SQL文);
        /// });
        /// </code>
        /// </example>
        /// <param name="transactionProcedure"></param>
        protected void ExecuteTransaction(Action transactionProcedure)
        {
            _con = new SqlCeConnection(this.ConnectionString);
            try
            {
                _con.Open();
                _tran = _con.BeginTransaction();

                //delegate
                transactionProcedure();

                _tran.Commit();
            }
            finally
            {
                if (_tran != null)
                {
                    //_tran.Rollback();
                    _con.Close();

                    _tran = null;
                    _con = null;
                }
            }
        }

        private void ExecuteOpen(SqlCeCommand cmd, Action procedure)
        {
            bool openMyself = false;
 
            if (_con == null)
            {
                _con = new SqlCeConnection(this.ConnectionString);
                _con.Open();
                openMyself = true;
            }
            try
            {
                cmd.Connection = _con;

                if (_tran != null)
                {
                    cmd.Transaction = _tran;
                }

                //delegate
                procedure();
            }
            finally
            {
                if (openMyself)
                {
                    _con.Close();
                    _con = null;
                }
            }
        }


        /// <summary>
        /// procedureの前に接続をオープンし、procedure終了後の接続をクローズします。
        /// 接続文字列はDbMgr.GetConnectionString() で取得されるものを使用します。
        /// </summary>
        /// <param name="procedure"></param>
        protected void ExecuteOpen(Action<SqlCeConnection> procedure)
        {
            var con = new SqlCeConnection(this.ConnectionString);
            con.Open();
            try
            {
                procedure(con);
            }
            finally
            {
                con.Close();
            }
        }

        /// <summary>
        /// ds に sql の実行結果をセットします。
        /// </summary>
        /// <param name="ds">出力先</param>
        /// <param name="sql">SQL文</param>
        /// <param name="values">パラメータ</param>
        protected void FillDataSet(DataSet ds, string sql, params object[] values)
        {
            using (var cmd = this.CreateSqlCommand(sql, values))
            {
                this.FillDataSet(ds, cmd);
            }
        }

        /// <summary>
        /// ds に sql の実行結果をセットします。
        /// </summary>
        /// <param name="ds">出力先</param>
        /// <param name="sql">SQL文</param>
        /// <param name="values">パラメータ</param>
        protected void FillDataSet(DataSet ds, string sql, Framework.Data.MappingData data)
        {
            using (var cmd = this.CreateSqlCommand(sql, data))
            {
                this.FillDataSet(ds, cmd);
            }
        }

        /// <summary>
        /// ds に cmd の実行結果をセットします。
        /// </summary>
        /// <param name="ds">出力先</param>
        /// <param name="cmd">コマンド</param>
        protected void FillDataSet(DataSet ds, SqlCeCommand cmd)
        {
            this.FillDataSet(ds, cmd, 0, 0, "Table1");
        }

        /// <summary>
        ///  ds に cmd の実行結果をセットします。
        /// </summary>
        /// <param name="ds">出力先</param>
        /// <param name="cmd">コマンド</param>
        /// <param name="startRecord">取得開始行</param>
        /// <param name="maxRecords">取得行数</param>
        /// <param name="srcTable">テーブル名</param>
        protected void FillDataSet(DataSet ds, SqlCeCommand cmd, int startRecord, int maxRecords, string srcTable)
        {
            //クエリをログ出力
            LoggerPool.GetLogger().Debug(this.GetQuery4Log(cmd));

            this.ExecuteOpen(cmd
                , delegate
                {
                    using (var adp = new SqlCeDataAdapter(cmd))
                    {
                        adp.Fill(ds, startRecord, maxRecords, srcTable);
                    }
                });
        }

        /// <summary>
        /// tbl に sql の実行結果をセットします。
        /// </summary>
        /// <param name="tbl">出力先</param>
        /// <param name="sql">SQL文</param>
        /// <param name="values">パラメータ</param>
        protected void FillDataTable(DataTable tbl, string sql, params object[] values)
        {
            using (var cmd = this.CreateSqlCommand(sql, values))
            {
                this.FillDataTable(tbl, cmd, 0, 0);
            }
        }

        /// <summary>
        /// tbl に sql の実行結果をセットします。
        /// </summary>
        /// <param name="tbl">出力先</param>
        /// <param name="sql">SQL文</param>
        /// <param name="data">パラメータ</param>
        protected void FillDataTable(DataTable tbl, string sql, Framework.Data.MappingData data)
        {
            using (var cmd = this.CreateSqlCommand(sql, data))
            {
                this.FillDataTable(tbl, cmd, 0, 0);
            }
        }

        /// <summary>
        /// tbl に cmd の実行結果をセットします。
        /// </summary>
        /// <param name="tbl"></param>
        /// <param name="cmd"></param>
        protected void FillDataTable(DataTable tbl, SqlCeCommand cmd)
        {
            this.FillDataTable(tbl, cmd, 0, 0);
        }

        /// <summary>
        ///  tbl に cmd の実行結果をセットします。
        /// </summary>
        /// <param name="ds">出力先</param>
        /// <param name="cmd">コマンド</param>
        /// <param name="startRecord">取得開始行</param>
        /// <param name="maxRecords">取得行数。0を指定した場合は全件を取得します</param>
        /// <param name="srcTable">テーブル名</param>
        protected void FillDataTable(DataTable tbl, SqlCeCommand cmd, int startRecord, int maxRecords)
        {
            //クエリをログ出力
            LoggerPool.GetLogger().Debug(this.GetQuery4Log(cmd));

            this.ExecuteOpen(cmd
                , delegate
                {
                    using (var adp = new SqlCeDataAdapter(cmd))
                    {
                        adp.Fill(startRecord, maxRecords, tbl);
                    }
                });
        }


        /// <summary>
        /// sql の 実行結果を返します。
        /// </summary>
        /// <param name="sql">SQL文</param>
        /// <param name="data">パラメータデータ</param>
        /// <returns> sql の 実行結果</returns>
        protected DataSet GetDataSet(string sql, Framework.Data.MappingData data)
        {
            using (var cmd = CreateSqlCommand(sql, data))
            {
                return this.GetDataSet(cmd);
            }
        }

        /// <summary>
        /// sql の 実行結果を返します。
        /// </summary>
        /// <param name="sql">SQL文</param>
        /// <param name="values">パラメータ</param>
        /// <returns> sql の 実行結果</returns>
        protected DataSet GetDataSet(string sql, params object[] values)
        {
            using (var cmd = CreateSqlCommand(sql, values))
            {
                return this.GetDataSet(cmd);
            }
        }

        /// <summary>
        /// cmd の 実行結果を返します。
        /// </summary>
        /// <param name="cmd"></param>
        /// <returns> sql の 実行結果</returns>
        protected DataSet GetDataSet(SqlCeCommand cmd)
        {
            DataSet ds = new DataSet();
            this.FillDataSet(ds, cmd);

            return ds;
        }

        /// <summary>
        /// sql の 実行結果を返します。
        /// </summary>
        /// <param name="sql">SQL文</param>
        /// <param name="values">パラメータ</param>
        /// <returns>sql の 実行結果</returns>
        protected DataTable GetDataTable(string sql, params object[] values)
        {
            return this.GetDataSet(sql, values).Tables[0];
        }

        /// <summary>
        /// sql の 実行結果を返します。
        /// </summary>
        /// <param name="sql">SQL文</param>
        /// <param name="data">パラメータ</param>
        /// <returns>sql の 実行結果</returns>
        protected DataTable GetDataTable(string sql, Framework.Data.MappingData data)
        {
            return this.GetDataSet(sql, data).Tables[0];
        }

        /// <summary>
        /// sql の 実行結果を返します。
        /// </summary>
        /// <param name="cmd"></param>
        /// <returns>sql の 実行結果</returns>
        protected DataTable GetDataTable(SqlCeCommand cmd)
        {
            return this.GetDataSet(cmd).Tables[0];
        }

        /// <summary>
        /// sql の 実行結果を返します。1行ｘ1列
        /// </summary>
        /// <param name="sql"></param>
        /// <param name="values"></param>
        /// <returns></returns>
        protected object ExecuteScalar(string sql, params object[] values)
        {
            using (var cmd = CreateSqlCommand(sql, values))
            {
                return this.ExecuteScalar(cmd);
            }
        }

        /// <summary>
        /// sql の 実行結果を返します。1行ｘ1列
        /// </summary>
        /// <param name="sql"></param>
        /// <param name="data"></param>
        /// <returns></returns>
        protected object ExecuteScalar(string sql, Framework.Data.MappingData data)
        {
            using (var cmd = CreateSqlCommand(sql, data))
            {
                return this.ExecuteScalar(cmd);
            }
        }

        /// <summary>
        /// sql の 実行結果を返します。1行ｘ1列
        /// </summary>
        /// <param name="cmd"></param>
        /// <returns></returns>
        protected object ExecuteScalar(SqlCeCommand cmd)
        {
            //クエリをログ出力
            LoggerPool.GetLogger().Debug(this.GetQuery4Log(cmd));

            object obj = null;

            this.ExecuteOpen(cmd
                , delegate
                {
                    obj = cmd.ExecuteScalar();
                });

            return obj;
        }

        /// <summary>
        /// 更新系sqlを実行します。
        /// </summary>
        /// <param name="sql"></param>
        /// <param name="values"></param>
        /// <returns></returns>
        protected int ExecuteNonQuery(string sql, params object[] values)
        {
            using (var cmd = CreateSqlCommand(sql, values))
            {
                return this.ExecuteNonQuery(cmd);
            }
        }

        /// <summary>
        /// 更新系sqlを実行します。
        /// </summary>
        /// <param name="sql"></param>
        /// <param name="data"></param>
        /// <returns></returns>
        protected int ExecuteNonQuery(string sql, Framework.Data.MappingData data)
        {
            using (var cmd = CreateSqlCommand(sql, data))
            {
                return this.ExecuteNonQuery(cmd);
            }
        }

        /// <summary>
        /// 更新系sqlを実行します。
        /// </summary>
        /// <param name="cmd"></param>
        /// <returns></returns>
        protected int ExecuteNonQuery(SqlCeCommand cmd)
        {
            int ret = 0;

            //クエリをログ出力
            LoggerPool.GetLogger().Debug(this.GetQuery4Log(cmd));

            this.ExecuteOpen(
                cmd
                , delegate
                {
                    ret = cmd.ExecuteNonQuery();
                });

            return ret;
        }

        /// <summary>
        /// SqlCommand を生成します。
        /// </summary>
        /// <param name="sql"></param>
        /// <param name="values"></param>
        /// <returns></returns>
        protected SqlCeCommand CreateSqlCommand(string sql, params object[] values)
        {
            var cmd = new SqlCeCommand();

            cmd.CommandText = sql;

            if (values.Length > 0)
            {
                string[] bindParams = this.GetBindParams(sql);
                if (bindParams.Length != values.Length)
                {
                    throw new Exception("パラメータの数が一致していません。");
                }

                for (int i = 0; i < bindParams.Length; i++)
                {
                    cmd.Parameters.AddWithValue("@" + bindParams[i], values[i]);
                }
            }

            return cmd;
        }

        /// <summary>
        /// SqlCommand を生成します。
        /// </summary>
        /// <param name="sql"></param>
        /// <param name="data"></param>
        /// <returns></returns>
        protected SqlCeCommand CreateSqlCommand(string sql, Framework.Data.MappingData data)
        {
            var cmd = new SqlCeCommand();

            cmd.CommandText = sql;

            string[] bindParams = this.GetBindParams(sql);
            foreach (string p in bindParams)
            {
                string val = data[p];
                if (val == null)
                {
                    throw new Exception(string.Format("バインドパラメータ({0})に対応するデータがありません。", p));
                }
                else if (val == string.Empty)
                {
                    cmd.Parameters.AddWithValue("@" + p, DBNull.Value);
                }
                else
                {
                    var par = cmd.Parameters.Add("@" + p, SqlDbType.NVarChar);
                    par.Value = data[p];
                }
            }

            return cmd;
        }

        /// <summary>
        /// CreateSqlCommandのHashtable版（動作未確認）
        /// </summary>
        /// <param name="sql"></param>
        /// <param name="data"></param>
        /// <returns></returns>
        protected SqlCeCommand CreateSqlCommand(string sql, System.Collections.Hashtable data)
        {
            var cmd = new SqlCeCommand();

            cmd.CommandText = sql;

            string[] bindParams = this.GetBindParams(sql);
            foreach (string p in bindParams)
            {
                if (data.Contains(p) == false)
                {
                    throw new Exception(string.Format("バインドパラメータ({0})に対応するデータがありません。", p));
                }

                cmd.Parameters.AddWithValue("@" + p, data[p]);
            }

            return cmd;
        }

        /// <summary>
        /// CreateSqlCommandのDataRow版（動作未確認）
        /// </summary>
        /// <param name="sql"></param>
        /// <param name="data"></param>
        /// <returns></returns>
        protected SqlCeCommand CreateSqlCommand(string sql, System.Data.DataRow data)
        {
            var cmd = new SqlCeCommand();

            cmd.CommandText = sql;

            string[] bindParams = this.GetBindParams(sql);
            foreach (string p in bindParams)
            {
                if (data.Table.Columns.Contains(p) == false)
                {
                    throw new Exception(string.Format("バインドパラメータ({0})に対応するデータがありません。", p));
                }

                cmd.Parameters.AddWithValue("@" + p, data[p]);
            }

            return cmd;
        }

        private string[] GetBindParams(string sql)
        {
            return DBUtil.GetBindParams(sql, '@', true);
        }

        /// <summary>
        /// ログ出力用のクエリを返します。
        /// </summary>
        /// <param name="cmd"></param>
        /// <returns></returns>
        public string GetQuery4Log(SqlCeCommand cmd)
        {
            var sql = new System.Text.StringBuilder(cmd.CommandText);

            foreach (SqlCeParameter param in cmd.Parameters)
            {
                var value = param.Value.ToStr();
                if (param.Value is string)
                {
                    value = string.Format("'{0}'", value.Replace("'", "''"));
                }
                else if (param.Value == DBNull.Value)
                {
                    value = "NULL";
                }
                sql.Replace(param.ParameterName, value);
            }

            return Environment.NewLine + "[SQL]" + Environment.NewLine + sql.ToStr();
        }
    }
}
