#!/bin/bash -f
#
# リポジトリの登録確認用スクリプト
# ・指定リポジトリの存在確認
# ・新規リポジトリの作成
#
# Developed by Yasuhiro Fujitsuki 2010/3
#
# 返り値：
# 0 = 成功
# 1 = 失敗
# 9 = ユーザー作成を実行せず
# 99 = Usage 表示/オプション設定ミス
#

###################################
# 返り値定義 / 条件を変える場合、この部分を変更する
SUCCESS=0
FAILED=1
PASSED=9
USAGE=99
###################################
# 環境変数
# 実行ファイル
GREP=/bin/grep
YUM=/usr/bin/yum
YUM_OPS_REPO="--disablerepo=* --enablerepo="
CP="/bin/cp"
RM="/bin/rm"
LS="/bin/ls -l"
GREP="/bin/grep"

# コンフィグファイル 
YUM_REPO_DIR=/etc/yum.repos.d
###################################

# 引数確認
# 引数が指定数無い場合は usage 表示を行い、終了
MODE=$1
REPO=$2
FILE=$3

RESULT=$SUCCESS

if [ "$MODE" = "check" ]; then
  if [ $# -lt 2 ]; then
    echo 
    echo 引数が足りません。
    RESULT=$FAILED
  fi
elif [ "$MODE" = "create" ]; then
  if [ $# -lt 3 ]; then
    echo 
    echo 引数が足りません。
    RESULT=$FAILED
  elif [ ! -e "$FILE" ]; then
    echo 
    echo $FILE が存在しません。
    RESULT=$FAILED
  elif [ -e $YUM_REPO_DIR/$FILE ]; then
    echo
    echo $FILE と同名のファイルが $YUM_REPO_DIR に存在します。
    echo 作業を行いません。
    RESULT=$FAILED
  fi
else
  echo
  echo 第１引数が不正です。
  RESULT=$FAILED
fi

if [ $RESULT -eq $FAILED ]; then
  echo 
  echo usage \> $0 [Mode] [Repo] [File]
  echo
  echo "[Mode]   ... モードの指定"
  echo "             check ... リポジトリが存在するかどうかの確認のみ"
  echo "             create ... リポジトリの確認および新規作成"
  echo "                        *[File]を/etc/yum.repo.にコピーする"
  echo "[Repo]   ... リポジトリ名"
  echo "[File]   ... コピー元リポジトリ(Modeがcheckの場合は不要)"
  echo
  exit $USAGE
fi


# リポジトリの存在確認作業

echo 
echo \#\#\# $REPO リポジトリの存在を確認します
echo 
echo \# ${YUM} ${YUM_OPS_REPO}${REPO} repolist \| $GREP "^$REPO"

${YUM} ${YUM_OPS_REPO}${REPO} repolist | $GREP "^$REPO"

# コマンド成功時は SUCCESS, 失敗時は FAILED をセット
if [ $? -eq 0 ]; then
  echo
  echo \#\#\# $REPO リポジトリは存在します。
  RESULT=$SUCCESS
else
  echo
  echo \#\#\# $REPO リポジトリは存在しません。
  RESULT=$FAILED
fi

# コピーを実施する
if [ $RESULT -eq $FAILED -a  $MODE = "create" ]; then
  echo
  echo \#\#\# $FILE を $YUM_REPO_DIR にコピーします。
  echo
  echo \# $CP $FILE $YUM_REPO_DIR

  $CP $FILE $YUM_REPO_DIR

  echo 
  echo \#\#\# $REPO リポジトリへの登録を確認します。
  echo 
  echo \# ${YUM} ${YUM_OPS_REPO}${REPO} repolist \| $GREP "^$REPO"

  ${YUM} ${YUM_OPS_REPO}${REPO} repolist | $GREP "^$REPO"
  
  # コマンド成功時は SUCCESS, 失敗時は FAILED をセット
  if [ $? -eq 0 ]; then
    echo
    echo \#\#\# $REPO リポジトリ登録を確認しました。
    RESULT=$SUCCESS
  else
    echo
    echo \#\#\# $REPO リポジトリ登録が確認できません。
    echo \#\#\# $YUM_REPO_DIR/$FILE を削除します。
    echo
    echo \# $RM $YUM_REPO_DIR/$FILE

    $RM $YUM_REPO_DIR/$FILE
    
    echo
    echo \# $LS $YUM_REPO_DIR/$FILE

    $LS $YUM_REPO_DIR/$FILE

    echo
    echo \#\#\# $FILE を見直してください。
    RESULT=$FAILED
  fi

fi

exit $RESULT

