#!/bin/bash -f
#
# ユーザー作成スクリプト
# ・ユーザーの作成
# ・ホームディレクトリの作成
# ・/etc/passwd によるユーザー作成確認
# ・ls によるホームディレクトリ有効確認
#
# Developed by Yasuhiro Fujitsuki 2010/3
#
# 返り値：
# 0 = 成功
# 1 = 失敗
# 9 = ユーザー作成を実行せず
# 99 = Usage 表示/オプション設定ミス
#

###################################
# 返り値定義 / 条件を変える場合、この部分を変更する
SUCCESS=0
FAILED=1
PASSED=9
USAGE=99
###################################
# 環境変数
# 実行ファイル
GREP=/bin/grep
WC=/usr/bin/wc 
WC_L="${WC} -l"
USERADD=/usr/sbin/useradd
MKDIR=/bin/mkdir
LS=/bin/ls
XARGS=/usr/bin/xargs
CP=/bin/cp
CHOWN=/bin/chown

# コンフィグファイル 
ETC_PASSWD=/etc/passwd
ETC_GROUP=/etc/group
# スクリプトデフォルトシェル
DEFAULT_SHELL=/bin/bash
# ホームディレクトリスケルトン
HOME_SKEL=/etc/skel
###################################


# 引数確認
# 引数が指定数無い場合は usage 表示を行い、終了
if [ $# -ne 6 ]; then
  echo 引数指定がされていません。
  echo 
  echo usage \> $0 [User Name] [User ID] [GROUP] [NAME] [HOME] [SHELL]
  echo
  echo "[User Name] ... ユーザーアカウント"
  echo "[User ID]   ... ユーザーID"
  echo "[GROUP]     ... グループ名"
  echo "[NAME]      ... 名前\(コメント部分に追加\)"
  echo "[HOME]      ... ホームディレクトリ"
  echo "                既存ディレクトリが存在する場合、そのまま利用します"
  echo "[SHELL]     ... デフォルトシェル"
  echo "                指定シェルが存在しない場合、$DEFAULT_SHELL に置き換えます"
  exit $USAGE
fi

# ユーザー情報の設定
USER_NAME=$1
USER_ID=$2
GROUP=$3
NAME=$4
HOME_DIR=$5
SHELL=$6


# 既存環境の確認
RESULT=$SUCCESS

# ユーザー名の重複確認
if [ `${GREP} ^${USER_NAME}: $ETC_PASSWD | ${WC_L} ` -eq 1 ]; then
  echo ユーザー ${USER_NAME} が既に存在します。
  RESULT=$PASSED
fi

# ユーザーIDの重複確認
if [ `${GREP} :x:${USER_ID}: $ETC_PASSWD | ${WC_L} ` -eq 1 ]; then
  echo ユーザーID ${USER_ID} が既に存在します。
  RESULT=$PASSED
fi

# グループの存在確認
if [ `${GREP} :${GROUP}: $ETC_GROUP | ${WC_L} ` -ne 1 ] && [ `${GREP} ^${GROUP}: $ETC_GROUP | ${WC_L}` -ne 1 ] ; then
  echo グループ ${GROUP} が存在しません。
  RESULT=$PASSED
fi

# シェル設定確認
if [ ! -x ${SHELL} ] ; then
  echo ${SHELL} が見付かりません。
  echo デフォルトシェルを ${DEFAULT_SHELL} に置き換えます。
  SHELL=${DEFAULT_SHELL}
fi

# 作成中止判定
if [ $RESULT -ne $SUCCESS ]; then
  echo ユーザー $USER_NAME の作成は行いませんでした。
  exit $RESULT
fi


#######################################
# 確認表示 
#######################################

echo 次のユーザーを作成します。
echo ユーザー名: $USER_NAME
echo ユーザーID: $USER_ID
echo グループ  : $GROUP
echo 名前\(comment\) : $NAME
echo ホーム : $HOME_DIR
echo シェル : $DEFAULT_SHELL
echo
echo \# ${USERADD} -M -d $HOME_DIR -g $GROUP -u $USER_ID -c "$NAME" -s $SHELL $USER_NAME
${USERADD} -M -d $HOME_DIR -g $GROUP -u $USER_ID -c "$NAME" -s $SHELL $USER_NAME

RESULT=$?

#######################################
# ユーザー作成確認 (/etc/passwdを確認)
#######################################

if [ $RESULT -ne $SUCCESS ]; then
  echo ユーザー作成時、エラーが発生しました
  echo コマンド : ${USERADD}
  echo エラーコード : $RESULT
  exit $FAILED
fi

echo 
echo ユーザー $USER_NAME の作成の確認
echo \# $GREP ^{$USER_NAME}: $ETC_PASSWD
$GREP ^${USER_NAME}: $ETC_PASSWD

#######################################
# ホームディレクトリ作成
#######################################

if [ -e $HOME_DIR ] ; then
  echo ホームディレクトリ $HOME_DIR は存在します。
  echo 作成を省略します。
else
  echo ホームディレクトリの作成をします
  echo \# $MKDIR -p $HOME_DIR
  $MKDIR -p $HOME_DIR
  RESULT=$?
  if [ $RESULT -ne $SUCCESS ]; then
     echo ディレクトリ $HOME_DIR の作成に失敗しました。
     exit $FAILED
  fi

  # スケルトンのコピー
  # スケルトンコピーのエラーでは中止しない
  if [ -d $ETC_SKEL ]; then
    echo \# $LS -A $HOME_SKEL \| $XARGS -i $CP -rv $HOME_SKEL/{} $HOME_DIR
    $LS -rA $HOME_SKEL | $XARGS -i $CP -rv $HOME_SKEL/{} $HOME_DIR
    RESULT=$?

    # コピー確認
    if [ $RESULT -ne $SUCCESS ]; then
      echo $ETC_SKEL : コンフィグファイルのコピーに失敗しました。
    fi
  else
    echo $ETC_SKEL : が存在しません。コピーを省略します。
  fi

  # ファイルの所有権変更
  # ホームディレクトリの所有権変更失敗時はエラー出力
  echo \# $CHOWN -R $USER_NAME:$GROUP $HOME_DIR
  $CHOWN -R $USER_NAME:$GROUP $HOME_DIR

  RESULT=$?
  if [ $RESULT -ne $SUCCESS ]; then
    echo ディレクトリ $HOME_DIR の所有権変更に失敗しました。
    exit $FAILED
  fi

  #######################################
  # ホームディレクトリ作成確認 (lsによる確認)
  #######################################
  echo ホームディレクトリ $HOME_DIR の作成の確認
  echo \# ${LS} -ld $HOME_DIR 
  ${LS} -ld $HOME_DIR
  echo \# ${LS} -lA $HOME_DIR
  ${LS} -lA $HOME_DIR 
  
fi

echo ユーザーの作成を終了しました。
exit $SUCCESS

