/**************************************************************************
 Stats2Excel - open Statistics with Excel from standard Tools menu.
 
 Copyright (C) 2013 Yu Tang
               Home page: http://sourceforge.jp/users/yu-tang/
               Support center: http://sourceforge.jp/users/yu-tang/

 This file is part of plugin for OmegaT.
 http://www.omegat.org/

 License: GNU GPL version 3 or (at your option) any later version.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **************************************************************************/

package jp.sourceforge.users.yutang.omegat.plugin.stats2excel;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import javax.swing.JOptionPane;
import org.omegat.core.Core;
import org.omegat.util.LFileCopy;
import org.omegat.util.Log;
import org.omegat.util.OConsts;
import org.omegat.util.StaticUtils;
import org.omegat.util.StringUtil;

/**
 *
 * @author Yu Tang
 */
public class ExcelModel {
    private static final String CONFIG_DIR_NAME = "stats2excel";
    private static final String WSF_NAME = "Stats2Excel.wsf";
    private static final String LEGACY_STANDARD_TEMPLATE_NAME = "StandardTemplate.xlt";
    private static final String STANDARD_TEMPLATE_NAME = "StandardTemplate.xltx";
    private static final String TXT_NAME = "Stats2Excel.txt";
    private static final String SETTINGS_FILE_NAME = "Stats2Excel.ini";
    private static final File _wsf;
    private static final File _txt;
    private static String windowTitle;
    private static int installedExcelVersion;

    private static enum TemplateVersion {
        NA, CUSTOMIZED, V1, V2
    }

    public static final int INSTALLED_VERSION_NOT_FOUND = -1;

    static {
        File tempDir = new File(System.getProperty("java.io.tmpdir"));
        _wsf = new File(tempDir, WSF_NAME);
        _txt = new File(tempDir, TXT_NAME);
        initInstalledExcelVersion();
    }

    private ExcelModel() { /* singleton */ }

    public static int getInstalledExcelVersion() {
        return installedExcelVersion;
    }

    public static void setWindowTitle(String title) {
        windowTitle = title;
    }

    private static void initInstalledExcelVersion() {
        final int RET_OK = 0;
        int ver = INSTALLED_VERSION_NOT_FOUND;
        try {
            final String EXCEL_VER_KEY = "Excel.Application\\CurVer";
            Command command = new Command();
            String s;
            if (RET_OK == command.execDOS("assoc", EXCEL_VER_KEY)) {
                s = command.getStdout();
                // s's data example）
                // -----------------------------------------------------
                //Excel.Application\\CurVer=Excel.Application.11
                //<-(\r\n)
                // -----------------------------------------------------
                // 末尾に空行が入るので注意。
                // Excel (for Windows) のバージョンと製品名の対応は、以下の通り。
                //
                // Excel.Application.10 = Excel 2000
                // Excel.Application.11 = Excel 2003 <= .xlt for template
                // Excel.Application.12 = Excel 2007 <= introduced .xltx for template
                // Excel.Application.14 = Excel 2010
                // Excel.Application.15 = Excel 2013
                if (s.startsWith(EXCEL_VER_KEY + "=Excel.Application.")) {
                    String versionString = s.substring(
                            (EXCEL_VER_KEY + "=Excel.Application.").length())
                            .replaceAll("\\r\\n", "");
                    ver = Integer.parseInt(versionString);
                }
            }
        } catch (Exception ex) {
            Log.log(ex);
        }
        installedExcelVersion = ver;
    }

    public static void display(String text) throws IOException, InterruptedException {
        final File txt = createTxt(text);
        final File template = getTemplate();

        // try to activate Excel workbook
        boolean succeed = activate();
        if (! succeed) {
            // Maybe Excel does not launch yet. So execute it.
            // 以下の処理は少し時間がかかるため、別スレッドに処理を委譲します。
            new Thread() {
                @Override
                public void run() {
                    try {
                        // launch Excel application
                        Command command = new Command();
                        String dataFilePath = txt.getCanonicalPath();
                        String templateFilePath = template.getCanonicalPath();
                        String job = "launch";
                        int ret = command.execWSF(job, dataFilePath, templateFilePath, windowTitle);

                        if (! command.stderr.isEmpty()) {
                            Log.log("Fail to launch Excel. error(" + ret + "): " + command.stderr);
                        } else {
                            onExcelApplicationQuit(ret);
                        }
                     } catch (IOException ex) {
                        Log.log(ex);
                    } catch (InterruptedException ex) {
                        Log.log(ex);
                    }
                }
            }.start();
        }
    }

    private static File getTemplate() throws IOException {
        // extension .xlt (Microsoft Excel 97-2003 Template) or .xltx
        File parent = new File(StaticUtils.getConfigDir(), CONFIG_DIR_NAME);
        if (! parent.isDirectory()) {
            parent.mkdirs();
        }

        String templateName = installedExcelVersion >= 12 ?
                STANDARD_TEMPLATE_NAME :       // Excel 2007 or later
                LEGACY_STANDARD_TEMPLATE_NAME; // Excel 2003 or earlier
        File template = new File(parent, templateName);

        // check if we should replace the template file with the latest one.
        final Settings settings = new Settings(new File(parent, SETTINGS_FILE_NAME));
        TemplateVersion ver = getTemplateVersion(settings, template);
        switch (ver) {
            case NA:
                // File not found, version not available.
                break;
            case V1:
                // Probably older version is found. Ask user for upgrade.
                String msg = L10n.getUpdateTemplateDialogMessage();
                String title = L10n.getUpdateTemplateDialogTitle();
                int ret = Core.getMainWindow().showConfirmDialog(msg, title,
                        JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE);
                if (ret == JOptionPane.YES_OPTION) {
                    template.delete();
                } else {
                    // set template version setting to CUSTOMIZED
                    settings.setTemplateVersion(TemplateVersion.CUSTOMIZED.name());
                    settings.save();
                }
                break;
            case V2:
                // OK, we have the latest version. Go on.
                break;
            case CUSTOMIZED:
                // Maybe user uses the customized template file. Don't touch it.
                break;
            default:
                // We shouldn't have come here. Something is wrong.
                throw new IOException("Unknown Template Version Error: " + ver.toString());
        }

        if (!template.exists()) {
            InputStream in = ExcelModel.class.getResourceAsStream(templateName);
            try {
                LFileCopy.copy(in, template);

                settings.setTemplateVersion(TemplateVersion.V2.name());
                settings.save();
            } finally {
                in.close();
            }
        }
        return template;
    }

    private static TemplateVersion getTemplateVersion(Settings settings, File template) throws IOException {
        if (!template.isFile()) {
            return TemplateVersion.NA;
        }

        String ver = settings.getTemplateVersion();
        if (StringUtil.isEmpty(ver)) {
            return TemplateVersion.V1;
        } else {
            try {
                return TemplateVersion.valueOf(ver);
            } catch (IllegalArgumentException ex) {
                return TemplateVersion.V1;
            }
        }
    }

    private static File getWSF() throws IOException {
        if (! _wsf.exists()) {
            InputStream in = ExcelModel.class.getResourceAsStream(WSF_NAME);
            try {
                LFileCopy.copy(in, _wsf);
            } finally {
                in.close();
            }
        }
        _wsf.deleteOnExit();
        return _wsf;
    }

    private static File createTxt(String content) throws IOException {
        String LINE_BREAK = "\r\n";
        //String projectName = Core.getProject().getProjectProperties().getProjectName();
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        // write UTF16LE BOM
        out.write(0xFF);
        out.write(0xFE);
        // write projectName
        //out.write((projectName + LINE_BREAK).getBytes(OConsts.UTF16LE));
        // write datetime
        //out.write((DateFormat.getInstance().format(new Date()) + LINE_BREAK + LINE_BREAK).getBytes(OConsts.UTF16LE));
        // write content
        out.write(content.replaceAll("\n", LINE_BREAK).getBytes(OConsts.UTF16LE));
        // get InputStream
        InputStream in = new ByteArrayInputStream(out.toByteArray());

        // output data file
        try {
            LFileCopy.copy(in, _txt);
        } finally {
            in.close();
        }
        _txt.deleteOnExit();
        return _txt;
    }

    public static boolean activate() throws IOException, InterruptedException {
        final Command command = new Command();
        final String job = "activate";
        final int ret = command.execWSF(job, windowTitle);
        return (ret == 0);
    }

    private static void onExcelApplicationQuit(final int returnCode) {
        deleteDataFile();
        deleteWSF();
    }

    public static void close() {
        // データファイルを削除すれば、WSF の方で Excel を終了してくれる。
        deleteDataFile();
    }

    private static void deleteDataFile() {
        if (_txt.exists()) {
            _txt.delete();
        }
    }

    private static void deleteWSF() {
        if (_wsf.exists()) {
            _wsf.delete();
        }
    }

    // バッファあふれ非対応のため、少量のテキスト（だいたい 500文字ていど）が
    // 予想される場合のみ利用してください。
    // また同期実行です。プロセスの終了を待機してから制御を返します。
    protected static class Command {

        private int exitCode = 0;
        private String stdout = "";
        private String stderr = "";

        public int getExitCode() {
            return exitCode;
        }

        public String getStdout() {
            return stdout;
        }

        public String getStderr() {
            return stderr;
        }

        public int exec(String... command) 
                throws IOException, InterruptedException {
            return startProcessAndWait(Arrays.asList(command));
        }

        public int execDOS(String... command) 
                throws IOException, InterruptedException {
            List<String> commands = new ArrayList<String>(command.length + 2);
            commands.add("cmd.exe");
            commands.add("/c");
            commands.addAll(Arrays.asList(command));
            
            return startProcessAndWait(commands);
        }

        public int execWSF(String job, String... command) 
                throws IOException, InterruptedException {
            String script = getWSF().getCanonicalPath();
            List<String> commands = new ArrayList<String>(command.length + 4);
            commands.add("cscript.exe");
            commands.add("//nologo");
            commands.add("//Job:" + job);
            commands.add(script);
            commands.addAll(Arrays.asList(command));

            return startProcessAndWait(commands);
        }

        private int startProcessAndWait(List<String> command) 
                throws IOException, InterruptedException {
            ProcessBuilder pb = new ProcessBuilder(command);
            Process process = pb.start();
            exitCode = process.waitFor();  // 0: succeed
            stdout = getString(process.getInputStream());
            stderr = getString(process.getErrorStream());
            return exitCode;
        }

        private String getString(InputStream is) throws IOException {
            byte[] b = new byte[1024];
            int size = is.read(b);
            if (size > 0) {
                return new String(b, 0, size);
            } else {
                return "";
            }
        }

    }
}
