/**************************************************************************
 Moenizer - Allow to set background image for OmegaT.
 
 Copyright (C) 2013-2014 Yu Tang
               Home page: http://sourceforge.jp/users/yu-tang/
               Support center: http://sourceforge.jp/users/yu-tang/pf/Moenizer/

 This file is part of plugin for OmegaT.
 http://www.omegat.org/

 License: GNU GPL version 3 or (at your option) any later version.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **************************************************************************/

package jp.sourceforge.users.yutang.omegat.plugin.moenizer;

import com.sun.java.swing.plaf.windows.WindowsMenuBarUI;
import com.vlsolutions.swing.docking.AutoHideButtonPanel;
import com.vlsolutions.swing.docking.DockView;
import com.vlsolutions.swing.docking.DockViewTitleBar;
import com.vlsolutions.swing.docking.Dockable;
import com.vlsolutions.swing.docking.DockableState;
import com.vlsolutions.swing.docking.DockingDesktop;
import com.vlsolutions.swing.docking.SplitContainer;
import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Graphics;
import java.awt.SystemColor;
import java.awt.image.BufferedImage;
import java.util.ArrayList;
import java.util.EnumMap;
import java.util.EnumSet;
import java.util.Map;
import javax.swing.JComponent;
import javax.swing.JEditorPane;
import javax.swing.JFrame;
import javax.swing.JMenuBar;
import javax.swing.JPanel;
import javax.swing.JRootPane;
import javax.swing.JScrollPane;
import javax.swing.JTextPane;
import javax.swing.SwingUtilities;
import javax.swing.plaf.ComponentUI;
import javax.swing.text.Highlighter;
import org.omegat.core.Core;
import org.omegat.core.CoreEvents;
import org.omegat.core.events.IProjectEventListener;
import org.omegat.util.Log;
import org.omegat.util.gui.UIThreadsUtil;

/**
 *
 * @author Yu-Tang
 */
public class MoeUI implements IProjectEventListener {

    public static final float OPACITY_OPAQUE = 1.0f;
    public static final float OPACITY_HALF = 0.5f;

    private static MoeUI moeUI;

    private final JFrame frame;
    private final JRootPane rootPane;
    private final Container contentPane;
    private final JMenuBar menuBar;
    private final DockingDesktop desktop;
    private final JPanel statusBar;
    private final AutoHideButtonPanel buttonPanel;
    private final Map<Parts, ComponentUI> uis;

    private boolean makeTransparentMenubar = true;
    private boolean makeTransparentButtonPanel = true;
    private boolean makeTransparentStatusbar = true;
    private boolean makeTransparentPaneTitlebar = true;
    private boolean isEditorTransparent = false;
    
    public enum Parts {
        MainWindow,
        // Excluding parts for transparency
        MenuBar, StatusBar, ButtonPanel, PaneTitleBar,
        // Built-in panes
        EDITOR, MATCHES, GLOSSARY, NOTES, COMMENTS, MACHINE_TRANSLATE,
        DICTIONARY, MULTIPLE_TRANS,
        // INSTANT_START (Another view of EDITOR)
        INSTANT_START;

        public static Parts[] configullables() {
            Parts[] values = Parts.values();
            int length = values.length - 1; // leave INSTANT_START out
            Parts[] ret = new Parts[length];
            System.arraycopy(values, 0, ret, 0, length);
            return ret;
        }
    }

    static {
        moeUI = null;
    }

    // Singleton. Not allow to Instanciate from outside. Use getInstance() to get instance.
    private MoeUI() {
        UIThreadsUtil.mustBeSwingThread();

        frame = Core.getMainWindow().getApplicationFrame();
        rootPane = frame.getRootPane();
        contentPane = frame.getContentPane();
        menuBar = frame.getJMenuBar();
        statusBar = getJPanel(contentPane);
        desktop = getDockingDesktop(contentPane);
        buttonPanel = getButtonPanel(desktop);
        uis = new EnumMap<Parts, ComponentUI>(Parts.class);
    }

    public static MoeUI getInstance() {
        if (moeUI == null) {
            moeUI = new MoeUI();
        }
        return moeUI;
    }

    public void setBackground(Parts target, BufferedImage image) {
        UIThreadsUtil.mustBeSwingThread();

        if (target == Parts.MainWindow) {
            getRootPaneUI().setBackgroundImage(image);

        } else if (target.compareTo(Parts.EDITOR) >= 0) {
            getTextPaneUI(target).setBackgroundImage(image);
            if (target == Parts.EDITOR) {
                getTextPaneUI(Parts.INSTANT_START).setBackgroundImage(image);
            }
        }
    }

    public void setBackground(Parts target, Color color) {
        UIThreadsUtil.mustBeSwingThread();

        if (target == Parts.MainWindow) {
            getRootPaneUI().setBackgroundColor(color);

        } else if (target.compareTo(Parts.EDITOR) >= 0) {
            getTextPaneUI(target).setBackgroundColor(color);
            if (target == Parts.EDITOR) {
                getTextPaneUI(Parts.INSTANT_START).setBackgroundColor(color);
            }
        }
    }

    public void setOpacity(Parts target, float opacity) {
        UIThreadsUtil.mustBeSwingThread();

        if (target == Parts.MainWindow) {
            getRootPaneUI().setOpacity(opacity);

        } else {
            // noop
            // メインウィンドウ以外の個別ペインは不透明度 1.0f 固定とします。
            // 半透過にすると、残像が残るためです。
            Log.log("warning: setOpacity(" + target.name() + ", " + opacity
                    + ") is ignored. Opacity is available only for MainWindow.");
        }  
    }

    // MainWindow と EDITOR に画像や色を割り当てる場合のみ使用します
    public void transparent(Parts target, EnumSet<Parts> exclude) {
        switch (target) {
            case MainWindow:
                if (exclude != null) {
                    makeTransparentMenubar = !exclude.contains(Parts.MenuBar);
                    makeTransparentButtonPanel = !exclude.contains(Parts.ButtonPanel);
                    makeTransparentStatusbar = !exclude.contains(Parts.StatusBar);
                    makeTransparentPaneTitlebar = !exclude.contains(Parts.PaneTitleBar);
                }
                transparent(exclude);
                break;

            case EDITOR:
                UIThreadsUtil.mustBeSwingThread();
                removeInstantStartBgColor();

                // エディター（インスタントスタートではない方） ペインに個別に
                // 背景を設定するためには、結局プロジェクトのロード時に現在の
                // エフェクト設定で再度エディターを取得して setUI し直す必要がある。
                // この行が実行される時点では、インスタントスタートと編集画面の
                // どちらのモードか不明なので、いったん編集画面の設定を試みて、
                // 失敗したらプロジェクトのロード時に再試行するクロージャを投げる。
                final JEditorPane editor = getJEditorPaneFromEditorView();
                final MoeTextPaneUI sourceUI = getTextPaneUI(Parts.INSTANT_START);
                if (editor == null) {
                    // Editor is not available yet. Wait for project open event.
                    DocumentEditorUIRunner editorUIRunner = new DocumentEditorUIRunner();
                    editorUIRunner.setBackgroundImage(sourceUI.getBackgroundImage());
                    editorUIRunner.setBackgroundColor(sourceUI.getBackgroundColor());
                    CoreEvents.registerProjectChangeListener(editorUIRunner);
                } else {
                    // Editor is found. Set UI and migrate highlighter.
                    MoeTextPaneUI ui = getTextPaneUI(Parts.EDITOR);
                    ui.setBackgroundImage(sourceUI.getBackgroundImage());
                    ui.setBackgroundColor(sourceUI.getBackgroundColor());
                    Highlighter h = editor.getHighlighter();  // save old highlighter
                    editor.setUI(ui);
                    editor.setHighlighter(h);                 // restore highlighter
                }
        }
    }

    private void transparent(EnumSet<Parts> exclude) {
        UIThreadsUtil.mustBeSwingThread();

        if (makeTransparentMenubar) {
            transparent(menuBar);
        } else {
            int marginTop = menuBar.getHeight();
            if (!makeTransparentPaneTitlebar) {
                marginTop += getDockViewTitleBarHeight(desktop);
            }
            MoeRootPaneUI ui = (MoeRootPaneUI) uis.get(Parts.MainWindow);
            ui.setMarginTop(marginTop);
        }
        transparentRecursive(contentPane, exclude);
        if (!exclude.contains(Parts.EDITOR)) {
            removeInstantStartBgColor();
        }

        // try to make transparent editor pane
        if (!exclude.contains(Parts.EDITOR) && !isEditorTransparent) {
            isEditorTransparent = transparentEditor();
            if (!isEditorTransparent) {
                CoreEvents.registerProjectChangeListener(this);
            }
        }

        frame.repaint();
    }

    @Override
    public void onProjectChanged(PROJECT_CHANGE_TYPE eventType) {
        //@@TODO Runnerに移行
        switch (eventType) {
        case CREATE:
        case LOAD:
            // ここですぐエディターの透過処理をしてもうまく適用されないので、
            // 遅延処理する。
            SwingUtilities.invokeLater(new Runnable() {
                @Override
                public void run() {
                    CoreEvents.unregisterProjectChangeListener(MoeUI.this);
                    isEditorTransparent = transparentEditor();
                }
            });

            // 一回 Editor の透過処理をしたら、透過状態が維持されるので、二回目
            // 以降は、透過処理は不要。
            //CoreEvents.unregisterProjectChangeListener(this); // ここで発行すると、スレッドエラーになるので注意
            break;
        case CLOSE:
            break;
        }
    }

    private int getDockViewTitleBarHeight(DockingDesktop desktop) {
        ArrayList<Dockable> dockables = desktop.getContext()
                .getDockablesByState(desktop, DockableState.STATE_DOCKED);
        if (!dockables.isEmpty()) {
            Dockable dockable = dockables.get(0);

            // DockView
            Container container = dockable.getComponent().getParent();
            DockView view = (DockView) container;

            // TitleBar
            DockViewTitleBar titleBar = view.getTitleBar();
            return titleBar.getHeight();            
        } else {
            return 0;
        }
    }

    private boolean transparentEditor() {
        // Editor に表示されるドキュメントは、以下のケースでまったく異なる。
        //
        // 1. インスタントスタートガイドが表示されている場合（初期状態）
        //    -> view = JTextPane, HTMLDocument
        // 2. プロジェクトをロードまたは新規作成して、分節編集画面が表示されている場合
        //    -> view = JEditorPane, DefaultStyledDocument
        //
        // そのため、2 のタイミングで透過処理をやり直す必要がある。
        // このメソッドは、2 の透過処理専用。

        UIThreadsUtil.mustBeSwingThread();

        JEditorPane editor = getJEditorPaneFromEditorView();
        if (editor == null) {
            return false;
        }
        
        /* ここで JEditorPane に半透明の背景色を設定すると、テキストの選択（反転）
         * などの色の変化が正常に更新されず色残りしてしまう。
         * そのため、JEditorPane への処理では単純に背景を透明にして、元画像自体で
         * 色味を調整しておく方針とする。
         * 
        int alpha = 100; // transparent <- 0...255 -> opaque
        Color color = SystemColor.menu;
        editor.setBackgroundImage(new Color( color.getRGB() & 0xffffff | 100 << 24, true));
        * */
        editor.setOpaque(false);
        isEditorTransparent = true;
        frame.repaint();
        return true;
    }

    private JEditorPane getJEditorPane(Parts target) {
        /*
        dockable.getDockKey().getKey()
          dockable.getComponent().class.name
          dockable.getComponent().getViewport().getView().class.name
        -------------
        EDITOR
          org.omegat.gui.main.DockablePanel
        MATCHES
          org.omegat.gui.main.DockableScrollPane
          org.omegat.gui.matches.MatchesTextArea
        GLOSSARY
          org.omegat.gui.main.DockableScrollPane
          org.omegat.gui.glossary.GlossaryTextArea
        NOTES
          org.omegat.gui.main.DockableScrollPane
          org.omegat.gui.notes.NotesTextArea
        COMMENTS
          org.omegat.gui.main.DockableScrollPane
          org.omegat.gui.comments.CommentsTextArea
        MACHINE_TRANSLATE
          org.omegat.gui.main.DockableScrollPane
          org.omegat.gui.exttrans.MachineTranslateTextArea
        DICTIONARY
          org.omegat.gui.main.DockableScrollPane
          org.omegat.gui.dictionaries.DictionariesTextArea
        MULTIPLE_TRANS
          org.omegat.gui.main.DockableScrollPane
          org.omegat.gui.multtrans.MultipleTransPane

                xxxTextArea は、基本的に全部 JEditorPane の拡張。
                MULTIPLE_TRANS も、同じ。
                EntryInfoThreadPane -> EntryInfoPane -> JTextPane -> JEditorPane という感じ。
                EDITOR と INSTANT_START だけちょっと特殊なので、分岐処理しておく。
        */
        switch (target) {
            case EDITOR:
                return getJEditorPaneFromEditorView();
            case INSTANT_START:
                return getJTextPaneFromInstantStartView();
            default:
                // エディター以外の場合
                final String PANE_NAME = target.name();

                Dockable dockable = desktop.getContext().getDockableByKey(PANE_NAME);
                if (dockable != null) {
                    JScrollPane sp = (JScrollPane) dockable.getComponent();
                    return (JTextPane) sp.getViewport().getView();                
                }                
        }
        return null;
    }

    private void transparent(JMenuBar menuBar) {
        //menuBar.setUI ( new BasicMenuBarUI () {
        // 上記だと初期表示時に、アクティブメニューがアクティブで描画されない。
        // マウス通過などのタイミングで再描画された際にアクティブカラーになるが、
        // 見栄えが悪いので、Windows 用の UI を指定する。
        // アクティブメニューがアクティブで描画されない問題はこれで解決するが、
        // 変わりに Mac や *nix など別 L&F 環境下では違和感があるかもしれない。
        menuBar.setUI ( new WindowsMenuBarUI () {
            @Override
            public void paint ( Graphics g, JComponent c ) {
                int alpha = 100; // transparent <- 0...255 -> opaque //@@TODO sdjust value
                Color oldColor = g.getColor();
                Color color = SystemColor.menu;
                g.setColor ( new Color( color.getRGB() & 0xffffff | alpha << 24, true));
                g.fillRect ( 0, 0, c.getWidth (), c.getHeight () );
                g.setColor ( oldColor ); // restore
            }
        } );
        menuBar.setOpaque(false);
    }
    
    private void transparentRecursive(Component component, EnumSet<Parts> exclude) {
        // StatusBar
        if (exclude.contains(Parts.StatusBar) && statusBar.equals(component)) {
            return;
        }
        
        // ButtonPanel
        if (exclude.contains(Parts.ButtonPanel) && buttonPanel.equals(component)) {
            return;
        }

        if (component instanceof JComponent) {
            JComponent c = (JComponent) component;
            if (c.isShowing() && c.isOpaque()) {
                if (exclude.contains(Parts.EDITOR) && c instanceof JTextPane &&
                        c.equals(getJTextPaneFromInstantStartView())) {
                    // skip INSTANT_START
                } else if (exclude.contains(Parts.EDITOR) && c instanceof JEditorPane &&
                        c.equals(getJEditorPaneFromEditorView())) {
                    // skip EDITOR
                } else if (exclude.contains(Parts.MATCHES) && c instanceof JEditorPane &&
                        c.equals(getJEditorPane(Parts.MATCHES))) {
                    // skip MATCHES
                } else if (exclude.contains(Parts.GLOSSARY) && c instanceof JEditorPane &&
                        c.equals(getJEditorPane(Parts.GLOSSARY))) {
                    // skip GLOSSARY
                } else if (exclude.contains(Parts.NOTES) && c instanceof JEditorPane &&
                        c.equals(getJEditorPane(Parts.NOTES))) {
                    // skip NOTES
                } else if (exclude.contains(Parts.COMMENTS) && c instanceof JEditorPane &&
                        c.equals(getJEditorPane(Parts.COMMENTS))) {
                    // skip COMMENTS
                } else if (exclude.contains(Parts.MACHINE_TRANSLATE) && c instanceof JEditorPane &&
                        c.equals(getJEditorPane(Parts.MACHINE_TRANSLATE))) {
                    // skip MACHINE_TRANSLATE
                } else if (exclude.contains(Parts.DICTIONARY) && c instanceof JEditorPane &&
                        c.equals(getJEditorPane(Parts.DICTIONARY))) {
                    // skip DICTIONARY
                } else if (exclude.contains(Parts.MULTIPLE_TRANS) && c instanceof JEditorPane &&
                        c.equals(getJEditorPane(Parts.MULTIPLE_TRANS))) {
                    // skip MULTIPLE_TRANS
                } else {
                    c.setOpaque(false);
                }
            }
        }

        if (component instanceof Container) {
            Container container = (Container) component;
            for (Component c: container.getComponents()) {
                transparentRecursive(c, exclude);
            }
        }

        if (component instanceof DockingDesktop) {
            transparentRecursive((DockingDesktop) component, exclude);
        }
    }
        
    private void transparentRecursive(DockingDesktop desktop, EnumSet<Parts> exclude) {
        //DockingPanel dockingPanel = dockingDesktop.getDockingPanel(); // Scoping NG
        // DockingPanel にアクセスできないので、下位要素から上にさかのぼって処理する。
        // 階層的には、こんな感じになっている。
        // -----------------------
        // DockingDesktop
        //  + DockingPanel                  <-- splitContainer.getParent()
        //    + splitContainer(HORIZONTAL)  <-- splitContainer.getParent()
        //      + splitContainer(VERTICAL)  <-- DockView.getParent()
        //        + DockView                <-- Dockable.getParent()
        //          + Dockable              <-- DockableState.getDockable()
        ArrayList<Dockable> dockables = desktop.getContext()
                .getDockablesByState(desktop, DockableState.STATE_DOCKED);
        for (Dockable d : dockables) {
            Parts part = Parts.valueOf(d.getDockKey().getKey());
            if (!exclude.contains(part)) {
                transparentRecursive(d);
            }
        }
    }

    private void transparentRecursive(Dockable dockable) {
        // DockView
        Container container = dockable.getComponent().getParent();
        DockView view = (DockView) container;
        if (view.isOpaque()) {
            view.setOpaque(false);
        }

        // TitleBar
        if (makeTransparentPaneTitlebar) {
            DockViewTitleBar titleBar = view.getTitleBar();
            if (titleBar.isOpaque()) {
                titleBar.setOpaque(false);
            }
            titleBar.setUI(new MoeDockViewTitleBarUI(titleBar));
        }

        // SplitContainer(VERTICAL)
        container = container.getParent();
        if (container == null) {
            return;
        } else if (container.isOpaque()) {
            ((SplitContainer) container).setOpaque(false);
        }

        // SplitContainer(HORIZONTAL)
        container = container.getParent();
        if (container == null) {
            return;
        } else if (container.isOpaque()) {
            ((SplitContainer) container).setOpaque(false);
        }
    }

    private void removeInstantStartBgColor() {
        // お手軽スタートは、以下のような HTML で背景色が指定されている。
        // <body ... bgcolor="white" ...>
        // そのため、コンポーネント自体を透過にしても、HTML Body 背景色の
        // 白指定が効いて透過にならない。そこで、背景色指定を削除する。
        JTextPane ep = getJTextPaneFromInstantStartView();
        if (ep != null) {
                ep.setText(ep.getText().replace(" bgcolor=\"white\"", ""));
                ep.setCaretPosition(0);
        }
    }
    
    private DockingDesktop getDockingDesktop(Container container) {
        for (Component c: container.getComponents()) {
            if (c instanceof DockingDesktop) {
                return (DockingDesktop) c;
            }
        }
        return null;
    }

    private JPanel getJPanel(Container contentPane) {
        // contentPane
        //   + javax.swing.JPanel // ステータスバーの実体
        //   + com.vlsolutions.swing.docking.DockingDesktop // メニューバーの下からステータスバーの上までの領域
        for (Component c: contentPane.getComponents()) {
            if (c instanceof JPanel) {
                return (JPanel) c;
            }
        }
        return null;
    }

    private AutoHideButtonPanel getButtonPanel(DockingDesktop desktop) {
        // desktop
        //   + com.vlsolutions.swing.docking.AutoHideExpandPanel // 非表示
        //   + com.vlsolutions.swing.docking.DockingDesktop$1    // ボタンパネルの上の領域、ボタンパネルが非表示の場合は下のパネルと同じサイズ
        //   + javax.swing.JPanel  // ボタンパネルを含む領域 <= this

        JPanel panel = null;
        for (Component c: desktop.getComponents()) {
            // AutoHideExpandPanel も JPanel 派生のため、instanceof で判定すると
            // 先にそっちを掴むので注意
            if (c.getClass().getName().equals("javax.swing.JPanel")) {
                panel = (JPanel) c;
                break;
            }
        }

        // panel
        //   + com.vlsolutions.swing.docking.AutoHideButtonPanel // 非表示、未使用
        //   + com.vlsolutions.swing.docking.AutoHideButtonPanel // 非表示、未使用
        //   + com.vlsolutions.swing.docking.AutoHideButtonPanel // ボタンがあれば表示（isShowing()、isVisible() ともに true） <= this
        //   + com.vlsolutions.swing.docking.AutoHideButtonPanel // 非表示、未使用
        //   + com.vlsolutions.swing.docking.DockingPanel  // ボタンパネルの上の領域

        // ここでは 3 番目の AutoHideButtonPanel を取得したい。
        // 状態が不明（非表示の場合は、他の AutoHideButtonPanel と識別できない）
        // なので、とりあえずインデックスで決め打ちする。
        return panel == null ? null : (AutoHideButtonPanel) panel.getComponent(2);
    }

    private JEditorPane getJEditorPaneFromEditorView() {
        JEditorPane pane = getJEditorPaneFromEditor();
        return pane instanceof JTextPane ? null : pane;
        
    }

    private JTextPane getJTextPaneFromInstantStartView() {
        JEditorPane pane = getJEditorPaneFromEditor();
        return pane instanceof JTextPane ? (JTextPane) pane : null;
        
    }

    private JEditorPane getJEditorPaneFromEditor() {
        Dockable dockable = desktop.getContext().getDockableByKey(Parts.EDITOR.name());
        if (dockable != null) {
            JScrollPane sp;

            // OmegaT 3.1.0 or later
            if (dockable.getComponent() instanceof JPanel) {
                // dockable.getComponent() => org.omegat.gui.main.DockablePanel
                //                            extends JPanel implements Dockable
                JPanel jp = (JPanel) dockable.getComponent();
                sp = (JScrollPane) jp.getComponent(0);

            // less than OmegaT 3.1.0
            } else {
                sp = (JScrollPane) dockable.getComponent();
            }

            return (JEditorPane) sp.getViewport().getView();
        }
        return null;
    }

    private MoeRootPaneUI getRootPaneUI() {
        MoeRootPaneUI ui;
        Parts target = Parts.MainWindow;
        if (uis.containsKey(target)) {
            ui = (MoeRootPaneUI) uis.get(target);
        } else {
            ui = new MoeRootPaneUI();
            rootPane.setUI(ui);
            uis.put(target, ui);
        }
        return ui;
    }

    private MoeTextPaneUI getTextPaneUI(Parts target) {
        MoeTextPaneUI ui;
        if (uis.containsKey(target)) {
            ui = (MoeTextPaneUI) uis.get(target);
        } else {
            ui = new MoeTextPaneUI();
            JEditorPane pane = getJEditorPane(target);
            if (pane != null) { // EDITOR の場合は null になることがあるため
                Highlighter h = pane.getHighlighter();  // save old highlighter
                pane.setUI(ui);
                pane.setHighlighter(h);                 // restore highlighter
                if (!pane.isOpaque()) {
                    pane.setOpaque(true);
                    if (target == Parts.EDITOR) {
                        isEditorTransparent = false;
                    }
                }                
            }
            uis.put(target, ui);
        }
        return ui;
    }

    /**
     * エディター専用の UI です。プロジェクトの開始時にエディターに自身をUI と 
     * して自動設定します。
     * 呼び出し側では、このクラスのインスタンスをプロジェクトイベントのリスナー
     * に登録しておく必要があります。
     */
    private class DocumentEditorUIRunner extends MoeTextPaneUI
            implements IProjectEventListener, Runnable {

        @Override
        public void run() {
            // 実行は一回きりで良いので、リスナー登録を解除する
            CoreEvents.unregisterProjectChangeListener(this);

            // エディターに UI（自身）を設定し、ハイライターを復元する
            JEditorPane editor = getJEditorPaneFromEditor();
            Highlighter h = editor.getHighlighter();  // save old highlighter
            editor.setUI(this);
            editor.setHighlighter(h);                 // restore highlighter

            // エディターが透明の場合は不透過に設定する
            if (!editor.isOpaque()) {
                editor.setOpaque(true);
                isEditorTransparent = false;
            }
        }

        @Override
        public void onProjectChanged(PROJECT_CHANGE_TYPE eventType) {
            switch (eventType) {
                case CREATE:
                case LOAD:
                    // ここですぐエディターの透過処理をしてもうまく適用されないので、
                    // 遅延処理する。
                    SwingUtilities.invokeLater(this);
            }
        }
        
    }
}
