/**************************************************************************
 FolderMenu - easy access to project folders from menu.
 
 Copyright (C) 2015 Yu Tang
               Home page: http://osdn.jp/users/yu-tang/

 This file is part of plugin for OmegaT.
 http://www.omegat.org/

 License: GNU GPL version 3 or (at your option) any later version.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **************************************************************************/

package jp.osdn.users.yutang.omegat.plugin.foldermenu;

import java.awt.Desktop;
import java.awt.Insets;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.util.Arrays;
import java.util.Comparator;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.MenuElement;
import javax.swing.MenuSelectionManager;
import javax.swing.SwingConstants;
import javax.swing.event.MenuEvent;
import javax.swing.event.MenuKeyEvent;
import javax.swing.event.MenuKeyListener;
import javax.swing.event.MenuListener;
import jp.osdn.users.yutang.omegat.plugin.foldermenu.filepreview.FilePreview;
import org.omegat.core.Core;
import org.omegat.util.Log;
import org.omegat.util.OStrings;
import org.omegat.util.StaticUtils;
import org.openide.awt.Mnemonics;

/**
 *
 * @author Yu Tang
 */
public class Listeners {

    private static final FileFilter fileFilter = new FileFilter() {
        @Override
        public boolean accept(File file) {
            // except dot started named files (i.e. ".svn") and hidden files
            return !file.getName().startsWith(".") && !file.isHidden();
        }
    };

    private static final Comparator<File> comparator = new Comparator<File>() {
        @Override
        public int compare(File f1, File f2) {
            if (f1.isFile() == f2.isFile()) {
                return f1.getName().compareToIgnoreCase(f2.getName());
            } else {
                return f1.isFile() ? 1 : -1;
            }
        }
    };

    private static final ActionListener actionListener = new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
            open((JMenuItem) e.getSource());
        }
    };

    /** Use for folder only */
    private static final MenuKeyListener menuKeyListener = new MenuKeyListener() {
            @Override
            public void menuKeyTyped(MenuKeyEvent e) { /* do nothing */ }

            @Override
            public void menuKeyPressed(MenuKeyEvent e) {
                if (e.getKeyCode() == KeyEvent.VK_ENTER) {
                    MenuSelectionManager manager = e.getMenuSelectionManager();
                    MenuElement[] selectedPath = manager.getSelectedPath();
                    MenuElement selection = selectedPath[selectedPath.length-1];
                    if (selection instanceof RunnableMenu) {
                        // ignore
                    } else if (selection instanceof JMenu) {
                        JMenu menu = (JMenu) selection;
                        if (menu.isEnabled()) {
                            manager.clearSelectedPath();
                            open(menu);
                        }
                    }
                }
            }

            @Override
            public void menuKeyReleased(MenuKeyEvent e) { /* do nothing */ }
    };

    /** Use for folder only */
    private static final MouseListener mouseListener = new MouseAdapter() {
        @Override
        public void mouseClicked(MouseEvent e) {
            open((JMenuItem) e.getSource());
        }
    };

    /** Use for folder only */
    private static final MenuListener menuListener = new MenuListener() {

        @Override
        public void menuSelected(MenuEvent e) {
            JMenu m = (JMenu) e.getSource();

            // create submenus
            if (m instanceof Runnable) {
                ((Runnable) m).run();
            } else {
                createSubmenu(m);
            }

            // Remove this listener
            m.removeMenuListener(this);
        }

        @Override
        public void menuDeselected(MenuEvent e) { /* do nothing */ }

        @Override
        public void menuCanceled(MenuEvent e) { /* do nothing */ }
    };

    /** Defeats instantiation. */
    private Listeners() {}

    public static ActionListener getActionListener() {
        return actionListener;
    }

    public static MenuKeyListener getMenuKeyListener() {
        return menuKeyListener;
    }

    public static MouseListener getMouseListener() {
        return mouseListener;
    }

    public static MenuListener getMenuListener() {
        return menuListener;
    }

    private static void open(JMenuItem item) {
        if (!item.isEnabled()) {
            return;
        }

        String path = item.getActionCommand();
        try {
            File file = new File(path);
            if (!FilePreview.open(file)) {
                Desktop.getDesktop().open(file);
            }
        } catch (IOException ex) {
            Log.log(ex);
            Core.getMainWindow().showMessageDialog(StaticUtils.format(
                    L10n.Entry.ERROR_FILE_HAS_NO_ASSOC.toString(), path));
        }
    }

    private static void createSubmenu(JMenu menu) {
        File folder = new File(menu.getActionCommand());
        File[] filteredListFiles = folder.listFiles(fileFilter);
        Arrays.sort(filteredListFiles, comparator);

        createSubmenu(menu, filteredListFiles);
    }

    private static void createSubmenu(JMenu menu, File[] files) {
        int maxViewableMenuItemCount = getMaxViewableMenuItemCount(menu);

        // Add "more" menu at the tail if the part of popup menu will be out of screen
        if (files.length > maxViewableMenuItemCount) {
            int limit = maxViewableMenuItemCount - 1;
            for (int i = 0; i < limit; i++) {
                menu.add(MenuFactory.createMenuItem(files[i]));
            }

            // create "More" menu
            final File[] moreFiles = Arrays.copyOfRange(files, limit, files.length);
            menu.add(createMoreMenu(moreFiles));
        } else {
            for (File file : files) {
                menu.add(MenuFactory.createMenuItem(file));
            }
        }
    }

    private static int getMaxViewableMenuItemCount(JMenu menu) {
        int h = menu.getHeight();
        int screenHeight = Toolkit.getDefaultToolkit().getScreenSize().height;
        Insets margin = menu.getPopupMenu().getMargin();
        Insets insets = menu.getPopupMenu().getInsets();
        int popupInsideHeight = screenHeight - margin.top - margin.bottom - insets.top - insets.bottom;
        return popupInsideHeight / h;
    }

    private static RunnableMenu createMoreMenu(final File[] files) {
        RunnableMenu m = new RunnableMenu() {
            @Override
            public void run() {
                createSubmenu(this, files);
            }
        };
        m.addMenuListener(Listeners.getMenuListener());
        Mnemonics.setLocalizedText(m, OStrings.getString("MW_MORE_SUBMENU"));
        m.setHorizontalAlignment(SwingConstants.CENTER);
        return m;
    }

    private static abstract class RunnableMenu extends JMenu implements Runnable {
        //
    }
}
