/**************************************************************************
 FolderMenu - easy access to project folders from menu.
 
 Copyright (C) 2013 Yu Tang
               Home page: http://sourceforge.jp/users/yu-tang/
               Support center: http://sourceforge.jp/users/yu-tang/pf/

 This file is part of plugin for OmegaT.
 http://www.omegat.org/

 License: GNU GPL version 3 or (at your option) any later version.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **************************************************************************/

package jp.sourceforge.users.yutang.omegat.plugin.foldermenu.filepreview;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import javax.swing.SwingUtilities;
import org.omegat.core.Core;
import org.omegat.core.CoreEvents;
import org.omegat.core.events.IApplicationEventListener;
import org.omegat.core.events.IProjectEventListener;
import org.omegat.core.events.IProjectEventListener.PROJECT_CHANGE_TYPE;
import static org.omegat.core.events.IProjectEventListener.PROJECT_CHANGE_TYPE.CLOSE;
import static org.omegat.core.events.IProjectEventListener.PROJECT_CHANGE_TYPE.COMPILE;

/**
 * must call init() before using this class.
 * 
 * @author Yu Tang
 */

public class FilePreview {

    // key = target file's canonical path
    private static final HashMap<String, IPreview> previews = new HashMap<String, IPreview>();
    
    private static IProjectEventListener projectEventListener = null;
    private static IApplicationEventListener applicationEventListener = null;
    
    static {
        // cleanup left temp files if they exists
        TempFileCleaner.cleanup();
        
        // call each Preview#init() 
        WordPreview.init();

        //@@TODO WSF reload 時にselectionがテキストボックスの場合、restore に失敗するバグ
    }

    public static boolean delete(final File originalFile) throws IOException {
        String key = originalFile.getCanonicalPath();
        IPreview deleted = previews.remove(key);
        if (previews.isEmpty())
            unhookProjectChangeEvent();
        return (deleted != null);
    }

    public static void init() {
        // force executing static initializer
    }

    public static boolean open(File file) throws IOException {
        // not available for directory
        if (! file.isFile())
            return false;

        // does file exists inside of the target folder?
        if (! isUnderTargetFolder(file))
            return false;

        // file type or environment is not supported 
        if (! available(file))
            return false;

        // Preview instance is already there?
        String key = file.getCanonicalPath();
        if (previews.containsKey(key)) {
            previews.get(key).activate();
            return true;
        }

        // open it
        IPreview p = new WordPreview(file);
        p.open();
        hookProjectChangeEvent();
        hookApplicationChangeEvent();
        previews.put(key, p);

        // add temp files to cleaner list
        TempFileCleaner.addToList(p.getTempFiles());
        
        return true;
    }

    public static int size(Class<?> classObj) {
        if (classObj == null) {
            return previews.size();
        } else {
            int i = 0;
            for (Object o: previews.values()) {
                if (classObj.isInstance(o))
                    i++;
            }
            return i;
        }
    }

    private static boolean available(File file) {
        return WordPreview.isAvailable(file);
    }

    /** hook project change event */
    private static void hookProjectChangeEvent() {
        if (projectEventListener != null)
            return;

        projectEventListener= new IProjectEventListener() {

            @Override
            public void onProjectChanged(PROJECT_CHANGE_TYPE eventType) {
                switch (eventType) {
                case CLOSE:
                    onProjectClose();
                    break;
                case COMPILE:
                    onProjectCompile();
                    break;
                }
            }

        };

        CoreEvents.registerProjectChangeListener(projectEventListener);
    }

    /** unhook project change event */
    private static void unhookProjectChangeEvent() {
        if (projectEventListener == null)
            return;

        CoreEvents.unregisterProjectChangeListener(projectEventListener);
        projectEventListener= null;
    }

    /** hook application change event */
    private static void hookApplicationChangeEvent() {
        if (applicationEventListener != null)
            return;

        applicationEventListener= new IApplicationEventListener() {

            @Override
            public void onApplicationStartup() {
                /* do nothing */
            }

            @Override
            public void onApplicationShutdown() {
                closeAllPreviews();
            }

        };

        CoreEvents.registerApplicationEventListener(applicationEventListener);
    }

    /** unhook project change event */
    private static void unhookApplicationChangeEvent() {
        if (applicationEventListener == null)
            return;

        CoreEvents.unregisterApplicationEventListener(applicationEventListener);
        applicationEventListener= null;
    }

    private static void onProjectClose() {
        closeAllPreviews();
        
        // イベントリスナーの登録解除をここで発行するとスレッドエラーになるので
        // 後で実行する。
        SwingUtilities.invokeLater(new Runnable() {
            @Override
            public void run() {
                unhookProjectChangeEvent();
                unhookApplicationChangeEvent();
            }
        });
    }

    private static void onProjectCompile() {
        reloadAllPreviews();
    }

    private static boolean isUnderTargetFolder(final File file) throws IOException {
        // does file exists inside of the target folder?
        String targetRoot = Core.getProject().getProjectProperties().getTargetRoot();
        return file.getCanonicalPath().startsWith(targetRoot);
    }

    private static void closeAllPreviews() {
        if (! previews.isEmpty()) {
            for (IPreview preview: previews.values()) {
                preview.close();
            }
        }
    }

    private static void reloadAllPreviews() {
        if (! previews.isEmpty()) {
            for (IPreview preview: previews.values()) {
                preview.reload();
            }
        }
    }

}
