/* RX On-chip data flash driver
 * Copyright 2014 Yoshinori Sato <ysato@users.sorceforge.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 *
 */

#include <common.h>
#include <environment.h>
#include <flash.h>
#include <asm/io.h>

#define FASTAT		0x007fc410
#define DFLRE0		0x007fc440
#define DFLRE1		0x007fc442
#define DFLWE0		0x007fc450
#define DFLWE1		0x007fc452
#define FCURAME		0x007fc454
#define FSTATR0		0x007fffb0
#define FSTATR1		0x007fffb1
#define FENTRYR		0x007fffb2
#define FPROTR		0x007fffb4
#define PCKAR		0x007fffe8
#define FWEPROR		0x0008c289

#define FASTAT_CMDLK	(1 << 4)

#define DFLRE0_KEY	0x2d00
#define DFLRE1_KEY	0xd200

#define DFLWE0_KEY	0x1e
#define DFLWE1_KEY	0xe1

#define FCURAME_KEY	0xc400
#define FCURAME_FCRME	0x01

#define FSTATR0_ILGLERR	(1 << 6)

#define FPROTR_KEY	0x5500
#define FPROTR_FPROTCN	0x0001

#define FENTRYR_KEY	0xaa00
#define FENTRYR_FENTRYD	0x80

#define FCU_CLK1	0xe9
#define FCU_CLK2	0x03
#define FCU_CLK345	0x0f0f
#define FCU_CLK6	0xd0

#define FCU_PROG	0xe8
#define FCU_PROG_DONE	0xd0

#define FCU_STATCLR	0x50

#define FCURAM_TOP	0x007f8000
#define FCUFIRM_TOP	0xfeffe000
#define FCUFIRM_SIZE	0x2000

flash_info_t flash_info[] = {
	{
		.size = 0x0800,
		.sector_count = 16,
		.flash_id = 0x52584454,
		.start[0] = 0x00100000,
		.protect[0] = 1,
		.start[1] = 0x00100800,
		.protect[1] = 1,
		.start[2] = 0x00101000,
		.protect[2] = 1,
		.start[3] = 0x00101800,
		.protect[3] = 1,
		.start[4] = 0x00102000,
		.protect[4] = 1,
		.start[5] = 0x00102800,
		.protect[5] = 1,
		.start[6] = 0x00103000,
		.protect[6] = 1,
		.start[7] = 0x00103800,
		.protect[7] = 1,
		.start[8] = 0x00104000,
		.protect[8] = 1,
		.start[9] = 0x00104800,
		.protect[9] = 1,
		.start[10] = 0x00105000,
		.protect[10] = 1,
		.start[11] = 0x00105800,
		.protect[11] = 1,
		.start[12] = 0x00106000,
		.protect[12] = 1,
		.start[13] = 0x00106800,
		.protect[13] = 1,
		.start[14] = 0x00107000,
		.protect[14] = 1,
		.start[15] = 0x00107800,
		.protect[15] = 1,
	},
};

static int wait_frdy(void)
{
	int i;
	for (i = 0; i < 10; i++) {
		if (__raw_readb(0x7fffb0) & 0x80)
			return 0;
		udelay(500);
	}
	return -1;
}
	
static int dataflash_pe_mode(unsigned long addr)
{
	/* Enter P/E mode */
	__raw_writew(FENTRYR_KEY | 0x00, FENTRYR);
	__raw_writew(FENTRYR_KEY | FENTRYR_FENTRYD, FENTRYR);
	__raw_writeb(0x01, FWEPROR);
	/* clear error status */
	if (__raw_readb(FSTATR0) & FSTATR0_ILGLERR) {
		if (__raw_readb(FASTAT) != FASTAT_CMDLK)
			__raw_writeb(FASTAT_CMDLK, FASTAT);
	}
	/* clock setting */
	__raw_writeb(FCU_STATCLR, addr);
	__raw_writew(CONFIG_SYS_CLK_FREQ / 1000000, PCKAR);
	__raw_writeb(FCU_CLK1, addr);
	__raw_writeb(FCU_CLK2, addr);
	__raw_writew(FCU_CLK345, addr);
	__raw_writew(FCU_CLK345, addr);
	__raw_writew(FCU_CLK345, addr);
	__raw_writeb(FCU_CLK6, addr);
	if(wait_frdy() < 0)
		return -1;
	return 0;
}

static void dataflash_normal_mode(void)
{
	__raw_writew(FENTRYR_KEY | 0x00, FENTRYR);
	__raw_writeb(0x02, FWEPROR);
}

#if defined(CONFIG_ENV_IS_IN_FLASH) || defined(CONFIG_ENV_ADDR_REDUND) || (CONFIG_SYS_MONITOR_BASE >= CONFIG_SYS_FLASH_BASE)
flash_info_t *flash_get_info(ulong base)
{
	flash_info_t *info;

	info = &flash_info[0];
	if (info->size && info->start[0] <= base &&
	    base <= info->start[0] + info->size - 1)
		return info;
	else
		return NULL;
}
#endif

unsigned long flash_sector_size(flash_info_t *info, flash_sect_t sect)
{
	return info->size;
}

int flash_real_protect (flash_info_t * info, long sector, int prot)
{
	unsigned short key;
	unsigned long dflwe;
	unsigned short reg;
	key = (sector < 8)?DFLWE0_KEY:DFLWE1_KEY;
	dflwe = DFLWE0 + 2 * (sector / 8);
	reg = __raw_readw(dflwe) & 0x00ff;
	if (prot)
		reg &= ~(1 << (sector & 7));
	else
		reg |= (1 << (sector & 7));
	__raw_writew((key << 8)| reg, dflwe);
	info->protect[sector] = prot;
	return 0;
}

int flash_erase(flash_info_t *info, int s_first, int s_last)
{
	int i;
	dataflash_pe_mode(info->start[s_first]);
	for (i = s_first; i <= s_last; i++) {
		__raw_writeb(0x20, info->start[i]);
		__raw_writeb(0xd0, info->start[i]);
		if (wait_frdy() < 0)
			return ERR_TIMOUT;
	}
	dataflash_normal_mode();
	return 0;
}

void flash_print_info(flash_info_t *info)
{
	int i;

	puts("RX On-chip data flash\n");
	for(i = 0; i < info->sector_count; i++) {
		printf("Sector %d: %ldKB (%08lx - %08lx)\n",
		       i, info->size / 1024,
		       info->start[i], info->start[i] + info->size - 1);
	}
}

int write_buff(flash_info_t *info, uchar *src, ulong addr, ulong cnt)
{
	unsigned long dst;
	int i;

	dataflash_pe_mode(addr);
	dst = addr & ~0x00000008;
	while(cnt > 0) {
		__raw_writeb(FCU_PROG, dst);
		__raw_writeb(128 / 2, dst);
		for(i = 0; i < 128; dst += 2, i += 2) {
			__raw_writew(*(unsigned short *)src, dst);
			cnt -= 2;
			src += 2;
		}
		__raw_writeb(FCU_PROG_DONE, dst);
		if (wait_frdy() < 0) {
			dataflash_normal_mode();
			return ERR_TIMOUT;
		}
	}
	dataflash_normal_mode();
	return 0;
}

unsigned long flash_init(void)
{
	/* setup FCU RAM */
	__raw_writew(FENTRYR_KEY | 0x00, FENTRYR);
	__raw_writew(FCURAME_KEY | FCURAME_FCRME, FCURAME);
	memcpy((void *)FCURAM_TOP, (void *)FCUFIRM_TOP, FCUFIRM_SIZE);
	/* dataflash read enable */
	__raw_writew(DFLRE0_KEY | 0xff, DFLRE0);
	__raw_writew(DFLRE1_KEY | 0xff, DFLRE1);
	return 32768;
}
