/*
 *  RX emulation
 *
 *  Copyright (c) 2018 Yoshinori Sato
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, see <http://www.gnu.org/licenses/>.
 */
#include "qemu/osdep.h"
#include "cpu.h"
#include "exec/helper-proto.h"
#include "exec/exec-all.h"
#include "exec/cpu_ldst.h"
#include "fpu/softfloat.h"

static inline void QEMU_NORETURN raise_exception(CPURXState *env, int index,
                                                 uintptr_t retaddr)
{
    CPUState *cs = CPU(rx_env_get_cpu(env));

    cs->exception_index = index;
    cpu_loop_exit_restore(cs, retaddr);
}

void QEMU_NORETURN helper_raise_privilege_violation(CPURXState *env)
{
    raise_exception(env, 20, GETPC());
}

void QEMU_NORETURN helper_raise_access_fault(CPURXState *env)
{
    raise_exception(env, 21, GETPC());
}

void QEMU_NORETURN helper_raise_illegal_instruction(CPURXState *env)
{
    raise_exception(env, 23, GETPC());
}

void QEMU_NORETURN helper_wait(CPURXState *env)
{
    CPUState *cs = CPU(rx_env_get_cpu(env));

    cs->halted = 1;
    env->in_sleep = 1;
    raise_exception(env, EXCP_HLT, 0);
}

void QEMU_NORETURN helper_debug(CPURXState *env)
{
    raise_exception(env, EXCP_DEBUG, 0);
}

void QEMU_NORETURN helper_rxint(CPURXState *env, uint32_t vec)
{
    CPUState *cs = CPU(rx_env_get_cpu(env));

    cs->interrupt_request |= CPU_INTERRUPT_SOFT;
    env->irq = vec;
    env->int_insn_len = 3;
    raise_exception(env, cs->singlestep_enabled?EXCP_DEBUG:EXCP_INTERRUPT,
                    env->pc + 3);
}

void QEMU_NORETURN helper_rxbrk(CPURXState *env)
{
    CPUState *cs = CPU(rx_env_get_cpu(env));

    cs->interrupt_request |= CPU_INTERRUPT_SOFT;
    env->irq = 0;
    env->int_insn_len = 1;
    raise_exception(env, cs->singlestep_enabled?EXCP_DEBUG:EXCP_INTERRUPT,
                    env->pc + 1);
}

static void update_fpsw(CPURXState *env, uintptr_t retaddr)
{
    int xcpt, cause, enable;

    xcpt = get_float_exception_flags(&env->fp_status);

    /* Clear the cause entries */
    env->fpsw &= ~FPSW_CAUSE_MASK;

    if (unlikely(xcpt)) {
        if (xcpt & float_flag_invalid) {
            env->fpsw |= FPSW_CAUSE_V;
        }
        if (xcpt & float_flag_divbyzero) {
            env->fpsw |= FPSW_CAUSE_Z;
        }
        if (xcpt & float_flag_overflow) {
            env->fpsw |= FPSW_CAUSE_O;
        }
        if (xcpt & float_flag_underflow) {
            env->fpsw |= FPSW_CAUSE_U;
        }
        if (xcpt & float_flag_inexact) {
            env->fpsw |= FPSW_CAUSE_X;
        }

        /* Accumulate in flag entries */
        env->fpsw |= (env->fpsw & FPSW_CAUSE_MASK)
                      << (FPSW_FLAG_SHIFT - FPSW_CAUSE_SHIFT);
        env->fpsw |= ((env->fpsw >> FPSW_FLAG_V) |
                      (env->fpsw >> FPSW_FLAG_O) |
                      (env->fpsw >> FPSW_FLAG_Z) |
                      (env->fpsw >> FPSW_FLAG_U) |
                      (env->fpsw >> FPSW_FLAG_X)) << FPSW_FLAG_S;
        
        /* Generate an exception if enabled */
        cause = (env->fpsw & FPSW_CAUSE_MASK) >> FPSW_CAUSE_SHIFT;
        enable = (env->fpsw & FPSW_ENABLE_MASK) >> FPSW_ENABLE_SHIFT;
        if (cause & enable) {
            raise_exception(env, 21, retaddr);
        }
    }
}

void helper_to_fpsw(CPURXState *env, uint32_t val)
{
    static const int roundmode[] = {
        float_round_nearest_even,
        float_round_to_zero,
        float_round_up,
        float_round_down,
    };
    env->fpsw = val & FPSW_MASK;
    set_float_rounding_mode(roundmode[val & FPSW_RM_MASK], &env->fp_status);
    set_flush_to_zero((val & FPSW_DN) != 0, &env->fp_status);
}

float32 helper_floatop(CPURXState *env, uint32_t op, float32 t0, float32 t1)
{
    static float32 (*fop[])(float32, float32, float_status *status) = {
        float32_sub,
        NULL,
        float32_add,
        float32_mul,
        float32_div,
    };
    int st, xcpt;
    if (op != 1) {
        t0 = fop[op](t0, t1, &env->fp_status);
        update_fpsw(env, GETPC());
    } else {
        st = float32_compare(t0, t1, &env->fp_status);
        xcpt = get_float_exception_flags(&env->fp_status);
        env->fpsw &= ~FPSW_CAUSE_MASK;

        if (xcpt & float_flag_invalid) {
            env->fpsw |= FPSW_CAUSE_V;
            if (env->fpsw & FPSW_ENABLE_V)
                raise_exception(env, 21, GETPC());
        }
        switch (st) {
        case float_relation_unordered:
            env->psw_o = 1;
            break;
        case float_relation_equal:
            env->psw_z = 1;
            break;
        case float_relation_less:
            env->psw_s = 1;
            break;
        }
    }
    return t0;
}

uint32_t helper_ftoi(CPURXState *env, float32 t0)
{
    uint32_t ret;
    ret = float32_to_int32_round_to_zero(t0, &env->fp_status);
    update_fpsw(env, GETPC());
    return ret;
}

uint32_t helper_round(CPURXState *env, float32 t0)
{
    uint32_t ret;
    ret = float32_to_int32(t0, &env->fp_status);
    update_fpsw(env, GETPC());
    return ret;
}

float32 helper_itof(CPURXState *env, uint32_t t0)
{
    float32 ret;
    ret = int32_to_float32(t0, &env->fp_status);
    update_fpsw(env, GETPC());
    return ret;
}

void helper_racw(CPURXState *env, uint32_t shift)
{
    int64_t acc;
    acc = env->acc_m;
    acc = (acc << 32) | env->acc_l;
    acc <<= shift;
    acc += 0x0000000080000000LL;
    if (acc > 0x00007FFF00000000LL)
        acc = 0x00007FFF00000000LL;
    else if (acc < 0xFFFF800000000000LL)
        acc = 0xFFFF800000000000LL;
    else
        acc &= 0xffffffff00000000;
    env->acc_m = (acc >> 32);
    env->acc_l = (acc & 0xffffffff);
}

void tlb_fill(CPUState *cs, target_ulong addr, int size,
              MMUAccessType access_type, int mmu_idx, uintptr_t retaddr)
{
    uint32_t address, physical, prot;
    
    /* Linear mapping */
    address = physical = addr & TARGET_PAGE_MASK;
    prot = PAGE_READ | PAGE_WRITE | PAGE_EXEC;
    tlb_set_page(cs, address, physical, prot, mmu_idx, TARGET_PAGE_SIZE);
}

