﻿using CleanAuLait.Core.Log;
using CleanAuLait.ObservableCollectionsMod;
using NLog;
using Prism.Navigation;
using Reactive.Bindings;
using Reactive.Bindings.Extensions;
using System.ComponentModel;
using System.Reactive.Disposables;
using TestNarou3.Adaptor.Boundary.Controller;
using TestNarou3.Adaptor.Boundary.Gateway.ViewModel;
using TestNarou3.Adaptor.Boundary.Gateway.ViewModel.Child;
using TestNarou3.Adaptor.Translator;
using TestNarou3.Domain.Model.Entity;
using TestNarou3.Domain.Model.Entity.Child;

namespace TestNarou3.Adaptor.Gateway.ViewModel
{
    internal class BookmarkCategoryViewModel
            : IBookmarkCategoryViewModel, INotifyPropertyChanged, IDestructible, IDisposable
    {
        private static readonly ILogger logger = LogManager.GetCurrentClassLogger();

#pragma warning disable CS0067
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore CS0067

        public ReactiveProperty<IBookmarkCategoryRowViewModel> SelectedItem { get; }

        public INotifyCollectionChangedListSynchronizedSingleView<BookmarkCategoryRow, IBookmarkCategoryRowViewModel> Rows { get; private set; } = null;

        public BookmarkCategory Source { get; private set; }

        public IAppWindowController WindowController { get; }

        private readonly IBookmarkCategoryRowViewModelTranslator vmTranslator;

        private readonly CompositeDisposable disposables = new();
        private bool disposedValue;

        public BookmarkCategoryViewModel(
                IAppWindowController wc,
                IBookmarkCategoryRowViewModelTranslator vmTranslator)
        {
            this.WindowController = wc;
            this.vmTranslator = vmTranslator;

            this.SelectedItem = new ReactiveProperty<IBookmarkCategoryRowViewModel>().AddTo(disposables);
            this.SelectedItem.Subscribe(OnSelectedItemChanged).AddTo(disposables);
        }

        #region event handler

        private void OnSelectedItemChanged(IBookmarkCategoryRowViewModel row)
        {
            if (row == null)
            {
                return;
            }
            this.WindowController.NavigateBookmarkListView(row.Index);
        }

        #endregion

        #region vm operation

        public void SynchronizeWith(BookmarkCategory source)
        {
            if (this.Source == null)
            {
                this.Rows = source.Rows
                        .ToSynchronizedCoupleView(r => this.vmTranslator.Translate(r, this))
                        .ToSynchronizedSingleView()
                        .WithINotifyCollectionChangedList()
                        .AddTo(disposables);

                logger.Trace("BookmarkCategoryViewModel.SynchronizeWith: source [{0}], collection view [{1}]",
                        source.ToHashString(), this.Rows.GetType().FullName);

                this.Source = source;

                // after transitioning to the bookmark page, the first category must be selected.
                if (source.Rows.Count() > 0)
                {
                    this.SelectedItem.Value = this.Rows.ElementAt(0);
                }
            }
            else
            {
                logger.Trace("BookmarkCategoryViewModel.SynchronizeWith: viewmodel already synchronized to model [{0}]",
                        this.Source.ToHashString());
            }
        }

        #endregion

        #region IDestructible, IDisposable

        public void Destroy()
        {
            this.Dispose();
        }

        protected virtual void Dispose(bool disposing)
        {
            if (!disposedValue)
            {
                if (disposing)
                {
                    this.disposables.Dispose();

                    logger.Trace("BookmarkCategoryViewModel disposed.");
                }

                disposedValue = true;
            }
        }

        public void Dispose()
        {
            Dispose(disposing: true);
            GC.SuppressFinalize(this);
        }

        #endregion

    }
}
