package net.waltzstudio.montecarlo.service;

import static net.waltzstudio.base.framework.util.SyntaxSugar.S;
import java.util.List;
import javax.annotation.Resource;
import org.seasar.framework.beans.util.Beans;
import net.waltzstudio.montecarlo.dao.account.AccountDao;
import net.waltzstudio.montecarlo.dao.project.ProjectDao;
import net.waltzstudio.montecarlo.dao.ticket.TicketDao;
import net.waltzstudio.montecarlo.dto.RequestInfoDto;
import net.waltzstudio.montecarlo.dto.SessionInfoDto;
import net.waltzstudio.montecarlo.entity.account.Account;
import net.waltzstudio.montecarlo.entity.project.Project;
import net.waltzstudio.montecarlo.entity.ticket.Ticket;
import net.waltzstudio.montecarlo.form.TicketForm;
import net.waltzstudio.montecarlo.types.TicketStatus;

/**
 * チケットService
 */
public class TicketService {
	/** リクエスト情報 */
	@Resource
	protected RequestInfoDto requestInfoDto;
	/** セッション情報 */
	@Resource
	protected SessionInfoDto sessionInfoDto;

	/** チケットDao */
	@Resource
	protected TicketDao ticketDao;
	/** プロジェクトDao */
	@Resource
	protected ProjectDao projectDao;
	/** アカウントDao */
	@Resource
	protected AccountDao accountDao;

	/**
	 * チケット一覧処理
	 *
	 * @return チケットEntityリスト
	 */
	public List<Ticket> search() {
		return ticketDao.findAllOrderById();
	}

	/**
	 * チケット取得
	 *
	 * @param id チケットID
	 * @return チケットEntity
	 */
	public Ticket getTicket(Long id) {
		return ticketDao.findById(id);
	}

	public List<Account> getAccounts() {
		return accountDao.findByProjectId(requestInfoDto.projectId);
	}

	/**
	 * チケット登録処理
	 *
	 * @param ticketForm チケットForm
	 * @return 登録したチケットEntity
	 */
	public Ticket register(TicketForm ticketForm) {
		Project project = projectDao.findByCodeAndAccountId(ticketForm.projectCode, sessionInfoDto.id);
		if (project == null) {
			throw new RuntimeException(S("プロジェクトが見つかりませんでした。 : ", ticketForm.projectCode));
		}
		Ticket ticket = Beans.createAndCopy(Ticket.class, ticketForm).excludes("status").dateConverter("yyyy/M/d", "fromDate", "toDate").execute();
		ticket.projectId = project.id;
		ticket.status = TicketStatus.create(ticketForm.status);
		ticket.registerId = sessionInfoDto.id;
		ticket.createId = sessionInfoDto.id;
		ticket.updateId = sessionInfoDto.id;
		ticketDao.insert(ticket);
		return ticket;
	}

}
