/**************************************************
opengate server

module to control both of ipfw and ip6fw 

Copyright (C) 1999 Opengate Project Team
Written by Katsuhiko Eguchi, 2005 
Modified by Yoshiaki Watanabe

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Email: watanaby@is.saga-u.ac.jp
**************************************************/
#include	"opengatesrv.h"

extern char ruleNumber4[WORDMAXLN];  /* ipfw rule number in string form  */
extern char ruleNumber6[WORDMAXLN];  /* ip6fw rule number in string form */
struct clientAddr *pClientAddr = NULL;
int ipStatus;

/**************************************/
/* Open client gate for IPv4 and IPv6 */
/**************************************/
int openClientGate(char *clientAddr4, char *macAddr4, char *clientAddr6, char *macAddr6, char *userid, char *userProperty, int ipStatus)
{
  int ret;
  struct clientAddr *pLastClientAddr=NULL;

  switch(ipStatus){

  /**** if client have IPv6 & IPv4 addresses, do below *******/
  case IPV46DUAL:
    
    /*** at first, open IPv4 gate *****/
    ret=OpenClientGate4(clientAddr4, userid, macAddr4, userProperty);
    if(ret==0){ /* normally open */

      /* open success */
      err_msg("OPEN: user %s from %s at %s", userid, clientAddr4, macAddr4);

      /* create new address-list. head and tail pointer point the same item. */
      pLastClientAddr=pClientAddr=CreateAddrListItem(clientAddr4,ruleNumber4,IPV4);
    }

    /* dupliacted rules */
    else  if(ret<-2){
      return FALSE;
    }

    /* some failure */
    else {
      PutClientMsg("Error: Please contact to the administrator");
      return FALSE;
    }


    /*** next, open IPv6 gate *****/
    ret=OpenClientGate6(clientAddr6, userid, macAddr6, userProperty);
    if(ret==0){ /* normally open */

      /* open success */
      err_msg("OPEN: user %s from %s at %s", userid, clientAddr6, macAddr6);

      /* add the address info to list */
      pLastClientAddr->next = CreateAddrListItem(clientAddr6,ruleNumber6,IPV6);
    }

    /* save dupliacted rule */
    else if(ret<-2){
      return FALSE;
    }

    /* some failure */
    else{
      /* remove previously registered ipv4 rule from list */ 
      if(pClientAddr!=NULL && pClientAddr->ipType==IPV4){
	CloseClientGate4(pClientAddr,userid,macAddr4);
	free(pClientAddr);
	pClientAddr=NULL;
      }
      PutClientMsg("Error: Please contact to the administrator");
      return FALSE;
    }
    break;  /* case IPV46DUAL end */


    /***** if client have IPv4 only, do below *******/
  case IPV4ONLY:

    /* open IPv4 gate */
    ret=OpenClientGate4(clientAddr4, userid, macAddr4, userProperty);
    if(ret==0){ /* normally open */

      /* open success */
      err_msg("OPEN: user %s from %s at %s", userid, clientAddr4, macAddr4);

      /* create new address list. head and tail pointer point the same item. */
      pLastClientAddr=pClientAddr
	=CreateAddrListItem(clientAddr4,ruleNumber4,IPV4);
    }

    /* dupliacted rule */
    else if(ret<-2){
      return FALSE;
    }

    /* some failure */
    else{
      PutClientMsg("Error: Please contact to the administrator");
      return FALSE;
    }
    break; /* case IPV4ONLY end */
    

    /***** if client have IPv6 only, do below ******/
  case IPV6ONLY:
    /* open IPv6 gate */
    ret=OpenClientGate6(clientAddr6, userid, macAddr6, userProperty);
    if(ret==0){ /* normally open */

      /* open success */
      err_msg("OPEN: user %s from %s at %s", userid, clientAddr6, macAddr6);

      /* create new address list. head and tail pointer point the same item. */
       pLastClientAddr=pClientAddr
	 =CreateAddrListItem(clientAddr6,ruleNumber6,IPV6);
    }

    /* dupliacted rule */
    else if(ret<-2){
      return FALSE;
     }

    /* open fail */
    else{
      PutClientMsg("Error: Please contact to the administrator");
      return FALSE;
    }
    break; /* case IPv6ONLY end */

  default:
    err_msg("ERR at %s#%d: abnormal IP versions %d",__FILE__,__LINE__,ipStatus);
    return FALSE;

  } /* switch end */
  
  return TRUE;
}

/****************************************************/
/* remove duplicated firewall rule for same address */
/****************************************************/
void removeDuplicateRule(int duplicateRule4, int duplicateRule6)
{
  char ruleNumStr[WORDMAXLN];

  /* if ipfw rule exist, delete it */
  if(duplicateRule4>0){
    snprintf(ruleNumStr,WORDMAXLN,"%d",duplicateRule4);
    DelIpfwRule(ruleNumStr);
  }

  /* if ip6fw rule exist, delete it */
  if(duplicateRule6>0){
    snprintf(ruleNumStr,WORDMAXLN,"%d",duplicateRule6);
    DelIp6fwRule(ruleNumStr);
  }

}

/*********************************************************************************/
/* create address list                                                           */
/* address-list is a linked-list of [struct clientAddr] definded in openatesrv.h */
/* it holds the addresses opened by this process                                 */
/*********************************************************************************/
struct clientAddr *createAddrListItem(char* ipAddr, char *ruleNumber, int ipType)
{
  struct clientAddr *pNew;

  /* allocate memory area for a linked-list item */ 
  if((pNew = (struct clientAddr *) malloc(sizeof(struct clientAddr))) == NULL){
    err_msg("ERR at %s#%d: malloc error",__FILE__,__LINE__);
    PutClientMsg("Error: Please contact to the administrator");
    exit(1);
  }

  /* set data in the linked-list item */
  strlcpy(pNew->ipAddr,ipAddr,ADDRMAXLN);
  strlcpy(pNew->ruleNumber,ruleNumber,WORDMAXLN);
  pNew->ipType = ipType;
  pNew->timeIn = time(NULL);
  pNew->activeStatus=TRUE;
  pNew->next = NULL;
  return pNew;
}

/****************************************/
/* Check version of enable IP addresses */
/* in clientAddr4,clientAddr6           */
/* out: IPV46DUAL,IPV4ONLY,IPV6ONLY     */
/****************************************/
int checkIpVersions(char *clientAddr4, char *clientAddr6){

  /* if the address includes '.', it is IPv4 */
  /* if the address includes ':', it is IPv6 */
  if(strstr(clientAddr4,".")!=NULL){
    if(strstr(clientAddr6,":")!=NULL){
      ipStatus = IPV46DUAL;
    }else{
      ipStatus = IPV4ONLY;
    }
  }else{
    if(strstr(clientAddr6,":")!=NULL){
      ipStatus = IPV6ONLY;
    }else{
      err_msg("ERR at %s#%d: Can not recognize IP versions for v4:%s v6:%s"
	      ,__FILE__,__LINE__,clientAddr4,clientAddr6);
      ipStatus=0;
    }
  }
  return ipStatus;
}

/*************************************************************/
/* get packet count from ipfw and ip6fw                      */
/* in: pClientAddr = pointer to the linked-list of addresses */
/* out: sum of packet count for the addresses                */
/*************************************************************/
int getPacketCount(struct clientAddr *pClientAddr)
{
  int packets=0;
  struct clientAddr *pNow;
  pNow=pClientAddr;

  while(pNow!=NULL){
    if(pNow->ipType==IPV4){
      packets+=GetPacketCount4(pNow->ruleNumber);
    }else if(pNow->ipType==IPV6){
      packets+=GetPacketCount6(pNow->ruleNumber);
    }
    pNow=pNow->next;
  }

  return packets;
}

/************************************/
/************************************/

int OpenClientGate(char *clientAddr4, char *macAddr4, char *clientAddr6, char *macAddr6, char *userid, char *userProperty, int ipStatus)
{
  int ret;
  if(debug>1) err_msg("DEBUG:=>openClientGate(%s,%s,%s,%s,%s,%s,%d)",clientAddr4,macAddr4,clientAddr6,macAddr6,userid,userProperty, ipStatus);
  ret = openClientGate(clientAddr4,macAddr4,clientAddr6,macAddr6,userid,userProperty, ipStatus);
  if(debug>1) err_msg("DEBUG:(%d)<=openClientGate()",ret);
  return ret;
}

struct clientAddr *CreateAddrListItem(char* ipAddr, char *ruleNumber, int ipType)
{
  struct clientAddr *ret;
  if(debug>1) err_msg("DEBUG:=>createAddrListItem(%s,%s,%d)",ipAddr,ruleNumber,ipType);
  ret = createAddrListItem(ipAddr, ruleNumber, ipType);
  if(debug>1) err_msg("DEBUG: [%p] <=createAddrListItem( )",ret);
  return ret;
}

int CheckIpVersions(char *clientAddr4, char *clientAddr6)
{
  int ret;
  if(debug>1) err_msg("DEBUG:=>checkIpVersions(%s,%s)", clientAddr4, clientAddr6);
  ret=checkIpVersions(clientAddr4, clientAddr6);
  if(debug>1) err_msg("DEBUG:(%d)<=checkIpVersions( )", ret);

  return ret;
}

void RemoveDuplicateRule(int duplicateRule4, int duplicateRule6){

  if(debug>1) err_msg("DEBUG:=>removeDuplicateRule(%d,%d)",duplicateRule4,duplicateRule6);
  removeDuplicateRule(duplicateRule4,duplicateRule6);
  if(debug>1) err_msg("DEBUG:<=removeDuplicateRule( )");
}

int GetPacketCount(struct clientAddr *pClientAddr)
{
  int ret;

  if(debug>1) err_msg("DEBUG:=>getPacketCount(%p)",pClientAddr);
  ret=getPacketCount(pClientAddr);
  if(debug>1) err_msg("DEBUG:(%d)<=getPacketCount( )",ret);

  return ret;
}
