/*************************************************
  opengate server
   module for Controlling user DB (SQLite3)

Copyright (C) 1999 Opengate Project Team
Written by Yoshiaki Watanabe

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Email: watanaby@is.saga-u.ac.jp
**************************************************/
#include "opengatesrv.h"
#include <sqlite3.h>

static int sqliteBusyTimeout=100;  /* value used in sqite3_busy_timeout() */

/*******************************************************************
 read sqlite busy timeout value from conf and set to static variable
*******************************************************************/
int setupSqliteBusyTimeoutValue(void){

  char *str;
  
  /* if set in conf, use the value. if not, use the above default. */
  str=GetConfValue("SqliteBusyTimeout");
  if(str!=NULL) sqliteBusyTimeout=atoi(str);

  return sqliteBusyTimeout;
}

/**************************************************************/
/* write session info to session control database at start    */ 
/**************************************************************/
int putSessionBeginToDb(char* cookie, char* userid, 
			char* clientAddr4, char* clientAddr6, 
			char* macAddr4, 
			char* ruleNumber4, char* ruleNumber6,
			int duration, int durationEntered,
			int cookieAuth, char *language){

  int rc;
  sqlite3 *db;
  char *pErrMsg;
  /* SQL CREATE COMMAND */
  char *createTblCmd="CREATE TABLE IF NOT EXISTS \
 session (cookie TEXT PRIMARY KEY,\
 userid TEXT, pid INTEGER, openTime TEXT, closeTime TEXT,\
 clientAddr4 TEXT, clientAddr6 TEXT, macAddr TEXT,\
 ruleNumber4 INTEGER, ruleNumber6 INTEGER,duration INTEGER,\
 durationEntered INTEGER, cookieAuth INTEGER, language TEXT, watchMode TEXT)";

  char *createIdx1Cmd="CREATE INDEX IF NOT EXISTS \
 clientAddr4Index ON session (clientAddr4)";

  char *createIdx2Cmd="CREATE INDEX IF NOT EXISTS \
 closeTimeIndex ON session (closeTime)";


  /* SQL INSERT COMMAND, where %x is replaced in snprintf */
  char *insertFormat="INSERT INTO session\
 (cookie, userid, pid, openTime, closeTime,\
 clientAddr4, clientAddr6, macAddr, ruleNumber4, ruleNumber6,\
 duration, durationEntered, cookieAuth, language, watchMode)\
 values ('%s','%s',%d,\
 datetime(%d,'unixepoch','localtime'),\
 '-','%s','%s','%s', '%s', '%s', %d, %d, %d, '%s', '-')";
  char *insertCmd;
  int resultFlag=TRUE;
  
  /* Open sqlite */
  if(sqlite3_open(GetConfValue("SqliteDb"),&db)!=SQLITE_OK){
    err_msg("ERR at %s#%d: sqlite3_open",__FILE__,__LINE__);
    sqlite3_close(db);
    return FALSE;
  }
  sqlite3_busy_timeout(db, sqliteBusyTimeout);

  /* create table if not exists */
  if(sqlite3_exec(db, createTblCmd, NULL, NULL, &pErrMsg)!=SQLITE_OK){
    resultFlag=FALSE;
    err_msg("ERR at %s#%d: sqlite3_exec: %s",__FILE__,__LINE__,pErrMsg);
  }

  /* create index for clientAddr4 if not exists */
  if(sqlite3_exec(db, createIdx1Cmd, NULL, NULL, &pErrMsg)!=SQLITE_OK){
    resultFlag=FALSE;
    err_msg("ERR at %s#%d: sqlite3_exec: %s",__FILE__,__LINE__,pErrMsg);
  }

  /* create index for closeTime if not exists */
  if(sqlite3_exec(db, createIdx2Cmd, NULL, NULL, &pErrMsg)!=SQLITE_OK){
    resultFlag=FALSE;
    err_msg("ERR at %s#%d: sqlite3_exec: %s",__FILE__,__LINE__,pErrMsg);
  }

  /* Prepare insert command */
  insertCmd=sqlite3_mprintf(insertFormat,cookie,userid, 
			    getpid(),time(NULL),clientAddr4,clientAddr6, 
			    macAddr4,ruleNumber4,ruleNumber6,
			    duration,durationEntered,cookieAuth,language);

  /* Execute insert to sqlite */
  if((rc=sqlite3_exec(db, insertCmd, NULL, NULL, &pErrMsg))!=SQLITE_OK){

    /* If the error is 'table session has no column named xxx' */
    /* It might be adding column in varsion up */
    if(rc==SQLITE_ERROR && strstr(pErrMsg, "has no column named")!=NULL){
      err_msg("ERR at %s#%d: DB format is changed in version up, remove '%s' and retry",__FILE__,__LINE__,GetConfValue("SqliteDb"));
      resultFlag=FALSE;
    }

    /* Other errors */
    else{
      resultFlag=FALSE;
      err_msg("ERR at %s#%d: sqlite3_exec: %s",__FILE__,__LINE__,pErrMsg);
    }
  }

  /*Memory free for sqlite3 string */
  sqlite3_free(insertCmd);

  /* Sqlite close */
  sqlite3_close(db);

  return resultFlag;
}

/**************************************************************/
/* write session info to session control database at stop     */ 
/**************************************************************/
int putSessionEndToDb(char* cookie, char* watchMode){

  sqlite3 *db;
  char *pErrMsg;

  /* SQL UPDATE COMMAND, where %x is replaced in mprintf */
  char *updateFormat="UPDATE session SET closeTime=\
 datetime(%d,'unixepoch','localtime'), watchMode='%s' WHERE cookie='%s'";
  char *updateCmd;
  int resultFlag=TRUE;

  /* open sqlite */
  if(sqlite3_open(GetConfValue("SqliteDb"),&db)!=SQLITE_OK){
    err_msg("ERR at %s#%d: sqlite3_open",__FILE__,__LINE__);
    sqlite3_close(db);
    return FALSE;
  }
  sqlite3_busy_timeout(db, sqliteBusyTimeout);
  
  /* prepare command */
  updateCmd=sqlite3_mprintf(updateFormat, time(NULL), watchMode, cookie);

  /* execute replace to sqlite */
  if(sqlite3_exec(db, updateCmd, NULL, NULL, &pErrMsg)!=SQLITE_OK){
    resultFlag=FALSE;
    err_msg("ERR at %s#%d: sqlite3_exec: %s",__FILE__,__LINE__,pErrMsg);
  }

  /*memory free for sqlite3 string */
  sqlite3_free(updateCmd);
  
  /* sqlite close */
  sqlite3_close(db);

  return resultFlag;
}

/**************************************************************/
/* write close time to db to fix no-close error               */ 
/**************************************************************/
int fixProcessEndInDb(int pid, char* watchMode){

  sqlite3 *db;
  char *pErrMsg;

  /* SQL UPDATE COMMAND, where %x is replaced in mprintf */
  char *updateFormat="UPDATE session SET closeTime=\
 datetime(%d,'unixepoch','localtime'), watchMode='%s' WHERE pid=%d";
  char *updateCmd;
  int resultFlag=TRUE;

  /* open sqlite */
  if(sqlite3_open(GetConfValue("SqliteDb"),&db)!=SQLITE_OK){
    err_msg("ERR at %s#%d: sqlite3_open",__FILE__,__LINE__);
    sqlite3_close(db);
    return FALSE;
  }
  sqlite3_busy_timeout(db, sqliteBusyTimeout);
  
  /* prepare command */
  updateCmd=sqlite3_mprintf(updateFormat, time(NULL), watchMode, pid);

  /* execute replace to sqlite */
  if(sqlite3_exec(db, updateCmd, NULL, NULL, &pErrMsg)!=SQLITE_OK){
    resultFlag=FALSE;
    err_msg("ERR at %s#%d: sqlite3_exec: %s",__FILE__,__LINE__,pErrMsg);
  }

  /*memory free for sqlite3 string */
  sqlite3_free(updateCmd);
  
  /* sqlite close */
  sqlite3_close(db);

  return resultFlag;
}

/*************************************************/
/* read userid and others from session database  */
/* search key is coocke                          */ 
/*************************************************/
int getSessionInfoFromDb(char* cookie, char* userid, 
			  char* clientAddr4, char *macAddr, 
			 int *duration, int *durationEntered, char *language,
			 char* closeTime){

  sqlite3 *db;
  sqlite3_stmt *stmt;
 
  /* SQL UPDATE COMMAND, where %x is replaced in snprintf */
  char *selectFormat="SELECT userid, clientAddr4, macAddr,\
 duration, durationEntered,language,closeTime FROM session WHERE cookie='%s'";
  char *selectCmd;
  int resultFlag=TRUE;

  /* open sqlite */
  if(sqlite3_open(GetConfValue("SqliteDb"),&db)!=SQLITE_OK){
    err_msg("ERR at %s#%d: sqlite3_open",__FILE__,__LINE__);
    sqlite3_close(db);
    return FALSE;
  }
  sqlite3_busy_timeout(db, sqliteBusyTimeout);

  /* prepare command string */
  selectCmd=sqlite3_mprintf(selectFormat, cookie);
  
  /* compile to internal statement */
  if(sqlite3_prepare(db, selectCmd, BUFFMAXLN, &stmt, NULL)!=SQLITE_OK){
    resultFlag=FALSE;
    err_msg("ERR at %s#%d: sqlite3_prepare:%s",__FILE__,__LINE__,
	    sqlite3_errmsg(db));

    /* finalize */
    sqlite3_free(selectCmd);
    sqlite3_finalize(stmt);
    sqlite3_close(db);
    return FALSE;
  }

  /* get first match item */
  if(sqlite3_step(stmt)==SQLITE_ROW){
    strlcpy(userid, (char*)sqlite3_column_text(stmt, 0), USERMAXLN);
    strlcpy(clientAddr4, (char*)sqlite3_column_text(stmt, 1), USERMAXLN);
    strlcpy(macAddr, (char*)sqlite3_column_text(stmt, 2), USERMAXLN);
    *duration=(int)sqlite3_column_int(stmt, 3);
    *durationEntered=(int)sqlite3_column_int(stmt, 4);
    strlcpy(language, (char*)sqlite3_column_text(stmt, 5), WORDMAXLN);
    strlcpy(closeTime, (char*)sqlite3_column_text(stmt, 6), WORDMAXLN);
    resultFlag=TRUE;
  }else{
    resultFlag=FALSE;
  }
  
  /* finalize */
  sqlite3_free(selectCmd);
  sqlite3_finalize(stmt);
  sqlite3_close(db);
  
  return resultFlag;
}

/********************************************************/
/* check nat insertion by reading info in opengatemd.db */ 
/********************************************************/
int checkNatInsertion(char* macAddr4, char* macAddr6, char* userid){

  sqlite3 *db;
  sqlite3_stmt *stmt;
  struct stat st;
  char* macAddress=NULL;
 
  /* SQL UPDATE COMMAND, where %x is replaced in snprintf */
  char *selectFormat="SELECT isNat FROM macinfo WHERE macAddress='%s'";
  char *selectCmd;
  int resultFlag;

  /* find not-null mac address */
  if(!isNull(macAddr4)) macAddress=macAddr4;
  else if(!isNull(macAddr6)) macAddress=macAddr6;
  else return FALSE;

  /* if db is not exist, ignore */
  if(stat(GetConfValue("SqliteDbMd"),&st)!=0) return FALSE;

  /* open sqlite */
  if(sqlite3_open(GetConfValue("SqliteDbMd"),&db)!=SQLITE_OK){
    err_msg("ERR at %s#%d: sqlite3_open",__FILE__,__LINE__);
    sqlite3_close(db);
    return FALSE;
  }
  sqlite3_busy_timeout(db, sqliteBusyTimeout);

  /* prepare command string */
  selectCmd=sqlite3_mprintf(selectFormat, macAddress);
  
  /* compile to internal statement */
  if(sqlite3_prepare(db, selectCmd, BUFFMAXLN, &stmt, NULL)!=SQLITE_OK){

    /* finalize */
    sqlite3_free(selectCmd);
    sqlite3_finalize(stmt);
    sqlite3_close(db);
    return FALSE;
  }

  /* get first match item */
  if(sqlite3_step(stmt)==SQLITE_ROW){
    resultFlag=(int)sqlite3_column_int(stmt, 0);
  }else{
    resultFlag=FALSE;
  }
  
  /* finalize */
  sqlite3_free(selectCmd);
  sqlite3_finalize(stmt);
  sqlite3_close(db);
  
  /* if found, write to log (0=direct, 1=via nat, 2=via valid router) */
  if(resultFlag==1){
    if(debug>0)err_msg("INFO: user [%s] accesses via NAT", userid);
  }
  return resultFlag;
}

/********************************************************/
/* get user property from property db (NOT implemented) */
/********************************************************/
int getUserProperty(char userid[USERMAXLN], char userProperty[BUFFMAXLN])
{
  /*****************************************************/
  /*  user database access routine                     */
  /*  userid[USERMAXLN]: input: user ID to search      */
  /*               Caution: "userid" if default server,*/
  /*                        "userid@serverid" if not.  */
  /*  userProperty[BUFFMAXLN]: output: user property   */
  /*                   the value goes to ipfwctrl.pl   */
  /*                   to determine permitting grade.  */     
  /*  return value: DENY or ACCEPT                     */
  /*     If DENY is returned, the user is denyed.      */
  /*     If ACCEPT, the value sent to firewall open.   */
  /*****************************************************/

  /* REPLACE THE FOLLOWING WITH YOUR DATABASE ACCESS ROUTINE */
  userProperty[0]='?';    
  userProperty[1]='\0';    /* The User has No Property */
  return ACCEPT;           /* The User is Accepted     */
}


/******************************************
find the duplicated entry in db for an ipv4 address
if redundant rule exists, positive value is returned
******************************************/
int findDuplicateInDbAndClose(char* clientAddr4, int* redundantRule4, 
			    int* redundantRule6, int* redundantPid ){

  sqlite3 *db;
  sqlite3_stmt *stmt;
  char *pErrMsg;
 
  /* SQL COMMAND, where %x is replaced in mprintf */
  char *selectMeFormat="SELECT ruleNumber4, ruleNumber6 \
 FROM session WHERE clientAddr4='%s' and closeTime='-' and pid=%d";
  char *selectOtherFormat="SELECT ruleNumber4, ruleNumber6, pid \
 FROM session WHERE clientAddr4='%s' and closeTime='-' and pid!=%d";
  char *selectCmd;

  /* SQL UPDATE COMMAND to write close time */
  char *updateFormat="UPDATE session SET closeTime=\
 datetime(%d,'unixepoch','localtime'), watchMode='%s' \
 WHERE clientAddr4='%s' and closeTime='-' and pid=%d";

  /* SQL TRANSACTION COMMAND */
  char *beginTransaction="BEGIN TRANSACTION";
  char *commitTransaction="COMMIT TRANSACTION";

  char *updateCmd;
  int resultFlag=TRUE;
  int ruleNumberMe4=0;
  int ruleNumberMe6=0;
  int ruleNumberOther4=0;
  int ruleNumberOther6=0;

  /* initial values */
  *redundantRule4=0;
  *redundantRule6=0;
  *redundantPid=0;

  /********************prepare db**********************/
  /* open sqlite */
  if(sqlite3_open(GetConfValue("SqliteDb"),&db)!=SQLITE_OK){
    err_msg("ERR at %s#%d: sqlite3_open",__FILE__,__LINE__);
    sqlite3_close(db);
    return FALSE;
  }
  sqlite3_busy_timeout(db, sqliteBusyTimeout);

  /* begin transaction */
  sqlite3_exec(db, beginTransaction, NULL, NULL, &pErrMsg);

  /***************** search other session in db******************/

  /* prepare command string to search other session */
  selectCmd=sqlite3_mprintf(selectOtherFormat, clientAddr4, getpid());
  
  /* compile to internal statement */
  if(sqlite3_prepare(db, selectCmd, BUFFMAXLN, &stmt, NULL)!=SQLITE_OK){
    resultFlag=FALSE;
    err_msg("ERR at %s#%d: sqlite3_prepare:%s",__FILE__,__LINE__,
	    sqlite3_errmsg(db));

    /* finalize */
    sqlite3_free(selectCmd);
    sqlite3_finalize(stmt);
    sqlite3_close(db);
    return FALSE;
  }

  /* get first match item */
  if(sqlite3_step(stmt)==SQLITE_ROW){
    ruleNumberOther4=(int)sqlite3_column_int(stmt, 0);
    ruleNumberOther6=(int)sqlite3_column_int(stmt, 1);
    *redundantPid=(int)sqlite3_column_int(stmt, 2);
    resultFlag=TRUE;
  }else{
    resultFlag=FALSE;
  }
  
  /* finalize */
  sqlite3_free(selectCmd);
  sqlite3_finalize(stmt);
  if(resultFlag==FALSE){
    sqlite3_close(db);
    return FALSE;
  }

  /***********if found other, search my session in db ***********/

  /* prepare command string to search my session */
  selectCmd=sqlite3_mprintf(selectMeFormat, clientAddr4, getpid());
  
  /* compile to internal statement */
  if(sqlite3_prepare(db, selectCmd, BUFFMAXLN, &stmt, NULL)!=SQLITE_OK){
    resultFlag=FALSE;
    err_msg("ERR at %s#%d: sqlite3_prepare:%s",__FILE__,__LINE__,
	    sqlite3_errmsg(db));

    /* finalize */
    sqlite3_free(selectCmd);
    sqlite3_finalize(stmt);
    sqlite3_close(db);
    return FALSE;
  }

  /* get first match item */
  if(sqlite3_step(stmt)==SQLITE_ROW){
    ruleNumberMe4=(int)sqlite3_column_int(stmt, 0);
    ruleNumberMe6=(int)sqlite3_column_int(stmt, 1);
    resultFlag=TRUE;
  }else{
    resultFlag=FALSE;
  }

  /* finalize */
  sqlite3_free(selectCmd);
  sqlite3_finalize(stmt);
  if(resultFlag==FALSE){
    sqlite3_close(db);
    return FALSE;
  }

  /*****************write close of my session**********/

  /* prepare command to write close time */
  updateCmd=sqlite3_mprintf(updateFormat, time(NULL), "NONE",
			    clientAddr4, getpid());

  /* execute replace to sqlite */
  if(sqlite3_exec(db, updateCmd, NULL, NULL, &pErrMsg)!=SQLITE_OK){
    resultFlag=FALSE;
    err_msg("ERR at %s#%d: sqlite3_exec: %s",__FILE__,__LINE__,pErrMsg);
  }

  /*memory free for sqlite3 string */
  sqlite3_free(updateCmd);

  /***************finalize db***********************/

  /* end transaction and sqlite close */
  sqlite3_exec(db, commitTransaction, NULL, NULL, &pErrMsg);
  sqlite3_close(db);

  /******************check overlapped rule*************/

  /* if rule number is not match, my rule is redundant */
  if(resultFlag==TRUE){
    if(ruleNumberMe4==ruleNumberOther4) *redundantRule4=0;
    else *redundantRule4=ruleNumberMe4;
    if(ruleNumberMe6==ruleNumberOther6) *redundantRule6=0;
    else *redundantRule6=ruleNumberMe6;
  }

  return resultFlag;
}

/********************************************
Delete old session records in db
 (closed before 1 month)
********************************************/
int deleteOldSessionInDb(void){
  char *pErrMsg=NULL;
  sqlite3 *db;

  /* SQL DELETE COMMAND */
  char *deleteCmd="DELETE FROM session WHERE closeTime<datetime('now','-1 month')";
  int resultFlag=TRUE;

   /* open sqlite */
  if(sqlite3_open(GetConfValue("SqliteDb"),&db)!=SQLITE_OK){
    err_msg("ERR at %s#%d: sqlite3_open",__FILE__,__LINE__);
    sqlite3_close(db);
    return FALSE;
  }
  sqlite3_busy_timeout(db, sqliteBusyTimeout);
 
  /* execute delete */
  if(sqlite3_exec(db, deleteCmd, NULL, NULL, &pErrMsg)!=SQLITE_OK){
    resultFlag=FALSE;
    err_msg("ERR at %s#%d: sqlite3_exec: %s",__FILE__,__LINE__,pErrMsg);
  }

  sqlite3_close(db);
  return resultFlag;
}


/***************************************************************/
/***************************************************************/
/* debug write routine */
int SetupSqliteBusyTimeoutValue(void){
  int ret;
  if(debug>1) err_msg("DEBUG:=>setupSqliteBusyTimeoutValue()");
  ret=setupSqliteBusyTimeoutValue();
  if(debug>1) err_msg("DEBUG:(%d)<=setupSqliteBusyTimeoutValue()",ret);
  return ret;
}

int PutSessionBeginToDb(char* cookie, char* userid, 
			char* clientAddr4, char* clientAddr6, 
			char* macAddr4, 
			char* ruleNumber4, char* ruleNumber6,
			int duration, int durationEntered, 
			int cookieAuth, char *language){
  int ret;

  if(debug>1) err_msg("DEBUG:=>putSessionBeginToDb(%s,%s,%s,%s,%s,%s,%s,%d,%d,%d,%s)",
		      cookie,userid,clientAddr4,clientAddr6,
		      macAddr4,ruleNumber4,ruleNumber6,
		      duration,durationEntered,cookieAuth,language);
  ret=putSessionBeginToDb(cookie,userid,clientAddr4,clientAddr6,
		      macAddr4,ruleNumber4,ruleNumber6,
			  duration,durationEntered,cookieAuth,language);
  if(debug>1) err_msg("DEBUG:(%d)<=putSessionBeginToDb()",ret);

  return ret;
}

int PutSessionEndToDb(char* cookie, char* watchMode){
  int ret;

  if(debug>1) err_msg("DEBUG:=>putSessionEndToDb(%s,%s)",cookie, watchMode);
  ret=putSessionEndToDb(cookie,watchMode);
  if(debug>1) err_msg("DEBUG:(%d)<=putSessionEndToDb()",ret);

  return ret;
}


int FixProcessEndInDb(int pid, char* watchMode){
  int ret;

  if(debug>1) err_msg("DEBUG:=>fixProcessEndInDb(%s,%s)", pid, watchMode);
  ret=fixProcessEndInDb(pid,watchMode);
  if(debug>1) err_msg("DEBUG:(%d)<=fixProcessEndInDb()",ret);

  return ret;
}


int GetSessionInfoFromDb(char* cookie, char* userid, char* clientAddr4, 
			 char *macAddr, int *duration, int *durationEntered,
			 char *language, char* closeTime){
  int ret;

  if(debug>1) err_msg("DEBUG:=>getInfoFromDb(%s)",cookie);
  ret=getSessionInfoFromDb(cookie,userid,clientAddr4, macAddr,
			   duration,durationEntered,language,closeTime);
  if(debug>1) err_msg("DEBUG:(%d)<=getInfoFromDb(%s,%s,%s,%s,%d,%d,%s)",
		      ret,cookie,userid,clientAddr4, macAddr, 
		      *duration,*durationEntered,language,closeTime);
  return ret;
}


int GetUserProperty(char *userid, char *userProperty)
{
  int ret;

  if(debug>1) err_msg("DEBUG:=>getUserProperty(%s,)",userid);
  ret=getUserProperty(userid,userProperty);
  if(debug>1) err_msg("DEBUG:(%d)<=getUserProperty(,%s)",ret, userProperty);

  return ret;
}

int CheckNatInsertion(char* macAddr4, char* macAddr6, char* userid){
  int ret;
  if(debug>1) err_msg("DEBUG:=>checkNatInsertion(%s,%s,%s)",macAddr4,macAddr6,userid);
  ret=checkNatInsertion(macAddr4,macAddr6,userid);
  if(debug>1) err_msg("DEBUG:(%d)<=checkNatInsertion( )",ret);
  return ret;
}

int FindDuplicateInDbAndClose(char* clientAddr4, int* redundantRule4, 
			    int* redundantRule6, int* redundantPid ){
  int ret;
  if(debug>1) err_msg("DEBUG:=>findDuplicateInDbAndClose(%s,)",clientAddr4);
  ret=findDuplicateInDbAndClose(clientAddr4,redundantRule4,redundantRule6, redundantPid);
  if(debug>1) err_msg("DEBUG:(%d)<=findDuplicateInDbAndClose(,%d,%d,%d)",ret, 
		      *redundantRule4,*redundantRule6, *redundantPid);
  return ret;
}

int DeleteOldSessionInDb(void){
  int ret;

  if(debug>1) err_msg("DEBUG:=>deleteOldSessionInDb()");
  ret=deleteOldSessionInDb();
  if(debug>1) err_msg("DEBUG:(%d)<=deleteOldSessionInDb()",ret);

  return ret;
}
