﻿using System;
using System.Windows.Forms;
using System.IO;
using Shell32;
using SHDocVw;

namespace DiskRefresh
{
    public partial class MainForm : Form
    {
        /// <summary>
        /// 検出するデバイスのフレンドリネーム(部分一致)
        /// </summary>
        private const string usbDiskFriendlyName = " USB ";

        /// <summary>
        /// エクスプローラのファイル名(ToUpperで比較すること)
        /// </summary>
        private const string explorerexe = "EXPLORER.EXE";

        /// <summary>
        /// ファイルを示すURI(先頭一致)
        /// </summary>
        private const string fileURI = "file:///";

        /// <summary>
        /// アクション後にタイマ動作(デバイスポーリング)を禁止する時間(ms)
        /// </summary>
        private const int refreshResultTimer = 2500;

        /// <summary>
        /// アクション後の経過時間カウンタ(refreshResultTimerからのダウンカウンタ) 
        /// </summary>
        private int refreshResultCounter = 0;

        /// <summary>
        /// 有効状態の論理ドライブ名並び
        /// </summary>
        private string[] enabledLogicalDrives = null;

        /// <summary>
        /// 無効状態の論理ドライブ名並び
        /// </summary>
        private string[] disabledLogicalDrives = null;
 
        /// <summary>
        /// デバイス無効化後の待ち時間(合計時間)
        /// </summary>
        private readonly int waitAfterDisabled = 1500;

        /// <summary>
        /// デバイス無効化後の待ち時間(DoEventsの間隔)
        /// </summary>
        private readonly int waitAfterDisabledUnit = 50;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public MainForm()
        {
            InitializeComponent();
        }

        /// <summary>
        /// フォームがロードされた
        /// </summary>
        /// <param name="sender">イベント送信元</param>
        /// <param name="e">イベントパラメータ</param>
        private void MainForm_Load(object sender, EventArgs e)
        {
            // 最初のラベルを設定
            isDiskConnected();

            // デバイスポーリングのためのタイマを開始
            RefreshTimer.Enabled = true;
        }

        /// <summary>
        /// タイマが一定時間経過した(100ms間隔)
        /// </summary>
        /// <param name="sender">イベント送信元</param>
        /// <param name="e">イベントパラメータ</param>
        private void RefreshTimer_Tick(object sender, EventArgs e)
        {
            // ドライブ並びが両方とも有効な場合は
            if ((enabledLogicalDrives != null) && (disabledLogicalDrives != null))
            {
                // 無効化したディスクに対応するドライブレターを検出し、エクスプローラで開く
                openDrive();
            }
            
            
            // タイマが有効な場合は
            if (refreshResultCounter > 0)
            {
                // Tickを減算
                refreshResultCounter -= RefreshTimer.Interval;

            }

            // タイマが無効な場合は
            if (refreshResultCounter <= 0)
            {
                // ディスクの接続チェック
                isDiskConnected();
            }
        }

        /// <summary>
        /// リフレッシュボタンが押された
        /// </summary>
        /// <param name="sender">イベント送信元</param>
        /// <param name="e">イベントパラメータ</param>
        private void RefreshButton_Click(object sender, EventArgs e)
        {
            // タイマを設定
            refreshResultCounter = refreshResultTimer;

            // ボタンを禁止
            RefreshButton.Enabled = false;

            // 砂時計カーソル
            this.Cursor = Cursors.WaitCursor;

            // デバイスの存在チェック
            GuideLabel.Text = "USBディスクの存在をチェックしています…";
            Application.DoEvents();
            bool result = SetupDi.isDevicePresent(usbDiskFriendlyName);
            if (!result)
            {
                this.Cursor = Cursors.Default;
                GuideLabel.Text = "USBディスクが見つかりません\nデバイスの接続を確認してから、再トライしてください";
                DeviceLabel.Text = "";
                return;
            }

            // リムーバブルドライブがエクスプローラで開かれていれば閉じる
            closeRemovableDrive();

            // 有効時の論理ドライブ並びを取得
            enabledLogicalDrives = getLogicalDrives();

            // デバイスの無効化
            GuideLabel.Text = "USBディスクをリフレッシュしています\nしばらくお待ちください…";
            Application.DoEvents();
            result = SetupDi.disableDevice(usbDiskFriendlyName);
            if (!result)
            {
                this.Cursor = Cursors.Default;
                GuideLabel.Text = "USBディスクを無効にできません\n開いているファイルをすべて閉じて、再トライしてください";
                return;
            }

            // ここでスリープ
            GuideLabel.Text = "USBディスクを無効化しました\nしばらくお待ちください…";
            Application.DoEvents();
            for (int waitTime = 0; waitTime < waitAfterDisabled; waitTime += waitAfterDisabledUnit)
            {
                System.Threading.Thread.Sleep(waitAfterDisabledUnit);
                Application.DoEvents();
            }

            // 無効時の論理ドライブ並びを取得
            disabledLogicalDrives = getLogicalDrives();

            // デバイスの有効化を試みる
            GuideLabel.Text = "USBディスクを有効化しています\nしばらくお待ちください";
            Application.DoEvents();
            result = SetupDi.enableDevice(usbDiskFriendlyName);
            if (!result)
            {
                this.Cursor = Cursors.Default;
                GuideLabel.Text = "USBディスクを有効にできません\nツールを再起動して、再トライしてください";
                return;
            }

            // デバイスをリフレッシュした
            this.Cursor = Cursors.Default;
            GuideLabel.Text = "USBディスクをリフレッシュしました\nエクスプローラが開くまで、しばらくお待ちください";
        }
        
        /// <summary>
        /// ディスクの接続チェック
        /// </summary>
        private void isDiskConnected()
        {
            string deviceName;
            
            // デバイスの存在有無を調べる
            bool result = SetupDi.isDevicePresent(usbDiskFriendlyName);

            if (result)
            {
                // ラベルを設定(USBディスクあり)
                GuideLabel.Text = "スキャン後に”リフレッシュ”ボタンを押してください";

                // デバイスのフレンドリネームを設定
                deviceName = "検出されたデバイス：" + SetupDi.getPresentFriendlyName();
                DeviceLabel.Text = deviceName;

                // ボタンを押せるようにする
                RefreshButton.Enabled = true;
            }
            else
            {
                // ラベルを設定(USBディスクなし)
                GuideLabel.Text = "USBディスクが接続されていません。接続を待っています…";

                // デバイスのフレンドリネームは無し
                DeviceLabel.Text = "検出されたデバイス：なし";

                // ボタンを押せないようにする
                RefreshButton.Enabled = false;
            }
        }

        /// <summary>
        /// リムーバブルドライブがエクスプローラで開かれているか調べ、開かれていれば閉じる
        /// </summary>
        private void closeRemovableDrive()
        {
            int hWnd;

            // リムーバブルドライブのルートパスを取得し、見つからないか2ドライブ以上なら何もしない
            string removableDriveName = SetupDi.getRemovableDriveName();
            if (removableDriveName == null)
            {
                return;
            }

            // エクスプローラで開かれている限り
            while (isOpenDrive(removableDriveName, out hWnd))
            {
                // ウィンドウハンドルにWM_CLOSEメッセージをポストし、閉じられるまで待つ
                SetupDi.closeWindowFromHandle(hWnd);
            }
        }

        /// <summary>
        /// 無効化したディスクに対応するドライブレターを検出し、エクスプローラで開く
        /// </summary>
        private void openDrive()
        {
            int hWnd;

            // 現在の論理ドライブ並びを取得
            string[] currentLogicalDrives = getLogicalDrives();

            // 現在の論理ドライブ並びが無効であれば何もしない
            if (currentLogicalDrives == null)
            {
                return;
            }

            // 有効時と異なっていれば、まだドライブが認識されていないことを示すため何もしない
            if (currentLogicalDrives.Length != enabledLogicalDrives.Length)
            {
                return;
            }

            // 無効時と同じであれば、ドライブの差を検出できないため何もしない
            if (currentLogicalDrives.Length == disabledLogicalDrives.Length)
            {
                return;
            }

            // 現在のドライブ名を回る
            foreach (string drive in currentLogicalDrives)
            {
                bool match = false;

                // 現在のドライブ名が、無効時のドライブ名並びに含まれている文字列か調べる
                foreach (string disabledDrive in disabledLogicalDrives)
                {
                    if (drive.Equals(disabledDrive))
                    {
                        match = true;
                        break;
                    }
                }

                // 含まれていれば、無効化したデバイスと関係ないドライブなので、continue
                if (match)
                {
                    continue;
                }

                // 有効時と無効時のドライブ並びを初期化
                enabledLogicalDrives = null;
                disabledLogicalDrives = null;

                // タイマを止める
                RefreshTimer.Enabled = false;

                // ドライブが開かれていなければ
                if (!isOpenDrive(drive, out hWnd))
                {
                    // このドライブをエクスプローラで開く
                    System.Diagnostics.Process.Start(drive);
                }

                // フォームを閉じる
                Close();

                // 終了
                return;
            }
        }

        /// <summary>
        /// 論理ドライブ名の並びを取得
        /// </summary>
        /// <returns>ドライブ文字:\スタイルの文字列配列</returns>
        private string[] getLogicalDrives()
        {
            string[] drives = null;

            try
            {
                // 論理ドライブ名を"<ドライブ文字>:\"の形式で取得
                drives = System.IO.Directory.GetLogicalDrives();
            }

            // I/Oエラーが発生した(ディスクエラーなど)
            catch (System.IO.IOException)
            {
                return null;
            }

            // 呼び出し元に必要なアクセス許可がない
            catch (System.Security.SecurityException)
            {
                return null;
            }

            // 文字列配列として返す
            return drives;
        }

        /// <summary>
        /// 指定されたドライブがエクスプローラで開かれているか調べる
        /// </summary>
        /// <param name="drive">ドライブ文字:\スタイルの文字列(エクスプローラのキャプションと一致していること)</param>
        /// <param name="hWnd">開かれている場合のウィンドウハンドル</param>
        /// <returns>ステータス(true:開かれている、false:開かれていない)</returns>
        private bool isOpenDrive(string drive, out int hWnd)
        {
            // ドライブレターは逆スラッシュからスラッシュに変更
            string driveLetter = drive.Replace("\\", "/");

            // fileURIを追加する
            string fileURIdrive = fileURI + driveLetter;
            
            // COMのShellクラスを作成(Microsoft Shell Controls And Automationの参照が必要:using Shell32)
            var shellAppType = Type.GetTypeFromProgID("Shell.Application");
            dynamic shell = Activator.CreateInstance(shellAppType);

            // IEとエクスプローラの一覧を取得(Microsoft Internet Controlの参照が必要:using SHDocVw)
            var win = shell.Windows();

            // すべてのウィンドウを回る
            foreach (IWebBrowser2 web in win)
            {
                // エクスプローラに限定
                if (Path.GetFileName(web.FullName).ToUpper() == explorerexe)
                {
                    // 一致比較
                    if (fileURIdrive.Equals(web.LocationURL))
                    {
                        // ウィンドウハンドルを持ち帰る
                        hWnd = web.HWND;

                        // エクスプローラで開かれている
                        return true;
                    }
                }
            }

            // エクスプローラで開かれていない
            hWnd = 0;
            return false;
        }

        /// <summary>
        /// 改版履歴ボタンが押された
        /// </summary>
        /// <param name="sender">イベント送信元</param>
        /// <param name="e">イベントパラメータ</param>
        private void HistoryButton_Click(object sender, EventArgs e)
        {
            // 新しいフォームを作成
            HistoryForm history = new HistoryForm();

            // モーダルダイアログとして表示
            history.ShowDialog();

            // 明示的に後片付け
            history.Dispose();
        }
    }
}
