/*
 * CAN Target Simulator
 * for M16C/5M family + NC30WA v5.45
 *
 * [ UART ]
 * ()ch=0  ch=3̂݃T|[g
 */

#include "common.h"
#include "target.h"
#include "sfr5m.h"
#include "log.h"
#include "debug.h"
#include "uart.h"

/*
 * UART
 * 萔
 */
#define UART_RX_BYTES				(0x80)
										/* Mobt@TCY(DMA) */

/*
 * UART
 * BRGWX^ݒl
 * f1NbN=32MHz̏ꍇ
 */
#define UART_BRG_38400				(51)
										/* 38400bps */
#define UART_BRG_76800				(25)
										/* 76800bps */
#define UART_BRG_115200				(16)
										/* 115200bps */
#define UART_BRG_250000				(7)
										/* 250000bps */
#define UART_BRG_500000				(3)
										/* 500000bps */

/*
 * UART
 * DMAC]vԍ
 */
#define UART0_TX_DSEL				(0x0a)
										/* UART0M荞 */
#define UART0_RX_DSEL				(0x0b)
										/* UART0M荞 */
#define UART3_TX_DSEL				(0x15)
										/* UART3M荞 */
#define UART3_RX_DSEL				(0x16)
										/* UART3M荞 */

/*
 * UART
 * Mobt@\
 */
typedef struct _uart_rx_buf
{
	uint8 NEAR *sfr_con;
										/* DMA䃌WX^ */
	uint16 NEAR *sfr_dar;
										/* DMA]惌WX^ */
	uint16 NEAR *sfr_tcr;
										/* DMA]JE^WX^ */
	uint8 NEAR *sfr_ic;
										/* DMA荞ݐ䃌WX^ */
	uint8 NEAR *sfr_urbh;
										/* UARTMobt@WX^() */
	uint8 NEAR *sfr_uc1;
										/* UARTM䃌WX^1 */
	uint16 overrun;
										/* I[o[G[JE^ */
	uint16 tcr;
										/* OQƂۂ̓]JE^WX^ */
	BOOL side;
										/* gpĂ(true: false:\) */
	uint16 head;
										/* obt@̐擪ʒu */
	uint16 tail;
										/* obt@̏I[ʒu */
	uint16 num;
										/* obt@̗LoCg */
	uint8 buf[UART_RX_BYTES * 2];
										/* obt@(_uobt@) */
} uart_rx_buf;

/*
 * UART
 * Mobt@
 */
static uart_rx_buf uart_rx_obj[2];
										/* Mobt@ */

/*
 * UART
 * MDMAJn
 * 荞݋֎~ŌĂ΂
 */
static void uart_rx_dma(uart_rx_buf NEAR *object)
{
	/* DMA~ */
	*(object->sfr_con) = 0x21;

	/* DMA]AhX */
	if (object->side == FALSE)
	{
		/* \ */
		*(object->sfr_dar) = (uint16)&(object->buf[ 0 ]);
	}
	else
	{
		/*  */
		*(object->sfr_dar) = (uint16)&(object->buf[ (uint16)(sizeof(object->buf) / 2) ]);
	}

	/* DMA]JE^ */
	*(object->sfr_tcr) = (uint16)((sizeof(object->buf) / 2) - 1);

	/* OQƎDMA]JE^WX^𓯈lɃZbg */
	object->tcr = (uint16)((sizeof(object->buf) / 2) - 1);

	/* DMA荞݂ */
	*(object->sfr_ic) = INTR_DEF_LEVEL;

	/* DMAJn */
	*(object->sfr_con) = 0x29;
}

/*
 * UART
 * MDMA
 * 荞݋֎~ŌĂ΂
 */
static void uart_rx_complete(uart_rx_buf NEAR *object)
{
	/* obt@̕\𔻒 */
	if (object->side == FALSE)
	{
		/* \ */
		ASSERT(object->tail < (sizeof(object->buf) / 2));

		/* \ */
		object->num += ((sizeof(object->buf) / 2) - object->tail);
		object->tail = (uint16)(sizeof(object->buf) / 2);

		/* obt@̊Jnʒu𔽓] */
		object->side = TRUE;
	}
	else
	{
		/*  */
		ASSERT((sizeof(object->buf) / 2) <= object->tail);

		/*  */
		object->num += (sizeof(object->buf) - object->tail);
		object->tail = 0;

		/* obt@̊Jnʒu𔽓] */
		object->side = FALSE;
	}

	/* head̍XVxAI[o[t[ꍇ */
	if (object->num >= sizeof(object->buf))
	{
		object->num = sizeof(object->buf);
		object->head = object->tail;
	}
}

/*
 * UART
 * MDMAĎ(|[O)
 * 荞݋֎~ŌĂ΂
 */
static void uart_rx_poll(uart_rx_buf NEAR *object)
{
	uint16 tcr;
	uint16 diff;

	/* ݂DMA]JE^WX^擾 */
	tcr = *(object->sfr_tcr);

	/* OQƒlƂ̍ꍇ̂ */
	if (object->tcr != tcr)
	{
		/* ŏIoCg͊荞݂ŏ邽߁Ał͈Ȃ */
		if (object->tcr > 0)
		{
			/* Zo */
			diff = object->tcr - tcr;

			/* QƒlŐVDMA]JE^WX^lɍXV */
			object->tcr = tcr;

			/* tailnumXV */
			object->tail += diff;
			object->num += diff;

			/* head̍XVxAI[o[t[ꍇ */
			if (object->num >= sizeof(object->buf))
			{
				object->num = sizeof(object->buf);
				object->head = object->tail;
			}
		}
	}

	/* I[o[G[`FbN */
	if ((*(object->sfr_urbh) & 0x10) != 0)
	{
		/* I[o[G[ */
		object->overrun++;

		/* Mꎞ~AĊJ */
		*(object->sfr_uc1) &= ~0x04;
		*(object->sfr_uc1) |= 0x04;

		LOG_U2("[UART]detect overrun error", object->overrun);
	}
}

/*
 * UART
 * DMACWX^M(M)
 */
static void uart_init_rx(uint8 ch)
{
	if (ch == 0)
	{
		/* DMA0~ */
		dm0con = 0x21;

		/* DMA0]v */
		dm0sl = UART0_RX_DSEL;

		/* DMA0]AhX */
		sar0h = 0x00;
		sar0m = (uint8)((uint16)(&u0rbl) >> 8);
		sar0l = (uint8)&u0rbl;

		/* DMA0]AhX */
		dar0h = 0x00;

		/* DMA0荞݂֎~ */
		dm0ic = 0x00;
	}
	else
	{
		/* DMA1~ */
		dm1con = 0x21;

		/* DMA1]v */
		dm1sl = UART3_RX_DSEL;

		/* DMA1]AhX */
		sar1h = 0x00;
		sar1m = (uint8)((uint16)(&u3rbl) >> 8);
		sar1l = (uint8)&u3rbl;

		/* DMA1]AhX */
		dar1h = 0x00;

		/* DMA1荞݂֎~ */
		dm1ic = 0x00;
	}
}

/*
 * UART
 * DMACWX^ݒ(M)
 */
static void uart_init_tx(uint8 ch)
{
	if (ch == 0)
	{
		/* DMA2~*/
		dm2con = 0x11;

		/* DMA2]v */
		dm2sl = UART0_TX_DSEL;

		/* DMA2]AhX */
		dar2h = 0x00;
		dar2m = (uint8)((uint16)(&u0tbl) >> 8);
		dar2l = (uint8)&u0tbl;

		/* DMA2荞݂֎~ */
		dm2ic = 0x00;
	}
	else
	{
		/* DMA3~ */
		dm3con = 0x11;

		/* DMA3]v */
		dm3sl = UART3_TX_DSEL;

		/* DMA3]AhX */
		dar3h = 0x00;
		dar3m = (uint8)((uint16)(&u3tbl) >> 8);
		dar3l = (uint8)&u3tbl;

		/* DMA3荞݂֎~ */
		dm3ic = 0x00;
	}
}

/*
 * UART
 * BRGWX^ݒl擾
 */
static uint8 uart_get_brg(uint16 baudrate)
{
	uint8 brg;

	/* ftHgl38400bpsɐݒ */
	brg = UART_BRG_38400;

	switch (baudrate)
	{
	case 384:
		brg = UART_BRG_38400;
		break;
	case 768:
		brg = UART_BRG_76800;
		break;
	case 1152:
		brg = UART_BRG_115200;
		break;
	case 2500:
		brg = UART_BRG_250000;
		break;
	case 5000:
		brg = UART_BRG_500000;
		break;
	}

	return brg;
}

/*
 * UART
 * Mobt@
 */
static void uart_init_rx_buf(uint8 ch)
{
	uart_rx_buf NEAR *object;

	/* IuWFNg擾 */
	if (ch == 0)
	{
		object = &uart_rx_obj[0];
	}
	else
	{
		object = &uart_rx_obj[1];
	}

	/* o() */
	if (ch == 0)
	{
		/* ch0: DMA0蓖Ă */
		object->sfr_con = (uint8 NEAR *)&dm0con;
		object->sfr_dar = (uint16 NEAR *)&dar0l;
		object->sfr_tcr = (uint16 NEAR *)&tcr0;
		object->sfr_ic = (uint8 NEAR *)&dm0ic;

		/* u0rbhu0c1 */
		object->sfr_urbh = (uint8 NEAR *)&u0rbh;
		object->sfr_uc1 = (uint8 NEAR *)&u0c1;
	}
	else
	{
		/* ch3: DMA1蓖Ă */
		object->sfr_con = (uint8 NEAR *)&dm1con;
		object->sfr_dar = (uint16 NEAR *)&dar1l;
		object->sfr_tcr = (uint16 NEAR *)&tcr1;
		object->sfr_ic = (uint8 NEAR *)&dm1ic;

		/* u3rbhu3c1 */
		object->sfr_urbh = (uint8 NEAR *)&u3rbh;
		object->sfr_uc1 = (uint8 NEAR *)&u3c1;
	}

	/* o() */
	object->head = 0;
	object->tail = 0;
	object->num = 0;

	/* I[o[G[Ȃ */
	object->overrun = 0;

	/* ŏ̓obt@̐擪M */
	object->side = FALSE;
}

/*
 * UART
 * 
 * ^C}ōs
 */
void uart_init(uint8 ch, uint16 baudrate)
{
	uint8 brg;

	/* UCLKSELݒ肷ƁAUART0`UART4Đݒ肷Kv */
	if (ch == 0)
	{
		/* `l0ɌAUCLKSEL(UART0`UART4ׂĂɉe) */
		uclksel0 = 0x00;

		/* UARTM~ */
		u0c1 = 0x02;

		/* 8bitUART[hANbNA1XgbvrbgApeBȂ */
		u0mr = 0x05;

		/* JEg\[Xf1SIOɐݒACTS/RTS@\֎~ALSBt@[Xg */
		u0c0 = 0x18;

		/* rbg[gݒ */
		brg = uart_get_brg(baudrate);
		u0brg = brg;

		/* G[tONA(Mobt@̃_~[ǂݍ) */
		brg = u0rbl;

		/* ps͂̕Ƃ */
		pd6_2 = 0;
		pd6_3 = 0;

		/* 荞ݗv؂ւ(UART0M/LIN0[xo) */
		ifsr43 = 0;

		/* UARTM荞݂֎~ */
		s0tic = 0x00;

		/* UARTM荞݂֎~ */
		s0ric = 0x00;

		/* UARTM */
		u0c1 = 0x07;
	}
	else
	{
		/* UARTM~ */
		u3c1 = 0x02;

		/* 8bitUART[hANbNA1XgbvrbgApeBȂ */
		u3mr = 0x05;

		/* JEg\[Xf1SIOɐݒACTS/RTS@\֎~ALSBt@[Xg */
		u3c0 = 0x18;

		/* rbg[gݒ */
		brg = uart_get_brg(baudrate);
		u3brg = brg;

		/* G[tONA(Mobt@̃_~[ǂݍ) */
		brg = u3rbl;

		/* ps͂̕Ƃ */
		pd3_1 = 0;
		pd3_2 = 0;

		/* 荞ݗv؂ւ(UART3M/CAN0G[) */
		ifsr25 = 0;

		/* 荞ݗv؂ւ(UART3M/CAN1EFCNAbv) */
		ifsr32 = 0;

		/* UARTM荞݂֎~ */
		s3tic = 0x00;

		/* UARTM荞݂֎~ */
		s3ric = 0x00;

		/* UARTM */
		u3c1 = 0x07;
	}

	/* DMAݒ(M) */
	uart_init_tx(ch);

	/* DMAݒ(M) */
	uart_init_rx(ch);

	/* Mobt@ */
	uart_init_rx_buf(ch);

	/* MJn */
	if (ch == 0)
	{
		uart_rx_dma(&uart_rx_obj[0]);
	}
	else
	{
		uart_rx_dma(&uart_rx_obj[1]);
	}
}

/*
 * UART
 * 1o
 */
static void uart_out(uint8 ch, uint8 dat)
{
	if (ch == 0)
	{
		/* Mobt@fBɂȂ܂ő҂ */
		while (txept_u0c0 == 0)
		{
			;
		}

		/* Mobt@փf[^ */
		u0tbl = dat;
	}
	else
	{
		/* Mobt@fBɂȂ܂ő҂ */
		while (txept_u3c0 == 0)
		{
			;
		}

		/* Mobt@փf[^ */
		u3tbl = dat;
	}
}

/*
 * UART
 * M
 * near̈悩瑗M邱
 */
void uart_tx(uint8 ch, uint8 NEAR *buf, uint16 len)
{
	/* 2oCgȏłDMAݒ */
	if (len > 1)
	{
		/* UART荞ݗvtONA */
		if (ch == 0)
		{
			s0tic = 0x00;
		}
		else
		{
			s3tic = 0x00;
		}

		/* DMAݒ */
		if (ch == 0)
		{
			/* DMA2~ */
			dm2con = 0x11;

			/* DMA2]AhX */
			sar2h = 0x00;
			sar2m = (uint8)((uint16)(&buf[1]) >> 8);
			sar2l = (uint8)(&buf[1]);

			/* DMA2]JE^ */
			tcr2 = (uint16)(len - 2);

			/* DMA2荞݂ */
			dm2ic = INTR_DEF_LEVEL;

			/* DMA2Jn */
			dmae_dm2con = 1;
		}
		else
		{
			/* DMA3~ */
			dm3con = 0x11;

			/* DMA3]AhX */
			sar3h = 0x00;
			sar3m = (uint8)((uint16)(&buf[1]) >> 8);
			sar3l = (uint8)(&buf[1]);

			/* DMA3]JE^ */
			tcr3 = (uint16)(len - 2);

			/* DMA3荞݂ */
			dm3ic = INTR_DEF_LEVEL;

			/* DMA3Jn */
			dmae_dm3con = 1;
		}
	}
	else
	{
		/* ȊODMA荞݂֎~ */
		if (ch == 0)
		{
			dm2ic = 0x00;
		}
		else
		{
			dm3ic = 0x00;
		}
	}

	/* 1oCgłUARTM荞݂ */
	if (len == 1)
	{
		if (ch == 0)
		{
			s0tic = INTR_DEF_LEVEL;
		}
		else
		{
			s3tic = INTR_DEF_LEVEL;
		}
	}
	else
	{
		/* 0oCg or 1oCg̏ꍇUARTM荞݂֎~ */
		if (ch == 0)
		{
			s0tic = 0x00;
		}
		else
		{
			s3tic = 0x00;
		}
	}

	/* 1oCgȏőMWX^֏ */
	if (len > 0)
	{
		uart_out(ch, buf[0]);
	}
}

/*
 * UART
 * ^XN
 */
void uart_task(void)
{
	uint8 flg;

	/* 荞݋֎~ */
	flg = cpu_di();

	/* M|[O(ch0) */
	uart_rx_poll(&uart_rx_obj[0]);

	/* M|[O(ch3) */
	uart_rx_poll(&uart_rx_obj[1]);

	/* 荞ݕA */
	cpu_ei(flg);
}

/*
 * UART
 * Mf[^擾
 */
uint16 uart_get(uint8 ch, uint8 NEAR *buf, uint16 bytes)
{
	uint8 flg;
	uint16 num;
	uint16 head;
	uint16 loop;
	uint16 rest;
	uart_rx_buf NEAR *object;

	/* IuWFNg擾 */
	if (ch == 0)
	{
		object = &uart_rx_obj[0];
	}
	else
	{
		object = &uart_rx_obj[1];
	}

	/* rest */
	rest = 0;

	/* 荞݋֎~ */
	flg = cpu_di();

	/* object->num擾 */
	num = object->num;

	/* object->head擾 */
	head = object->head;

	/* 荞ݕA */
	cpu_ei(flg);

	/* num0ȏɌ */
	if (num > 0)
	{
		/* bytesɑ΂num傫邩 */
		if (num > bytes)
		{
			/* numbytes܂Ő */
			num = bytes;
		}

		/* headȍ~̌ɑ΂đ傫邩 */
		if ((uint16)(sizeof(object->buf) - head) < num)
		{
			/* 2ɕăRs[ */
			rest = num - (uint16)(sizeof(object->buf) - head);
			num -= rest;
		}

		/* 1 */
		for (loop=0; loop<num; loop++)
		{
			*buf++ = object->buf[head++];
		}

		/* 2 */
		if (rest > 0)
		{
			for (loop=0; loop<rest; loop++)
			{
				*buf++ = object->buf[loop];
			}
		}
	}

	/* Rs[TCYԂ */
	return (num + rest);
}

/*
 * UART
 * Mf[^p
 */
void uart_discard(uint8 ch, uint16 bytes)
{
	uint8 flg;
	uart_rx_buf NEAR *object;

	/* IuWFNg擾 */
	if (ch == 0)
	{
		object = &uart_rx_obj[0];
	}
	else
	{
		object = &uart_rx_obj[1];
	}

	/* 荞݋֎~ */
	flg = cpu_di();

	/* bytesobject->num̂A菬ȕ */
	if (bytes > object->num)
	{
		bytes = object->num;
	}

	/* object->tailZ */
	object->tail += bytes;

	/* I[o[t[ĂΖ߂ */
	if (object->tail >= sizeof(object->buf))
	{
		object->tail -= (uint16)(sizeof(object->buf));
	}

	/* object->numZ */
	object->num -= bytes;

	/* 荞ݕA */
	cpu_ei(flg);
}

/*
 * UART0
 * M荞݃nh
 */
#pragma INTERRUPT /B uart0_tx_isr(vect=17)
void uart0_tx_isr(void)
{
	/* UART0M荞݂֎~ */
	s0tic = 0x00;

	/* OM荞 */
	log_isr(0);
}

/*
 * UART3
 * M荞݃nh
 */
#pragma INTERRUPT /B uart3_tx_isr(vect=50)
void uart3_tx_isr(void)
{
	/* UART3M荞݂֎~ */
	s3tic = 0x00;

	/* OM荞 */
	log_isr(3);
}

/*
 * DMA0
 * ]荞݃nh
 */
#pragma INTERRUPT /B dma0_isr(vect=11)
void dma0_isr(void)
{
	/* DMA] */
	uart_rx_complete(&uart_rx_obj[0]);

	/* _uobt@ŎMp */
	uart_rx_dma(&uart_rx_obj[0]);
}

/*
 * DMA1
 * ]荞݃nh
 */
#pragma INTERRUPT /B dma1_isr(vect=12)
void dma1_isr(void)
{
	/* DMA] */
	uart_rx_complete(&uart_rx_obj[1]);

	/* _uobt@ŎMp */
	uart_rx_dma(&uart_rx_obj[1]);
}

/*
 * DMA2
 * ]荞݃nh
 */
#pragma INTERRUPT /B dma2_isr(vect=41)
void dma2_isr(void)
{
	/* DMA2]荞݂֎~ */
	dm2ic = 0x00;

	/* UART0M荞݂ */
	s0tic = INTR_DEF_LEVEL;

	/* UART0MłȂ΁AUART0MƂ݂Ȃ */
	if (ti_u0c1 == 1)
	{
		/* UART0M荞݂֎~ */
		s0tic = 0x00;

		/* OM荞 */
		log_isr(0);
	}
}

/*
 * DMA3
 * ]荞݃nh
 */
#pragma INTERRUPT /B dma3_isr(vect=42)
void dma3_isr(void)
{
	/* DMA3]荞݂֎~ */
	dm3ic = 0x00;

	/* UART3M荞݂ */
	s3tic = INTR_DEF_LEVEL;

	/* UART3MłȂ΁AUART1MƂ݂Ȃ */
	if (ti_u3c1 == 1)
	{
		/* UART3M荞݂֎~ */
		s3tic = 0x00;

		/* OM荞 */
		log_isr(3);
	}
}
