/*
 * CAN Target Simulator
 * for M16C/5M family + NC30WA v5.45
 *
 * [ O ]
 */

#include "common.h"
#include "target.h"
#include "timer.h"
#include "uart.h"
#include "log.h"

/*
 * O
 * 萔
 */
#define LOG_UART_CH					(3)
										/* UART`l */
#define LOG_STACK_BYTES				(0x80)
										/* X^bNɊmۂobt@̃TCY */
#define LOG_BUF_BYTES				(0x200)
										/* Oobt@̃TCY */
#define LOG_STAMP_LEN				(19)
										/* ^CX^ṽoCg */
#define LOG_TERM_LEN				(2)
										/* I[̃oCg */
#define LOG_UINT8_LEN				(4)
										/* uint8\̃oCg */
#define LOG_UINT16_LEN				(6)
										/* uint16\̃oCg */
#define LOG_UINT32_LEN				(10)
										/* uint32\̃oCg */
#define LOG_5DEC_LEN				(5)
										/* 10i\(5)̃oCg */
#define LOG_TERM_CR					(0x0d)
										/* CR */
#define LOG_TERM_LF					(0x0a)
										/* LF */

/*
 * O
 * Oobt@
 */
static uint8 log_ring_buf[LOG_BUF_BYTES];
										/* Oobt@ */
static uint16 log_ring_num;
										/* Oobt@̗LoCg */
static uint16 log_ring_head;
										/* Oobt@̐擪ʒu */
static uint16 log_ring_tail;
										/* Oobt@̏I[ʒu */
static uint16 log_ring_tx;
										/* Oobt@̑MoCg */

/*
 * O
 * 
 */
void log_init(void)
{
	/* Oobt@ */
	log_ring_num = 0;
	log_ring_head = 0;
	log_ring_tail = 0;
	log_ring_tx = 0;
}

/*
 * O
 * UARTM
 * 荞݋֎~ԂŌĂ΂
 */
static void log_uart_tx(void)
{
	/* log_ring_numLłꍇ */
	if (log_ring_num > 0)
	{
		/* log_ring_txMłȂꍇ */
		if (log_ring_tx == 0)
		{
			/* I[܂ł̃oCg𓾂 */
			if (log_ring_head < log_ring_tail)
			{
				/* ׂĈCɑM */
				log_ring_tx = (uint16)(log_ring_tail - log_ring_head);
			}
			else
			{
				/* headI[܂ł𑗐M */
				log_ring_tx = (uint16)(LOG_BUF_BYTES - log_ring_head);
			}

			/* UARTDMAMJn */
			uart_tx(LOG_UART_CH, &log_ring_buf[log_ring_head], log_ring_tx);
		}
	}
}

/*
 * O
 * Oobt@֑}
 */
static void log_insert(uint8 NEAR *buf, uint16 len)
{
	uint8 flg;
	uint16 rest;
	uint16 loop;

	/* 荞݋֎~ */
	flg = cpu_di();

	/* Rs[TCY */
	if ((uint16)(log_ring_num + len) > LOG_BUF_BYTES)
	{
		len = LOG_BUF_BYTES - log_ring_num;
	}

	/* Rs[łꍇɌ */
	if (len > 0)
	{
		/* Rs[񐔂 */
		if ((uint16)(log_ring_tail + len) > LOG_BUF_BYTES)
		{
			/* 2 */
			rest = (uint16)(len - (uint16)(LOG_BUF_BYTES - log_ring_tail));
			len -= rest;
		}
		else
		{
			/* 1 */
			rest = 0;
		}

		/* 1ڂ̃Rs[ */
		for (loop=0; loop<len; loop++) {
			log_ring_buf[log_ring_tail++] = *buf++;
		}
		log_ring_num += len;
		if (log_ring_tail == LOG_BUF_BYTES)
		{
			log_ring_tail = 0;
		}

		/* 2ڂ̃Rs[ */
		if (rest > 0)
		{
			for (loop=0; loop<rest; loop++) {
				log_ring_buf[loop] = *buf++;
			}
			log_ring_tail = rest;
			log_ring_num += rest;
		}

		/* UARTM */
		log_uart_tx();
	}

	/* 荞ݕA */
	cpu_ei(flg);
}

/*
 * O
 * 10itH[}bg(3)
 */
static void log_format_3dec(uint8 NEAR *buf, uint16 value)
{
	uint8 div;
	uint8 mod;

	/* 100̌ */
	div = (uint8)(value / 100) + (uint8)'0';
	mod = (uint8)(value % 100);
	buf[0] = div;

	/* 10̌ */
	div = (uint8)(mod / 10) + (uint8)'0';
	mod = (uint8)(mod % 10);
	buf[1] = div;

	/* 1̌ */
	div = (uint8)(mod + '0');
	buf[2] = div;
}

/*
 * O
 * 10itH[}bg(4)
 */
static void log_format_4dec(uint8 NEAR *buf, uint16 value)
{
	uint8 div;
	uint16 mod;

	/* 1000̌ */
	div = (uint8)(value / 1000) + (uint8)'0';
	mod = (uint16)(value % 1000);
	buf[0] = div;

	/* 100̌ȍ~ */
	log_format_3dec(&buf[1], mod);
}

/*
 * O
 * 10itH[}bg(5)
 */
static void log_format_5dec(uint8 NEAR *buf, uint16 value)
{
	uint8 div;
	uint16 mod;

	/* 10000̌ */
	div = (uint8)(value / 10000) + (uint8)'0';
	mod = (uint16)(value % 10000);
	buf[0] = div;

	/* 1000̌ȍ~ */
	log_format_4dec(&buf[1], mod);
}

/*
 * O
 * ^CX^vt
 * 19
 */
static void log_format_stamp(uint8 NEAR *buf)
{
	timer_info info;

	/* ^CX^v擾 */
	timer_get(&info);

	/* sec(hi) */
	log_format_5dec(&buf[0], info.sec_hi);
	buf[5] = (uint8)',';

	/* sec(lo) */
	log_format_4dec(&buf[6], info.sec_lo);
	buf[10] = (uint8)',';

	/* ms */
	log_format_3dec(&buf[11], info.ms);
	buf[14] = (uint8)',';

	/* us */
	log_format_3dec(&buf[15], info.us);
	buf[18] = (uint8)',';
}

/*
 * O
 * 16itH[}bg(2)
 */
static void log_format_2hex(uint8 NEAR *buf, uint8 value)
{
	uint8 nibble;

	/* 4bit */
	nibble = (uint8)(value >> 4);
	if (nibble > 9)
	{
		buf[0] = (uint8)(0x37 + nibble);
	}
	else
	{
		buf[0] = (uint8)(0x30 + nibble);
	}

	/* 4bit */
	nibble = (uint8)(value & 0x0f);
	if (nibble > 9)
	{
		buf[1] = (uint8)(0x37 + nibble);
	}
	else
	{
		buf[1] = (uint8)(0x30 + nibble);
	}
}

/*
 * O
 * 16itH[}bg(4)
 */
static void log_format_4hex(uint8 NEAR *buf, uint16 value)
{
	uint8 half;

	/* ʃoCg */
	half = (uint8)(value >> 8);
	log_format_2hex(buf, half);

	/* ʃoCg */
	half = (uint8)value;
	log_format_2hex(&buf[2], half);
}

/*
 * O
 * 16itH[}bg(8)
 */
static void log_format_8hex(uint8 NEAR *buf, uint32 value)
{
	uint16 half;

	/* ʃ[h */
	half = (uint16)(value >> 16);
	log_format_4hex(buf, half);

	/* ʃoCg */
	half = (uint16)value;
	log_format_4hex(&buf[4], half);
}

/*
 * O
 * o
 * far̈ɔzu
 */
void log_msg(const char FAR *msg)
{
	uint8 len;
	uint8 loop;
	uint8 count;
	uint8 buf[LOG_STACK_BYTES];

	/* ^CX^v */
	log_format_stamp(buf);

	/* msgRs[ */
	count = 0;
	for (loop=0; loop<(sizeof(buf) - LOG_STAMP_LEN - LOG_TERM_LEN); loop++)
	{
		if (msg[count] != '\0')
		{
			/* Rs[ */
			buf[LOG_STAMP_LEN + count] = (uint8)msg[count];
			count++;
		}
		else
		{
			/* for[vł؂点 */
			loop = sizeof(buf);
		}
	}

	/* I[ */
	buf[LOG_STAMP_LEN + count + 0] = LOG_TERM_CR;
	buf[LOG_STAMP_LEN + count + 1] = LOG_TERM_LF;

	/* O֑} */
	log_insert(buf, (uint16)(LOG_STAMP_LEN + count + LOG_TERM_LEN));
}

/*
 * O
 * +uint8o
 * far̈ɔzu
 */
void log_uint8(const char FAR *msg, uint8 value)
{
	uint8 len;
	uint8 loop;
	uint8 count;
	uint8 buf[LOG_STACK_BYTES];

	/* ^CX^v */
	log_format_stamp(buf);

	/* msgRs[ */
	count = 0;
	for (loop=0; loop<(sizeof(buf) - LOG_STAMP_LEN - LOG_UINT8_LEN + LOG_TERM_LEN); loop++)
	{
		if (msg[count] != '\0')
		{
			/* Rs[ */
			buf[LOG_STAMP_LEN + count] = (uint8)msg[count];
			count++;
		}
		else
		{
			/* for[vł؂点 */
			loop = sizeof(buf);
		}
	}

	/* uint8 */
	buf[LOG_STAMP_LEN + count + 0] = (uint8)'(';
	log_format_2hex(&buf[LOG_STAMP_LEN + count + 1], value);
	buf[LOG_STAMP_LEN + count + LOG_UINT8_LEN - 1] = (uint8)')';

	/* I[ */
	buf[LOG_STAMP_LEN + count + LOG_UINT8_LEN + 0] = LOG_TERM_CR;
	buf[LOG_STAMP_LEN + count + LOG_UINT8_LEN + 1] = LOG_TERM_LF;

	/* O֑} */
	log_insert(buf, (uint16)(LOG_STAMP_LEN + count + LOG_UINT8_LEN + LOG_TERM_LEN));
}

/*
 * O
 * +uint16o
 * far̈ɔzu
 */
void log_uint16(const char FAR *msg, uint16 value)
{
	uint8 len;
	uint8 loop;
	uint8 count;
	uint8 buf[LOG_STACK_BYTES];

	/* ^CX^v */
	log_format_stamp(buf);

	/* msgRs[ */
	count = 0;
	for (loop=0; loop<(sizeof(buf) - LOG_STAMP_LEN - LOG_UINT16_LEN + LOG_TERM_LEN); loop++)
	{
		if (msg[count] != '\0')
		{
			/* Rs[ */
			buf[LOG_STAMP_LEN + count] = (uint8)msg[count];
			count++;
		}
		else
		{
			/* for[vł؂点 */
			loop = sizeof(buf);
		}
	}

	/* uint16 */
	buf[LOG_STAMP_LEN + count + 0] = (uint8)'(';
	log_format_4hex(&buf[LOG_STAMP_LEN + count + 1], value);
	buf[LOG_STAMP_LEN + count + LOG_UINT16_LEN - 1] = (uint8)')';

	/* I[ */
	buf[LOG_STAMP_LEN + count + LOG_UINT16_LEN + 0] = LOG_TERM_CR;
	buf[LOG_STAMP_LEN + count + LOG_UINT16_LEN + 1] = LOG_TERM_LF;

	/* O֑} */
	log_insert(buf, (uint16)(LOG_STAMP_LEN + count + LOG_UINT16_LEN + LOG_TERM_LEN));
}

/*
 * O
 * +uint32o
 * far̈ɔzu
 */
void log_uint32(const char FAR *msg, uint32 value)
{
	uint8 len;
	uint8 loop;
	uint8 count;
	uint8 buf[LOG_STACK_BYTES];

	/* ^CX^v */
	log_format_stamp(buf);

	/* msgRs[ */
	count = 0;
	for (loop=0; loop<(sizeof(buf) - LOG_STAMP_LEN - LOG_UINT32_LEN + LOG_TERM_LEN); loop++)
	{
		if (msg[count] != '\0')
		{
			/* Rs[ */
			buf[LOG_STAMP_LEN + count] = (uint8)msg[count];
			count++;
		}
		else
		{
			/* for[vł؂点 */
			loop = sizeof(buf);
		}
	}

	/* uint32 */
	buf[LOG_STAMP_LEN + count + 0] = (uint8)'(';
	log_format_8hex(&buf[LOG_STAMP_LEN + count + 1], value);
	buf[LOG_STAMP_LEN + count + LOG_UINT32_LEN - 1] = (uint8)')';

	/* I[ */
	buf[LOG_STAMP_LEN + count + LOG_UINT32_LEN + 0] = LOG_TERM_CR;
	buf[LOG_STAMP_LEN + count + LOG_UINT32_LEN + 1] = LOG_TERM_LF;

	/* O֑} */
	log_insert(buf, (uint16)(LOG_STAMP_LEN + count + LOG_UINT32_LEN + LOG_TERM_LEN));
}

/*
 * O
 * ffo
 * far̈ɔzu
 */
void log_assert(const char FAR *expr, const char FAR *file, int line)
{
	uint8 buf[LOG_STACK_BYTES];

	/* G[bZ[W */
	log_msg("Assertion Failed!");

	/* ]Ɏs */
	log_msg(expr);

	/* \[Xt@Cl[ */
	log_msg(file);

	/* sԍ(^CX^v) */
	log_format_stamp(buf);

	/* sԍ({) */
	log_format_5dec(&buf[LOG_STAMP_LEN], (uint16)line);

	/* I[ */
	buf[LOG_STAMP_LEN + LOG_5DEC_LEN + 0] = LOG_TERM_CR;
	buf[LOG_STAMP_LEN + LOG_5DEC_LEN + 1] = LOG_TERM_LF;

	/* O֑} */
	log_insert(buf, (uint16)(LOG_STAMP_LEN + LOG_5DEC_LEN + LOG_TERM_LEN));

	/* 荞݋ */
	cpu_force_ei();

	/* [v */
	for (;;)
	{
		cpu_nop();
	}
}

/*
 * O
 * UART荞
 * 荞݋֎~ŌĂ΂
 */
void log_isr(uint8 uart_ch)
{
	/* UART`lvꍇ̂ */
	if (uart_ch == LOG_UART_CH)
	{
		/* log_ring_headXV */
		log_ring_head += log_ring_tx;
		if (log_ring_head == LOG_BUF_BYTES)
		{
			log_ring_head = 0;
		}

		/* log_ring_numXV */
		log_ring_num -= log_ring_tx;

		/* log_ring_tx͑M̂ŃNA */
		log_ring_tx = 0;

		/* UARTM */
		log_uart_tx();
	}
}
