package jp.gr.java_conf.wutse.purewiki;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import jp.gr.java_conf.wutse.purewiki.exception.PureWikiValidateException;
import jp.gr.java_conf.wutse.purewiki.exception.ValidateError;
import jp.gr.java_conf.wutse.purewiki.util.PureWikiUtil;

public class PureWikiEngine {
	private static final String EMPTY_TEXT = "";
	private static final String WIKI_EXTENSION = ".wiki";
	private static final String HTML_EXTENSION = ".html";
	private PureWikiConfig config = null;
	private PureWikiTranslator translator = null;

	public PureWikiEngine() {
		config = new PureWikiConfig();
		translator = new PureWikiTranslator(this);
	}

	public PureWikiData createTemporaryWikiData(String wikiName) {
		PureWikiData data = new PureWikiData();
		data.setWikiName(wikiName);
		data.setWikiText(EMPTY_TEXT);
		data.setOutFile(new File(config.getOutDirPath() + "/_tempolary.html"));
		return data;
	}

	/**
	 * srcファイルより、PureWikiDataを生成する。
	 * 
	 * @param srcFile
	 *            srcファイル
	 * @return PureWikiData
	 */
	public PureWikiData createPureWikiData(File srcFile) {
		PureWikiData data = new PureWikiData();
		data.setSrcFile(srcFile);
		data.setOutFile(getOutputFile(srcFile));
		String path = getPath(srcFile);
		data.setPath(path);
		if (path.endsWith(WIKI_EXTENSION)) {
			data.setWikiName(path.substring(0, path.length() - WIKI_EXTENSION.length()));
		} else {
			data.setWikiName(path);
		}
		return data;
	}

	/**
	 * PureWikiDataを保存する。
	 * 
	 * @param data
	 * @throws PureWikiValidateException
	 */
	public void storePureWikiData(PureWikiData data) throws PureWikiValidateException {
		File srcFile = data.getSrcFile();
		if (srcFile == null) {
			List<ValidateError> errorList = new ArrayList<>();
			if (PureWikiUtil.isBlank(data.getWikiName())) {
				errorList.add(new ValidateError("wikiName", "Wiki名を入力してください"));
			}
			if (!errorList.isEmpty()) {
				throw new PureWikiValidateException(errorList);
			}
			data.setPath(data.getWikiName() + WIKI_EXTENSION);
			srcFile = new File(config.getSrcDirPath() + "/" + data.getPath());
			data.setSrcFile(srcFile);
			data.getOutFile().delete();
			data.setOutFile(getOutputFile(srcFile));
		}
		PureWikiUtil.storeFile(srcFile, data.getWikiText());
		data.setOrgWikiText(data.getWikiText());
	}

	/**
	 * PureWikiDataを削除する。
	 * 
	 * @param data
	 */
	public void removePureWikiData(PureWikiData data) {
		File srcFile = data.getSrcFile();
		if (srcFile != null && srcFile.exists()) {
			srcFile.delete();
		}
		File outFile = data.getOutFile();
		if (outFile != null && outFile.exists()) {
			outFile.delete();
		}
	}

	/**
	 * wikiからhtmlへ変換する。
	 * 
	 * @param context
	 * @throws IOException
	 */
	public void translateWiki(PureWikiData data) throws IOException {
		if (data.isWiki()) {
			if (!data.isNew()) {
				loadPureWikiData(data, false);
			}
			PureWikiContext context = new PureWikiContext();
			context.setEngine(this);
			context.setStatus(PureWikiContext.RUNNING);
			context.setWikiData(data);
			translator.translateWiki(context);
		} else {
			PureWikiUtil.copyFile(data.getSrcFile(), data.getOutFile());
		}
		return;
	}

	/**
	 * srcディレクトリ以下のwikiリストをロードします。
	 */
	public List<PureWikiData> getWikiDataList() {
		List<PureWikiData> dataList = new ArrayList<>();
		File srcDir = new File(getConfig().getSrcDirPath());
		loadWikiData(dataList, srcDir);
		Collections.sort(dataList);
		return dataList;
	}

	private void loadWikiData(List<PureWikiData> dataList, File file) {
		if (!file.exists()) {
			return;
		}
		if (file.isDirectory()) {
			File[] children = file.listFiles();
			for (File child : children) {
				loadWikiData(dataList, child);
			}
		} else {
			dataList.add(createPureWikiData(file));
		}
	}

	/**
	 * PureWikiDataのファイルを読み込む。
	 * 
	 * @param data
	 *            PureWikiData
	 * @param force
	 *            強制的に再読み込みを行う場合true
	 */
	public void loadPureWikiData(PureWikiData data, boolean force) {
		// TODO:ファイルが裏で更新された場合も取得したい。＆ 更新されている検知
		if (!data.isLoaded() || (data.isModified() && force)) {
			File srcFile = data.getSrcFile();
			String wikiText = PureWikiUtil.loadFile(data.getSrcFile());
			data.setOrgWikiText(wikiText);
			data.setLastModified(srcFile.lastModified());
			data.setWikiText(wikiText);
		}
	}

	/**
	 * srcファイルよりoutファイルを取得する。
	 * 
	 * @param srcFile
	 *            srcファイル
	 * @return outファイル
	 */
	public File getOutputFile(File srcFile) {
		String srcDirPath = config.getSrcDirPath();
		String outDirPath = config.getOutDirPath();
		String srcFilePath = srcFile.getAbsolutePath();
		String outFilePath = null;

		if (srcFilePath.endsWith(WIKI_EXTENSION)) {
			String wikiName = srcFilePath.substring(srcDirPath.length(), srcFilePath.length()
					- WIKI_EXTENSION.length());
			outFilePath = outDirPath + wikiName + HTML_EXTENSION;
		} else {
			outFilePath = outDirPath + srcFilePath.substring(srcDirPath.length());
		}
		return new File(outFilePath);
	}

	/**
	 * 指定したURLが、ローカルのものかどうかを判別する。
	 * 
	 * @param url
	 *            URL
	 * @return ローカルの場合true、外部の場合false
	 */
	public boolean isLocalUrl(String url) {
		String outDirPath = config.getOutDirPath();
		url = url.replace("/", "\\");
		int index = url.indexOf(outDirPath);
		return index >= 0;
	}

	/**
	 * 指定したURLが、テンポラリのものかどうかを判別する。
	 * 
	 * @param url
	 *            URL
	 * @return テンポラリの場合true、テンポラリでない場合、false
	 */
	public boolean isTemporaryUrl(String url) {
		return url.endsWith("\\_tempolary.html");
	}

	/**
	 * URLより、ローカルwikiへの相対パスを取得する。
	 * 
	 * @param url
	 *            URL
	 * @return ローカルwikiへの相対パス。ローカルのURLでない場合はnull。
	 */
	public String getWikiNameFromUrl(String url) {
		if (!isLocalUrl(url)) {
			return null;
		}
		String outDirPath = config.getOutDirPath();
		url = url.replace("/", "\\");
		int index = url.indexOf(outDirPath);
		String wikiName = null;
		// 拡張子がhtmlのものはwikiに置き換える。
		if (url.endsWith(HTML_EXTENSION)) {
			wikiName = url.substring(index + outDirPath.length() + 1,
					url.length() - HTML_EXTENSION.length());
//			path = path + WIKI_EXTENSION;
		}
		// html以外の拡張子はそのまま
		else {
			wikiName = url.substring(index + outDirPath.length() + 1);
		}
		return wikiName;
	}

	/**
	 * ファイルより、ローカルwikiへの相対パスを取得する。
	 * 
	 * @param srcFile
	 *            ファイル
	 * @return ローカルwikiへの相対パス。設定したソースディレクトリにない場合はnull。
	 */
	public String getPath(File srcFile) {
		String srcDirPath = config.getSrcDirPath();
		String filePath = srcFile.getAbsolutePath();
		int index = filePath.indexOf(srcDirPath);
		if (index >= 0) {
			String path = filePath.substring(index + srcDirPath.length() + 1);
			return path.replace("\\", "/");
		} else {
			return null;
		}
	}

	/**
	 * 相対パスより、wiki名を取得する。
	 * 
	 * @param path
	 *            相対パス
	 * @return wiki名
	 */
	public String getWikiName(String path) {
		// 拡張子がwikiの場合は、拡張子を削除
		if (path.endsWith(WIKI_EXTENSION)) {
			return path.substring(0, path.length() - WIKI_EXTENSION.length());
		} else {
			return null;
		}
	}

	/**
	 * 設定を取得する。
	 * 
	 * @return 設定オブジェクト
	 */
	public PureWikiConfig getConfig() {
		return config;
	}

	/**
	 * 設定を保存する。
	 */
	public void storePureWikiConfig() {
		config.store();
	}

}
