package jp.gr.java_conf.wutse.purewiki;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.Properties;

import jp.gr.java_conf.wutse.purewiki.syntax.BlockSyntax;
import jp.gr.java_conf.wutse.purewiki.syntax.LineSyntax;
import jp.gr.java_conf.wutse.purewiki.syntax.WordSyntax;
import jp.gr.java_conf.wutse.purewiki.util.PureWikiUtil;

public class PureWikiConfig {
	private static final String SRC_DIR = "purewiki.srcdir.";
	private static final String OUT_DIR = "purewiki.outdir";
	private static final String CSS_FILE = "purewiki.cssFile";
	private static final String DEFAULT_WIKI = "purewiki.defaultWiki";
	private static final String WINDOW_MAX = "purewiki.window.maximized";
	private static final String BLOCK_SYNTAX = "purewiki.blockSyntax.";
	private static final String LINE_SYNTAX = "purewiki.lineSyntax.";
	private static final String WORD_SYNTAX = "purewiki.wordSyntax.";

	private Properties systemConfig = null;
	private Properties userConfig = null;
	private List<String> srcDirPathList = new ArrayList<>();
	private String outDirPath = null;
	private String cssFile = null;
	private String defaultWiki = null;
	private boolean maximized = false;
	private List<LineSyntax> lineSyntaxList = new ArrayList<>();
	private List<BlockSyntax> blockSyntaxList = new ArrayList<>();
	private List<WordSyntax> wordSyntaxList = new ArrayList<>();

	public PureWikiConfig() {
		systemConfig = new Properties();

		try {
			systemConfig.load(PureWikiConfig.class.getResourceAsStream("/PureWiki.properties"));
		} catch (IOException ex) {
		}

		userConfig = new Properties();
		Enumeration<Object> keys = systemConfig.keys();
		while (keys.hasMoreElements()) {
			String key = (String) keys.nextElement();
			userConfig.setProperty(key, systemConfig.getProperty(key));
		}

		File configFile = new File("./PureWikiConfig.properties");
		if (configFile.exists()) {
			try {
				userConfig.load(new FileInputStream("./PureWikiConfig.properties"));
			} catch (IOException ex) {
			}
		}

		int i = 0;
		while (userConfig.containsKey(BLOCK_SYNTAX + i)) {
			blockSyntaxList.add(createBlockSyntax(BLOCK_SYNTAX + i));
			i++;
		}
		i = 0;
		while (userConfig.containsKey(LINE_SYNTAX + i)) {
			lineSyntaxList.add(createLineSyntax(LINE_SYNTAX + i));
			i++;
		}
		i = 0;
		while (userConfig.containsKey(WORD_SYNTAX + i)) {
			wordSyntaxList.add(createWordSyntax(WORD_SYNTAX + i));
			i++;
		}

		i = 0;
		while (userConfig.containsKey(SRC_DIR + i)) {
			srcDirPathList.add(userConfig.getProperty(SRC_DIR + i));
			i++;
		}

		// ユーザー設定を読み込む
		if (srcDirPathList.isEmpty()) {
			File srcDir = new File("wiki");
			if (!srcDir.exists()) {
				srcDir.mkdir();
				// サンプルwikiコピー
				InputStream is = PureWikiConfig.class.getResourceAsStream("/TopPage.wiki");
				PureWikiUtil.storeFile(new File(".\\wiki\\TopPage.wiki"),
						PureWikiUtil.loadStream(is));
				// サンプル画像コピー
				try {
					InputStream isJpg = PureWikiConfig.class.getResourceAsStream("/Sample.jpg");
					OutputStream outJpg = new FileOutputStream(".\\wiki\\Sample.jpg");
					PureWikiUtil.copyStream(isJpg, outJpg);
				} catch (IOException ex) {
					ex.printStackTrace();
				}
			}
			srcDirPathList.add(srcDir.getAbsolutePath());
		}

		outDirPath = userConfig.getProperty(OUT_DIR);
		if (PureWikiUtil.isBlank(outDirPath)) {
			File outDir = new File("out");
			if (!outDir.exists()) {
				outDir.mkdir();
			}
			outDirPath = outDir.getAbsolutePath();
		}
		cssFile = userConfig.getProperty(CSS_FILE);
		if (cssFile == null) {
			cssFile = "";
		}
		defaultWiki = userConfig.getProperty(DEFAULT_WIKI);
		if (PureWikiUtil.isBlank(defaultWiki)) {
			defaultWiki = "TopPage";
		}

		maximized = toBoolean(userConfig.getProperty(WINDOW_MAX));
	}

	private BlockSyntax createBlockSyntax(String propKey) {
		try {
			Class<?> blockSyntaxClass = Class.forName(userConfig.getProperty(propKey));
			if (BlockSyntax.class.isAssignableFrom(blockSyntaxClass)) {
				return (BlockSyntax) blockSyntaxClass.newInstance();
			}
		} catch (Exception ex) {
		}
		return null;
	}

	private LineSyntax createLineSyntax(String propKey) {
		try {
			Class<?> lineSyntaxClass = Class.forName(userConfig.getProperty(propKey));
			if (LineSyntax.class.isAssignableFrom(lineSyntaxClass)) {
				return (LineSyntax) lineSyntaxClass.newInstance();
			}
		} catch (Exception ex) {
		}
		return null;
	}

	private WordSyntax createWordSyntax(String propKey) {
		try {
			Class<?> wordSyntaxClass = Class.forName(userConfig.getProperty(propKey));
			if (WordSyntax.class.isAssignableFrom(wordSyntaxClass)) {
				return (WordSyntax) wordSyntaxClass.newInstance();
			}
		} catch (Exception ex) {
		}
		return null;
	}

	public List<String> getSrcDirPathList() {
		return srcDirPathList;
	}

	public String getSrcDirPath() {
		return srcDirPathList.get(0);
	}

	public String getOutDirPath() {
		return outDirPath;
	}

	public void setOutDirPath(String outDirPath) {
		File outDir = new File(outDirPath);
		this.outDirPath = outDir.getAbsolutePath();
	}

	public String getCssFile() {
		return cssFile;
	}

	public void setCssFile(String cssFile) {
		this.cssFile = cssFile;
	}

	public String getDefaultWiki() {
		return defaultWiki;
	}

	public void setDefaultWiki(String defaultWiki) {
		this.defaultWiki = defaultWiki;
	}

	public boolean isMaximized() {
		return maximized;
	}

	public void setMaximized(boolean maximized) {
		this.maximized = maximized;
	}

	public List<LineSyntax> getLineSyntaxList() {
		return lineSyntaxList;
	}

	public void setLineSyntaxList(List<LineSyntax> lineSyntaxList) {
		this.lineSyntaxList = lineSyntaxList;
	}

	public List<BlockSyntax> getBlockSyntaxList() {
		return blockSyntaxList;
	}

	public void setBlockSyntaxList(List<BlockSyntax> blockSyntaxList) {
		this.blockSyntaxList = blockSyntaxList;
	}

	public List<WordSyntax> getWordSyntaxList() {
		return wordSyntaxList;
	}

	public void setWordSyntaxList(List<WordSyntax> wordSyntaxList) {
		this.wordSyntaxList = wordSyntaxList;
	}

	public String getStyle() throws IOException {
		StringBuffer cssSb = new StringBuffer("");
		InputStream cssIs = null;
		Reader cssReader = null;
		BufferedReader cssBr = null;
		try {
			if (!PureWikiUtil.isBlank(getCssFile())) {
				cssReader = new FileReader(getCssFile());
			} else {
				cssIs = PureWikiConfig.class.getResourceAsStream("/PureWiki.css");
//				cssIs = new FileInputStream("./PureWiki.css");
				cssReader = new InputStreamReader(cssIs);
			}
			cssBr = new BufferedReader(cssReader);
			String line = null;
			while ((line = cssBr.readLine()) != null) {
				cssSb.append(line);
				cssSb.append(System.lineSeparator());
			}
		} finally {
			PureWikiUtil.close(cssBr);
			PureWikiUtil.close(cssReader);
			PureWikiUtil.close(cssIs);
		}
		return cssSb.toString();
	}

	public void store() {
		OutputStream os = null;
		try {
			for (int i = 0; i < srcDirPathList.size(); i++) {
				String srcDirPath = srcDirPathList.get(i);
				userConfig.setProperty(SRC_DIR + i, srcDirPath);
			}
			userConfig.setProperty(OUT_DIR, outDirPath);
			userConfig.setProperty(CSS_FILE, cssFile);
			userConfig.setProperty(DEFAULT_WIKI, defaultWiki);
			userConfig.setProperty(WINDOW_MAX, toString(maximized));

			os = new FileOutputStream("./PureWikiConfig.properties");
			userConfig.store(os, "PureWiki Config");
		} catch (IOException ex) {
			ex.printStackTrace();
		} finally {
			PureWikiUtil.close(os);
		}
	}

	private boolean toBoolean(String value) {
		if (value == null) {
			return false;
		}
		return Boolean.parseBoolean(value);
	}

	private String toString(boolean value) {
		return Boolean.toString(value);
	}
}
