package jp.gr.java_conf.wutse.purewiki.swt;

import java.io.IOException;
import java.util.List;

import jp.gr.java_conf.wutse.purewiki.PureWikiConfig;
import jp.gr.java_conf.wutse.purewiki.PureWikiData;
import jp.gr.java_conf.wutse.purewiki.PureWikiManager;
import jp.gr.java_conf.wutse.purewiki.event.WikiSelectionEvent;
import jp.gr.java_conf.wutse.purewiki.event.WikiSelectionListener;
import jp.gr.java_conf.wutse.purewiki.exception.PureWikiValidateException;
import jp.gr.java_conf.wutse.purewiki.exception.ValidateError;
import jp.gr.java_conf.wutse.purewiki.util.PureWikiUtil;

import org.eclipse.swt.SWT;
import org.eclipse.swt.browser.Browser;
import org.eclipse.swt.browser.LocationAdapter;
import org.eclipse.swt.browser.LocationEvent;
import org.eclipse.swt.custom.StackLayout;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DragSource;
import org.eclipse.swt.dnd.DragSourceAdapter;
import org.eclipse.swt.dnd.DragSourceEvent;
import org.eclipse.swt.dnd.DropTarget;
import org.eclipse.swt.dnd.DropTargetAdapter;
import org.eclipse.swt.dnd.DropTargetEvent;
import org.eclipse.swt.dnd.FileTransfer;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.dnd.TransferData;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.MenuAdapter;
import org.eclipse.swt.events.MenuEvent;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.ShellAdapter;
import org.eclipse.swt.events.ShellEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.TabFolder;
import org.eclipse.swt.widgets.TabItem;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.wb.swt.SWTResourceManager;

public class PureWikiShell extends Shell implements WikiSelectionListener {
	public final GuiMode EDIT = new Edit();
	public final GuiMode VIEW = new View();

	private static final String ITEM_TYPE = "type";
	private static final Integer ITEM_TYPE_ROOT = new Integer(0);
	private static final Integer ITEM_TYPE_DIR = new Integer(1);
	private static final Integer ITEM_TYPE_WIKDATA = new Integer(2);
	private static final String ITEM_DATA = "data";

	private PureWikiManager manager = null;
	private GuiMode mode = VIEW;

	private Image dirImage;
	private Image fileImage;

	private Menu menu;
	private MenuItem fileMenuItem;
	private Menu fileMenu;
	private MenuItem storeMenuItem;
	private MenuItem reloadMenuItem;
	private MenuItem exitMenuItem;
	private MenuItem preferenceMenuItem;

	private Tree tree;
	private Menu treeMenu;
	private TabFolder tabFolder;
	private TabItem viewTabItem;
	private TabItem editTabItem;
	private Browser browser;
	private Composite titleComposite;
	private StackLayout titleStack;
	private Label titleLabel;
	private Composite newTitleComposite;
	private Text titleText;
	private Text wikiText;
	private Button newButton;

	/**
	 * Create the shell.
	 * 
	 * @param display
	 */
	public PureWikiShell(Display display) {
		super(display, SWT.SHELL_TRIM);

		dirImage = new Image(getDisplay(), getClass().getResourceAsStream("/icons/fldr_obj.gif"));
		fileImage = new Image(getDisplay(), getClass().getResourceAsStream("/icons/file_obj.gif"));

		initializeManager();
		initializeWindow();
		initializePureWiki();
	}

	private void initializeManager() {
		manager = new PureWikiManager();
		manager.addWikiSelectionListener(this);
	}

	/**
	 * Wikiリストを再取得する。
	 * 
	 * @param force
	 *            FileSystemから再取得する場合はtrue、キャッシュから再取得する場合は、false。
	 */
	public void reloadWikiList() {
		PureWikiData curData = manager.getCurrentData();
		List<PureWikiData> dataList = manager.getPureWikiDataList();
		tree.removeAll();

		TreeItem rootItem = new TreeItem(tree, SWT.NULL);
		rootItem.setData(ITEM_TYPE, ITEM_TYPE_DIR);
		rootItem.setText(0, "ROOT");
		rootItem.setImage(dirImage);

		for (PureWikiData data : dataList) {
			TreeItem item = new TreeItem(rootItem, SWT.NULL);
			item.setData(ITEM_TYPE, ITEM_TYPE_WIKDATA);
			item.setData(ITEM_DATA, data);
			item.setText(0, data.getWikiName());
			item.setImage(fileImage);
		}

		rootItem.setExpanded(true);

		PureWikiData nextData = null;
		if (curData.isNew()) {
			nextData = curData;
		}
		else {
			nextData = manager.getPureWikiData(curData.getWikiName());
			if (nextData == null) {
				nextData = manager.getPureWikiData(manager.getEngine().getConfig().getDefaultWiki());
			}
		}
		manager.changeCurrentData(nextData);
		selectWiki(nextData.getWikiName());
	}

	/**
	 * 画面モードを切り替える。 forceにfalseを指定した場合は、モードの変更が無い時に何も行わない。 forceにtrueを指定した場合は、
	 * 
	 * @param mode
	 *            GuiMode.EDIT(編集モード),GuiMode.VIEW(参照モード)
	 * @param force
	 */
	public void setMode(GuiMode mode, boolean force) {
		if (force || this.mode != mode) {
			if (this.mode != null) {
				boolean ret = this.mode.exit();
				if (!ret) {
					return;
				}
			}
			this.mode = mode;
			mode.entry();
		}
	}

	/**
	 * 現在の画面モードを取得する。
	 * 
	 * @return
	 */
	public GuiMode getMode() {
		return mode;
	}

	private void initializeWindow() {
		PureWikiConfig config = manager.getEngine().getConfig();

		// ///////////////////////////////////////////////////////
		// ウィンドウ構築
		// ///////////////////////////////////////////////////////
		setSize(900, 700);
		setLayout(new GridLayout(2, false));
		setText("- PureWiki");
		setMaximized(config.isMaximized());

		// ///////////////////////////////////////////////////////
		// メニュー
		// ///////////////////////////////////////////////////////
		menu = new Menu(this, SWT.BAR);
		setMenuBar(menu);

		fileMenuItem = new MenuItem(menu, SWT.CASCADE);
		fileMenuItem.setText("File(&f)");

		fileMenu = new Menu(fileMenuItem);
		fileMenuItem.setMenu(fileMenu);

		storeMenuItem = new MenuItem(fileMenu, SWT.NONE);
		storeMenuItem.setText("保存");
		storeMenuItem.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				storeWiki();
			}
		});

		reloadMenuItem = new MenuItem(fileMenu, SWT.NONE);
		reloadMenuItem.setText("ファイルの再読み込み");
		reloadMenuItem.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				manager.clearPureWikiDataList();
				reloadWikiList();
			}
		});

		exitMenuItem = new MenuItem(fileMenu, SWT.NONE);
		exitMenuItem.setText("終了");
		exitMenuItem.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				close();
			}
		});

		preferenceMenuItem = new MenuItem(menu, SWT.CASCADE);
		preferenceMenuItem.setText("設定");
		preferenceMenuItem.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				showPreference();
			}
		});

		// ///////////////////////////////////////////////////////
		// wikiリスト
		// ///////////////////////////////////////////////////////
		tree = new Tree(this, SWT.BORDER);
		tree.setFont(SWTResourceManager.getFont("メイリオ", 9, SWT.NORMAL));
		GridData gd_tree = new GridData(SWT.LEFT, SWT.FILL, false, true, 1, 1);
		gd_tree.widthHint = 155;
		tree.setLayoutData(gd_tree);
		tree.setHeaderVisible(false);

		// Wikiリストから選択時
		tree.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetDefaultSelected(SelectionEvent event) {
				try {
					TreeItem item = (TreeItem)event.item;
					if (ITEM_TYPE_WIKDATA == item.getData(ITEM_TYPE)) {
						PureWikiData wikiData = (PureWikiData)item.getData(ITEM_DATA);
						changeWikiData(wikiData.getWikiName());
						updateWindowTitle();
					}
				}
				catch (Exception ex) {
					ex.printStackTrace();
				}
			}
		});

//		TreeColumn trclmnTest = new TreeColumn(tree, SWT.NONE);
//		trclmnTest.setWidth(170);
//		trclmnTest.setText("Wiki Name");

		DropTarget dropTarget = new DropTarget(tree, DND.DROP_DEFAULT | DND.DROP_COPY);
		final FileTransfer transfer = FileTransfer.getInstance();
		Transfer[] types = new Transfer[] { transfer };
		dropTarget.setTransfer(types);
		dropTarget.addDropListener(new DropTargetAdapter() {
			@Override
			public void dragEnter(DropTargetEvent event) {
				event.detail = DND.DROP_COPY;
			}

			@Override
			public void drop(DropTargetEvent event) {
				for (TransferData trans : event.dataTypes) {

				}

				String[] files = (String[])event.data;
				for (String file : files) {
					System.out.println(file);
					// TODO
				}
			}
		});

		DragSource dragSource = new DragSource(tree, DND.DROP_COPY);
		dragSource.setTransfer(new Transfer[] { FileTransfer.getInstance() });
		dragSource.addDragListener(new DragSourceAdapter() {
			@Override
			public void dragStart(DragSourceEvent event) {
				TreeItem[] items = tree.getSelection();
				for (int i = 0; i < items.length; i++) {
					PureWikiData wikiData = (PureWikiData)items[i].getData(ITEM_DATA);
					if (wikiData == null) {
						event.doit = false;
						return;
					}
				}
			}

			@Override
			public void dragSetData(DragSourceEvent event) {
				TreeItem[] items = tree.getSelection();
				String[] data = new String[items.length];
				for (int i = 0; i < items.length; i++) {
					PureWikiData wikiData = (PureWikiData)items[i].getData(ITEM_DATA);
					if (wikiData != null) {
						data[i] = wikiData.getSrcFile().getAbsolutePath();
					}
					else {
						event.doit = false;
						return;
					}
				}
				event.data = data;
			}
		});

		treeMenu = new Menu(tree);
		tree.setMenu(treeMenu);
		treeMenu.addMenuListener(new MenuAdapter() {
			@Override
			public void menuShown(MenuEvent e) {
				createRightClickMenu();
				super.menuShown(e);
			}
		});

		// ///////////////////////////////////////////////////////
		// 表示エリア(タブ)
		// ///////////////////////////////////////////////////////
		tabFolder = new TabFolder(this, SWT.BOTTOM);
		tabFolder.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1));

		// ///////////////////////////////////////////////////////
		// 表示エリア(VIEW)
		// ///////////////////////////////////////////////////////
		viewTabItem = new TabItem(tabFolder, SWT.NONE);
		viewTabItem.setText("View");
		viewTabItem.setData("status", VIEW);

		browser = new Browser(tabFolder, SWT.NONE);

		browser.addKeyListener(new KeyAdapter() {
			@Override
			public void keyReleased(KeyEvent e) {
				// バックスペースによる「戻る」禁止
				if (e.keyCode == SWT.BS) {
					e.doit = false;
				}
			}

			@Override
			public void keyPressed(KeyEvent e) {
				// バックスペースによる「戻る」禁止
				if (e.keyCode == SWT.BS) {
					e.doit = false;
				}
			}
		});
		browser.addLocationListener(new LocationAdapter() {
			@Override
			public void changing(LocationEvent event) {
				if (!manager.getEngine().isTemporaryUrl(event.location)
						&& manager.getEngine().isLocalUrl(event.location)) {
					String dataName = manager.getEngine().getWikiNameFromUrl(event.location);
					if (!changeWikiData(dataName)) {
						event.doit = false;
					}
				}
				updateWindowTitle();
			}
		});

		viewTabItem.setControl(browser);

		// ///////////////////////////////////////////////////////
		// 表示エリア(EDIT)
		// ///////////////////////////////////////////////////////
		editTabItem = new TabItem(tabFolder, SWT.NONE);
		editTabItem.setText("Edit");
		editTabItem.setData("status", EDIT);

		Composite composite = new Composite(tabFolder, SWT.NONE);
		editTabItem.setControl(composite);
		composite.setLayout(new GridLayout(1, false));

		titleComposite = new Composite(composite, SWT.NONE);
		titleStack = new StackLayout();
		titleComposite.setLayout(titleStack);
		titleComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false, 1, 1));

		titleLabel = new Label(titleComposite, SWT.NONE);
		titleLabel.setFont(SWTResourceManager.getFont("メイリオ", 18, SWT.BOLD | SWT.ITALIC));

		newTitleComposite = new Composite(titleComposite, SWT.NONE);
		newTitleComposite.setLayout(new GridLayout(2, false));

		titleText = new Text(newTitleComposite, SWT.BORDER);
		titleText.setFont(SWTResourceManager.getFont("メイリオ", 14, SWT.NORMAL));
		GridData gd_titleText = new GridData(SWT.FILL, SWT.CENTER, true, false, 1, 1);
		gd_titleText.heightHint = 21;
		titleText.setLayoutData(gd_titleText);

		titleText.addModifyListener(new ModifyListener() {
			@Override
			public void modifyText(ModifyEvent e) {
				manager.getCurrentData().setWikiName(titleText.getText());
				updateWindowTitle();
			}
		});

		newButton = new Button(newTitleComposite, SWT.NONE);
		newButton.setText("作成");
		newButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				storeWiki();
			}
		});

		wikiText = new Text(composite, SWT.BORDER | SWT.V_SCROLL | SWT.MULTI);
		wikiText.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1));

		wikiText.addKeyListener(new KeyAdapter() {
			@Override
			public void keyPressed(KeyEvent e) {
				if (e.stateMask == SWT.CTRL && e.keyCode == 115) {
					storeWiki();
				}
				else if (e.stateMask == SWT.CTRL && e.keyCode == 97) {
					wikiText.selectAll();
				}
				super.keyPressed(e);
			}
		});

		wikiText.addModifyListener(new ModifyListener() {
			@Override
			public void modifyText(ModifyEvent e) {
				manager.getCurrentData().setWikiText(wikiText.getText());
				updateWindowTitle();
			}
		});

		addShellListener(new ShellAdapter() {
			@Override
			public void shellClosed(ShellEvent e) {
				if (!confirmStore(manager.getCurrentData())) {
					e.doit = false;
					return;
				}
				MessageBox msg = new MessageBox((Shell)e.getSource(), SWT.OK | SWT.CANCEL | SWT.ICON_WARNING);
				msg.setText("確認");
				msg.setMessage("アプリケーションを終了しますか？");
				int ret = msg.open();
				if (SWT.CANCEL == ret) {
					e.doit = false;
				}
				PureWikiConfig config = manager.getEngine().getConfig();
				config.setMaximized(getMaximized());
				manager.exit();
			}
		});

		tabFolder.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent event) {
				TabItem tabItem = (TabItem)event.item;
				GuiMode status = (GuiMode)tabItem.getData("status");
				setMode(status, true);
			}
		});

		createContents();
	}

	private void createRightClickMenu() {
		// 前回のメニューリストをクリア
		MenuItem items[] = treeMenu.getItems();
		for (MenuItem item : items) {
			item.dispose();
		}
		// 新規作成
		MenuItem newWikiMenu = new MenuItem(treeMenu, SWT.NONE);
		newWikiMenu.setText("新規作成");
		newWikiMenu.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				manager.createTemporaryData();
				setMode(EDIT, true);
			}
		});

		// コピー
		MenuItem copyWikiMenu = new MenuItem(treeMenu, SWT.NONE);
		copyWikiMenu.setText("コピーして新規作成");
		copyWikiMenu.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				try {
					TreeItem[] selectedItems = tree.getSelection();
					if (selectedItems != null && selectedItems.length > 0) {
						PureWikiData wikiData = (PureWikiData)selectedItems[0].getData(ITEM_DATA);
						manager.createTemporaryData();
						PureWikiData newData = manager.getCurrentData();
						if (!wikiData.isLoaded()) {
							manager.getEngine().loadPureWikiData(wikiData, false);
						}
						newData.setWikiName(wikiData.getWikiName() + "_copy");
						newData.setWikiText(wikiData.getOrgWikiText());
					}
				}
				catch (Exception ex) {
					ex.printStackTrace();
				}

				setMode(EDIT, true);
			}
		});

		// 削除
		MenuItem deleteWikiMenu = new MenuItem(treeMenu, SWT.NONE);
		deleteWikiMenu.setText("削除");
		deleteWikiMenu.setEnabled(true);
		deleteWikiMenu.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				try {
					TreeItem[] selectedItems = tree.getSelection();
					if (selectedItems != null && selectedItems.length > 0) {
						PureWikiData wikiData = (PureWikiData)selectedItems[0].getData(ITEM_DATA);
						if (confirmDelete(wikiData)) {
							setMode(getMode(), true);
						}
					}
				}
				catch (Exception ex) {
					ex.printStackTrace();
				}
			}
		});

		// 最新の情報を表示
		MenuItem reloadListMenu = new MenuItem(treeMenu, SWT.NONE);
		reloadListMenu.setText("最新の情報を表示");

		reloadListMenu.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				manager.clearPureWikiDataList();
				reloadWikiList();
			}
		});

	}

	private void initializePureWiki() {
		reloadWikiList();

		setMode(VIEW, false);
	}

	private void showPreference() {
		if (confirmStore(manager.getCurrentData())) {
			PureWikiPreference prefDialog = new PureWikiPreference(this, manager.getEngine().getConfig());
			int ret = prefDialog.open();
			if (SWT.OK == ret) {
				initializeManager();
				manager.clearPureWikiDataList();
				reloadWikiList();
				// manager.getEngine().translateAll(); // 必要？
				setMode(VIEW, true);
			}
		}
	}

	private void updateWindowTitle() {
		PureWikiData data = manager.getCurrentData();
		if (data.isNew()) {
			setText("New Page [編集中] - PureWiki");
		}
		else {
			StringBuffer sb = new StringBuffer();
			sb.append(data.getPath());
			sb.append(" ");
			if (data.isModified()) {
				sb.append("[編集中] ");
			}
			sb.append("- PureWiki");
			setText(sb.toString());
		}
	}

	private void storeWiki() {
		try {
			PureWikiData curWikiData = manager.getCurrentData();
			curWikiData.setWikiText(wikiText.getText());
			manager.storeWiki();
			reloadWikiList();
			setMode(EDIT, true);
			updateWindowTitle();
		}
		catch (PureWikiValidateException e) {
			showValidateMessage(e);
		}
	}

	private void showValidateMessage(PureWikiValidateException exception) {
		ValidateError[] errors = exception.getValidateErrors();
		StringBuilder sb = new StringBuilder();
		for (ValidateError error : errors) {
			sb.append(error.getErrorMessage());
			sb.append("\n");
		}
		MessageBox msg = new MessageBox(this, SWT.OK | SWT.ICON_ERROR);
		msg.setText("入力エラー");
		msg.setMessage(sb.toString());
		msg.open();

	}

	private boolean confirmStore(PureWikiData fromData) {
		if (fromData != null && fromData.isModified()) {
			MessageBox msg = new MessageBox(this, SWT.YES | SWT.NO | SWT.CANCEL | SWT.ICON_WARNING);
			msg.setText("確認");
			msg.setMessage("内容が変更されています。保存しますか？");
			int ret = msg.open();
			if (SWT.YES == ret) {
				fromData.setWikiText(wikiText.getText());
				try {
					manager.storeWiki(fromData);
				}
				catch (PureWikiValidateException e) {
					showValidateMessage(e);
					return false;
				}
			}
			else if (SWT.NO == ret) {
				if (fromData.isNew()) {
					fromData.getOutFile().delete();
				}
				else {
					manager.loadWikiText();
				}
				wikiText.setText(fromData.getWikiText());
			}
			if (SWT.YES == ret || SWT.NO == ret) {
				return true;
			}
			else {
				return false;
			}
		}
		else {
			return true;
		}
	}

	private boolean confirmDelete(PureWikiData wikiData) {
		MessageBox msg = new MessageBox(this, SWT.YES | SWT.NO | SWT.CANCEL | SWT.ICON_WARNING);
		msg.setText("確認");
		msg.setMessage("「" + wikiData.getWikiName() + "」を削除します。よろしいですか？");
		int ret = msg.open();
		if (SWT.YES == ret) {
			manager.removeWiki(wikiData);
			reloadWikiList();
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * Create contents of the shell.
	 */
	protected void createContents() {}

	@Override
	protected void checkSubclass() {
		// Disable the check that prevents subclassing of SWT components
	}

	@Override
	public void wikiChanging(WikiSelectionEvent e) {
		PureWikiData curData = e.getCurrentWikiData();
		PureWikiData nextData = e.getNextWikiData();
		if (curData != nextData && !confirmStore(curData)) {
			e.eventCancel();
		}
	}

	@Override
	public void wikiChanged(WikiSelectionEvent e) {
		PureWikiData curData = e.getCurrentWikiData();
		PureWikiData nextData = e.getNextWikiData();
		if (nextData == null) {
			return;
		}
		mode.changeWiki(curData, nextData);
		if (manager.getCurrentData().isNew()) {
			setMode(EDIT, true);
		}
		else {
			setMode(getMode(), true);
		}
	}

	private boolean changeWikiData(String wikiName) {
		PureWikiData curWikiData = manager.getCurrentData();
		PureWikiData nextWikiData = manager.getPureWikiData(wikiName);
		if (curWikiData == nextWikiData) {
			return true;
		}
		return manager.changeCurrentData(wikiName);
	}

	private void selectWiki(String wikiName) {
		if (PureWikiUtil.isBlank(wikiName)) {
			tree.deselectAll();
			return;
		}
		TreeItem[] rootItems = tree.getItems();
		TreeItem rootItem = rootItems[0];

		TreeItem[] wikiItems = rootItem.getItems();
		TreeItem selectItem = null;
		for (TreeItem wikiItem : wikiItems) {
			PureWikiData itemData = (PureWikiData)wikiItem.getData(ITEM_DATA);
			if (wikiName.equals(itemData.getWikiName())) {
				selectItem = wikiItem;
				break;
			}
		}
		if (selectItem != null) {
			tree.setSelection(selectItem);
		}
		else {
			tree.deselectAll();
		}
	}

	interface GuiMode {
		void entry();

		boolean exit();

		boolean changeWiki(PureWikiData fromData, PureWikiData toData);
	}

	class View implements GuiMode {
		public View() {}

		@Override
		public void entry() {
			// Viewタブが選択されていなかったら選択させる
			TabItem[] selectionTabs = tabFolder.getSelection();
			if (selectionTabs.length == 1 && viewTabItem != selectionTabs[0]) {
				tabFolder.setSelection(viewTabItem);
			}
			try {
				manager.translateWiki();
			}
			catch (IOException ex) {
				ex.printStackTrace();
			}
//			if (!manager.getCurrentData().getUrl().equals(browser.getUrl())) {
			browser.setUrl(manager.getCurrentData().getUrl());
//			}
		}

		@Override
		public boolean exit() {
			return true;
		}

		@Override
		public boolean changeWiki(PureWikiData fromData, PureWikiData toData) {
			selectWiki(toData.getWikiName());
			return true;
		}
	}

	class Edit implements GuiMode {
		public Edit() {}

		@Override
		public void entry() {
			// Editタブが選択されていなかったら選択させる
			TabItem[] selectionTabs = tabFolder.getSelection();
			if (selectionTabs.length == 1 && editTabItem != selectionTabs[0]) {
				tabFolder.setSelection(editTabItem);
			}

			layout();
			if (manager.getCurrentData().isNew()) {
				titleText.setFocus();
			}
			else {
				wikiText.setFocus();
			}
		}

		@Override
		public boolean exit() {
			return true;
		}

		@Override
		public boolean changeWiki(PureWikiData fromData, PureWikiData toData) {
			layout();
			titleText.setFocus();
			return true;
		}

		private void layout() {
			if (manager.getCurrentData().isNew()) {
				String wikiName = manager.getCurrentData().getWikiName();
				if (wikiName == null) {
					wikiName = "";
				}
				titleText.setText(wikiName);
				titleStack.topControl = newTitleComposite;
				titleComposite.layout();
				String wikiTextStr = manager.getCurrentData().getWikiText();
				if (wikiTextStr == null) {
					wikiTextStr = "";
				}
				wikiText.setText(wikiTextStr);
			}
			else {
				titleLabel.setText(manager.getCurrentData().getWikiName());
				titleStack.topControl = titleLabel;
				titleComposite.layout();
				wikiText.setText(manager.getCurrentData().getWikiText());
			}
		}
	}

}
