package jp.gr.java_conf.wutse.purewiki;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.StringReader;
import java.io.Writer;
import java.util.List;

import jp.gr.java_conf.wutse.purewiki.syntax.BlockSyntax;
import jp.gr.java_conf.wutse.purewiki.syntax.LineSyntax;
import jp.gr.java_conf.wutse.purewiki.syntax.WordSyntax;
import jp.gr.java_conf.wutse.purewiki.syntax.line.NoneSyntax;
import jp.gr.java_conf.wutse.purewiki.util.PureWikiUtil;

public class PureWikiTranslator {
	private static final String LINE_SEPARATOR = System.lineSeparator();
	private PureWikiEngine engine = null;

	public PureWikiTranslator(PureWikiEngine engine) {
		this.engine = engine;
	}

	public void translateWiki(PureWikiContext context) throws IOException {
		List<BlockSyntax> blockSyntaxList = engine.getConfig().getBlockSyntaxList();
		List<LineSyntax> lineSyntaxList = engine.getConfig().getLineSyntaxList();
		List<WordSyntax> wordSyntaxList = engine.getConfig().getWordSyntaxList();
		BufferedReader reader = null;
		Writer writer = null;
		try {
			String wikiText = context.getWikiData().getWikiText();
			reader = new BufferedReader(new StringReader(wikiText));
			File outFile = context.getWikiData().getOutFile();
			File outParent = outFile.getParentFile();
			if (!outParent.exists()) {
				outParent.mkdirs();
			}
			writer = new OutputStreamWriter(new FileOutputStream(outFile), "UTF-8");

			String line = null;
			writeHtmlHeader(context, writer);
			writeBodyHeader(context, writer);
			context.setCurrentLineSyntax(new NoneSyntax());
			do {
				line = reader.readLine();
				context.setOriginalLine(line);
				context.setLine(line);

				if (line != null) {
					// ブロック構文の開始
					if (line.startsWith(">>> ")) {
						BlockSyntax targetSyntax = context.getCurrentBlockSyntax();
						// 既にブロック中の場合は無視
						if (targetSyntax != null) {
							continue;
						}
						String[] blockLine = line.split(" ");
						if (blockLine.length < 2) {
							continue;
						}
						String blockName = blockLine[1];
						for (BlockSyntax blockSyntax : blockSyntaxList) {
							String name = blockSyntax.getName();
							if (name.equals(blockName)) {
								targetSyntax = blockSyntax;
								break;
							}
						}
						if (targetSyntax != null) {
							line = targetSyntax.begin(context);
							context.setCurrentBlockSyntax(targetSyntax);
						}
						else {
							continue;
						}
					}
					else if (line.startsWith("<<<")) {
						BlockSyntax targetSyntax = context.getCurrentBlockSyntax();
						if (targetSyntax != null) {
							line = targetSyntax.end(context);
							context.setCurrentBlockSyntax(null);
						}
						else {
							continue;
						}
					}
					else {
						if (context.isEnableHtmlEncode()) {
							line = encode(line);
						}
						BlockSyntax targetSyntax = context.getCurrentBlockSyntax();
						if (targetSyntax != null) {
							context.setLine(line);
							line = targetSyntax.line(context);
						}
						if (context.isEnableLineSyntax()) {
							LineSyntax lastLineSyntax = context.getCurrentLineSyntax();
							LineSyntax targetLineSyntax = null;
							for (LineSyntax lineSyntax : lineSyntaxList) {
								if (lineSyntax.condition(context)) {
									targetLineSyntax = lineSyntax;
									break;
								}
							}
							context.setLine(line);

							StringBuffer sb = new StringBuffer();
							if (targetLineSyntax == null) {
								if (lastLineSyntax != null) {
									sb.append(lastLineSyntax.line(context));
								}
								else {
									sb.append(line);
								}
							}
							else {
								if (targetLineSyntax != lastLineSyntax) {
									if (lastLineSyntax != null) {
										sb.append(lastLineSyntax.end(context));
									}
									sb.append(targetLineSyntax.begin(context));
									sb.append(targetLineSyntax.line(context));
								}
								else {
									sb.append(targetLineSyntax.line(context));
								}
								context.setCurrentLineSyntax(targetLineSyntax);
							}
							line = sb.toString();
							context.setLine(line);
						}
						if (context.isEnableWordSyntax()) {
							for (WordSyntax wordSyntax : wordSyntaxList) {
								line = wordSyntax.line(context);
								context.setLine(line);
							}
						}
					}
				}
				else {
					context.setStatus(PureWikiContext.END);
					// 行構文
					LineSyntax lastLineSyntax = context.getCurrentLineSyntax();
					lastLineSyntax.end(context);
					// ブロック構文が終わっていない場合はクローズする。
					BlockSyntax targetBlockSyntax = context.getCurrentBlockSyntax();
					if (targetBlockSyntax != null) {
						targetBlockSyntax.end(context);
						context.setCurrentBlockSyntax(null);
					}
				}
				if (!PureWikiUtil.isBlank(line)) {
					writer.write(line);
					writer.write(LINE_SEPARATOR);
				}
			}
			while (!context.isEnd());
			writeBodyFooter(context, writer);
			writeHtmlFooter(context, writer);
		}
		finally {
			if (reader != null) {
				try {
					reader.close();
				}
				catch (IOException ex) {}
			}
			if (writer != null) {
				try {
					writer.close();
				}
				catch (IOException ex) {}
			}
		}
	}

	private void writeHtmlHeader(PureWikiContext context, Writer writer) throws IOException {
		String cssFile = engine.getConfig().getCssFile();
		writer.write("<!DOCTYPE html>");
		writer.write(LINE_SEPARATOR);
		writer.write("<html>");
		writer.write(LINE_SEPARATOR);
		writer.write("<head>");
		writer.write(LINE_SEPARATOR);
		writer.write("<meta charset=\"UTF-8\">");
		writer.write(LINE_SEPARATOR);
		if (!PureWikiUtil.isBlank(context.getWikiData().getWikiName())) {
			writer.write("<title>");
			writer.write(context.getWikiData().getWikiName());
			writer.write("</title>");
			writer.write(LINE_SEPARATOR);
		}
		if (!PureWikiUtil.isBlank(cssFile)) {
			writer.write("<link rel=\"stylesheet\" href=\"");
			writer.write(cssFile);
			writer.write("\" type=\"text/css\"/>");
		}
		else {
			writer.write("<style type=\"text/css\">");
			writer.write(LINE_SEPARATOR);
			writer.write("<!--");
			writer.write(LINE_SEPARATOR);
			writer.write(engine.getConfig().getStyle());
			writer.write(LINE_SEPARATOR);
			writer.write("-->");
			writer.write(LINE_SEPARATOR);
			writer.write("</style>");
			writer.write(LINE_SEPARATOR);
		}
		writer.write("</head>");
		writer.write(LINE_SEPARATOR);
		writer.write("<body>");
		writer.write(LINE_SEPARATOR);
	}

	private void writeHtmlFooter(PureWikiContext context, Writer writer) throws IOException {
		writer.write("</body>");
		writer.write(LINE_SEPARATOR);
		writer.write("</html>");
		writer.write(LINE_SEPARATOR);
	}

	private void writeBodyHeader(PureWikiContext context, Writer writer) throws IOException {
		writer.write("<H1>");
		writer.write(context.getWikiData().getWikiName());
		writer.write("</H1>\n");
	}

	private void writeBodyFooter(PureWikiContext context, Writer writer) throws IOException {

	}

	private String encode(String text) {
		// HTMLエンコード
		text = text.replaceAll("&", "&amp;");
		text = text.replaceAll("\"", "&quot;");
		text = text.replaceAll("<", "&lt;");
		text = text.replaceAll(">", "&gt;");
		return text;
	}
}
