﻿//
//  UdpServer.cs
//
//  Author:
//       tsntsumi <tsntsumi at tsntsumi.com>
//
//  Copyright (c) 2015 tsntsumi
//
//  This program is free software: you can redistribute it and/or modify
//  it under the terms of the GNU Lesser General Public License as published by
//  the Free Software Foundation, either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.

/// @file
/// <summary>
/// UDP サーバを実装します。
/// </summary>
/// @since 2015.8.15
using System;
using System.Collections.Generic;
using System.Threading.Tasks;
using System.Net;
using System.Net.Sockets;

namespace SocketNet
{
	/// <summary>
	/// UDP サーバ。
	/// </summary>
	public abstract class UdpServer: IDisposable
	{
		protected UdpClient udpClient;

		protected volatile bool acceptingConnections;

		private bool disposed = false;

		/// <summary>
		/// データを受信したときに発行されます。
		/// </summary>
		public event EventHandler<UdpDataReceivedEventArgs> DataReceived;

		/// <summary>
		/// Udp サーバがバインドされる IP アドレスを取得します。
		/// </summary>
		public IPAddress IPAddress { get; protected set; }

		/// <summary>
		/// Udp サーバがバインドされるマルチキャストアドレスを取得します。
		/// </summary>
		public IPAddress MulticastAddress { get; protected set; }

		/// <summary>
		/// Udp サーバがバインドされるポート番号を取得します。
		/// </summary>
		public int Port { get; protected set; }

		/// <summary>
		/// サーバの状態を取得します。
		/// </summary>
		public bool IsRunning
		{
			get
			{
				return acceptingConnections;
			}
		}

		/// <summary>
		/// 関連付けられた転送タイプを取得します。
		/// </summary>
		public TransmissionType TransmissionType { get; protected set; }

		/// <summary>
		/// ブロードキャストUDPサーバを作成します。
		/// </summary>
		/// <returns>ブロードキャストサーバ。</returns>
		/// <param name="port">ポート番号。</param>
		/// <param name="local">ローカルブロードキャストサーバを作成するなら <c>true</c>。</param>
		public static UdpServer CreateServer(int port, bool local = false)
		{
			if (local)
			{
				return new LocalBroadcastUdpServer(port);
			}
			return new BroadcastUdpServer(port);
		}

		/// <summary>
		/// ユニキャストUDPサーバを作成します。
		/// </summary>
		/// <returns>ユニキャストUDPサーバ。</returns>
		/// <param name="unicastAddress">ユニキャストアドレス。</param>
		/// <param name="port">ポート番号。</param>
		public static UdpServer CreateServer(IPAddress unicastAddress, int port)
		{
			return new UnicastUdpServer(unicastAddress, port);
		}

		/// <summary>
		/// マルチキャストUDPサーバを作成します。
		/// </summary>
		/// <returns>マルチキャストUDPサーバ。</returns>
		/// <param name="port">ポート番号。</param>
		/// <param name="multicastAddress">マルチキャストアドレス。</param>
		public static UdpServer CreateServer(int port, IPAddress multicastAddress)
		{
			return new MulticastUdpServer(multicastAddress, port);
		}

		/// <summary>
		/// コンストラクタ。
		/// </summary>
		/// <param name="ipAddress">バインドするIPアドレス。</param>
		/// <param name="port">バインドするポート番号。</param>
		/// <param name="transmissionType">転送タイプ。</param>
		protected UdpServer(IPAddress ipAddress, int port, TransmissionType transmissionType)
		{
			IPAddress = ipAddress;
			Port = port;
			TransmissionType = transmissionType;
		}

		/// <summary>
		/// オブジェクトに関連付けられたリソースを解放します。
		/// </summary>
		public void Dispose()
		{
			Dispose(true);
			GC.SuppressFinalize(this);
		}

		/// <summary>
		/// オブジェクトに関連付けられたリソースを解放します。
		/// </summary>
		/// <param name="disposing">
		/// メソッドの呼び出し元が Dispose() メソッドか (値は <c>true</c>)、
		/// それともファイナライザーか (値は <c>false</c>) を示します。
		/// </param>
		private void Dispose(bool disposing)
		{
			if (disposed)
			{
				return;
			}

			if (disposing)
			{
				Stop();
				udpClient = null;
			}

			disposed = true;
		}

		/// <summary>
		/// Udpサーバを開始してデータ受信を開始します。
		/// </summary>
		public async Task Start()
		{
			IPEndPoint ipEndPoint = new IPEndPoint(IPAddress, Port);

			Setup(ipEndPoint);

			try
			{
				acceptingConnections = true;

				while (acceptingConnections)
				{
					var result = await udpClient.ReceiveAsync();
					var data = result.Buffer;

					if (data != null && data.Length > 0)
					{
						OnDataReceived(new UdpDataReceivedEventArgs(result.RemoteEndPoint, data));
					}
				}
			}
			catch (ObjectDisposedException)
			{
				// Suppress error
				acceptingConnections = false;
			}
		}

		/// <summary>
		/// UdpClient を作成します。
		/// </summary>
		/// <param name="ipEndPoint">IP エンドポイント。</param>
		protected virtual void Setup(IPEndPoint ipEndPoint)
		{
			udpClient = new UdpClient();
			udpClient.Client.SetSocketOption(SocketOptionLevel.Socket, SocketOptionName.ReuseAddress, true);
			udpClient.Client.Bind(ipEndPoint);
		}

		/// <summary>
		/// Udpサーバを停止します。
		/// </summary>
		public virtual void Stop()
		{
			acceptingConnections = false;

			if (udpClient != null)
			{
				udpClient.Close();
			}
		}

		/// <summary>
		/// DetaReceived イベントを発行します。
		/// </summary>
		/// <param name="e">イベントデータを格納するオブジェクト。</param>
		protected void OnDataReceived(UdpDataReceivedEventArgs e)
		{
			if (DataReceived != null)
			{
				DataReceived(this, e);
			}
		}
	}
}
