﻿// TcpConnection.cs
//
// Author:
// tsntsumi <tsntsumi at tsntsumi.com>
//
// Copyright (c) 2015 tsntsumi
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public License
//	along with this program.  If not, see <http://www.gnu.org/licenses/>.

/// @file
/// <summary>
/// TCP/IP の接続を表現します。
/// </summary>
/// @since 2015.8.12
using System;
using System.Collections.Generic;
using System.Threading.Tasks;
using System.Net;
using System.Net.Sockets;
using System.IO;

namespace SocketNet
{
	/// <summary>
	/// 単一の TCP/IP 接続を表現します。
	/// </summary>
	/// <remarks>
	/// <para>
	/// この接続は、TCP/IPでパケットを非同期的に受信します。
	/// 接続が切断されると <see cref="Disconnected"/> イベントが発行されます。
	/// パケットを完全に受信すると <see cref="DataReceived"/> イベントが発行されます。
	/// </para>
	/// <para>
	/// パケットは、ヘッダとペイロードとフッタの3つのパートから構成されていることを前提としています。
	/// ただし、フッタ長は0バイトでも構いません。
	///
	/// <table>
	/// <tr><td>ヘッダ</td><td>ペイロード</td><td>フッタ</td></tr>
	/// </table>
	///
	/// ヘッダにはペイロード長が格納されているものとして、
	/// <see cref="Packet.ObtainPayloadLength"/> メソッドを使って取り出します。
	/// デフォルトでは、ヘッダ4バイト、フッタ0バイトで、
	/// ヘッダの4バイトがネットワークバイトオーダーのペイロード長としてパケットを受信します。
	///
	/// <table>
	/// <tr><td colspan="4">ヘッダ</td></tr>
	/// <tr><td>pl1</td><td>pl2</td><td>pl3</td><td>pl4</td></tr>
	/// </table>
	///
	/// </para>
	/// </remarks>
	public sealed class TcpConnection: IDisposable
	{
		private static readonly int ReceivedDataBufferSize = 65535;

		private readonly PacketSpec PacketSpec;

		private object syncObject;
		private NetworkStream networkStream;
		private volatile bool isClosed;

		private DataReceiveState receiveState;
		private AsyncCallback dataReceivedCallback;
		private byte[] partiallyReceivedData;
		private List<byte> completelyReceivedData;
		private byte[] header;
		private byte[] packet;
		private int packetPayloadLength;
		private int bytesToProcessRemaining;

		private bool disposed = false;

		/// <summary>
		/// データを受信する間に接続が取りうる状態です。
		/// </summary>
		private enum DataReceiveState
		{
			/// <summary>
			/// ヘッダの受信中。
			/// </summary>
			PacketHeader = 0,

			/// <summary>
			/// ペイロードの受信中。
			/// </summary>
			PacketPayload,

			/// <summary>
			/// パケットのフッタ。
			/// </summary>
			PacketFooter
		}

		/// <summary>
		/// 接続が切断された時に発行されます。
		/// </summary>
		public event EventHandler<TcpConnectionEventArgs> Disconnected;

		/// <summary>
		/// データが受信された時に発行されます。
		/// </summary>
		public event EventHandler<TcpDataReceivedEventArgs> DataReceived;

		/// <summary>
		/// ソケットを取得します。
		/// </summary>
		/// <remarks>
		/// このプロパティは、基本的に手動でデータを操作するのに使用します。
		/// </remarks>
		public Socket Client { get; private set; }

		/// <summary>
		/// リーダを取得します。
		/// </summary>
		public BinaryReader Reader { get; private set; }

		/// <summary>
		/// ライタを取得します。
		/// </summary>        
		public BinaryWriter Writer { get; private set; }

		/// <summary>
		/// コンストラクタ。
		/// </summary>
		/// <param name="client">ソケット。</param>
		/// <param name="packetSpec">受信するパケットの各部の長さの設定を格納するオブジェクト。</param>
		public TcpConnection(Socket client,	PacketSpec packetSpec)
		{
			syncObject = new object();
			Client = client;
			networkStream = new NetworkStream(client);
			Reader = new BinaryReader(networkStream);
			Writer = new BinaryWriter(networkStream);

			PacketSpec = packetSpec;

			partiallyReceivedData = new byte[ReceivedDataBufferSize];
			completelyReceivedData = new List<byte>();
			header = new byte[PacketSpec.HeaderLength];
			packet = null;
			packetPayloadLength = 0;
			bytesToProcessRemaining = 0;
		}

		/// <summary>
		/// 関連付けられたリソースを解放します。
		/// </summary>
		public void Dispose()
		{
			Dispose(true);
			GC.SuppressFinalize(this);
		}

		/// <summary>
		/// オブジェクトに関連付けられたリソースを解放します。
		/// </summary>
		/// <param name="disposing">
		/// メソッドの呼び出し元が Dispose() メソッドか (値は <c>true</c>)、
		/// それともファイナライザーか (値は <c>false</c>) を示します。
		/// </param>
		private void Dispose(bool disposing)
		{
			if (disposed)
			{
				return;
			}
			if (disposing)
			{
				if (!isClosed)
				{
					Close();
				}
			}
			disposed = true;
		}

		/// <summary>
		/// 現在の接続と関連付けられたヘルパーオブジェクトをクローズします。
		/// </summary>
		public void Close()
		{
			lock (syncObject)
			{
				Reader.Close();
				Writer.Close();
				networkStream.Close();
				if (Client.Connected)
				{
#if __MonoCS__
					try
					{
						Client.Shutdown(SocketShutdown.Both);
						Client.Close();
					}
					catch (SocketException)
					{
						// Ignore socket exception
						// Mono C# 環境では、なぜか Client.Connected が true にも関わらず
						// Client.Shutdown() が Connect されていないという例外を発生させる。
					}
#else
					Client.Shutdown(SocketShutdown.Both);
					Client.Close();
#endif
				}

				dataReceivedCallback = null;
				isClosed = true;
				OnDisconnected(new TcpConnectionEventArgs(this));
			}
		}

		/// <summary>
		/// パケットを受信するための非同期システムを開始します。
		/// </summary>
		public void ReceiveDataAsync()
		{
			dataReceivedCallback = new AsyncCallback(DataReceivedCallback);
			BeginReceive();
		}

		/// <summary>
		/// バッファをクリアして、Socket.BeginReceive() を実行します。
		/// </summary>
		private void BeginReceive()
		{
			Array.Clear(partiallyReceivedData, 0, partiallyReceivedData.Length);
			Client.BeginReceive(partiallyReceivedData, 0, partiallyReceivedData.Length, SocketFlags.None, dataReceivedCallback, null);
		}

		/// <summary>
		/// Socket.BeginRceive() のコールバック。
		/// </summary>
		/// <param name="asyncResult"></param>
		private void DataReceivedCallback(IAsyncResult asyncResult)
		{
			if (isClosed)
			{
				return;
			}

			try
			{
				int bytesReceived = Client.EndReceive(asyncResult);
				if (bytesReceived > 0)
				{
					for (int i = 0; i < bytesReceived; i++)
					{
						completelyReceivedData.Add(partiallyReceivedData[i]);
					}
					bytesToProcessRemaining = completelyReceivedData.Count;

					while (bytesToProcessRemaining > 0)
					{
						switch (receiveState)
						{
						case DataReceiveState.PacketHeader:
							if (completelyReceivedData.Count >= PacketSpec.HeaderLength)
							{
								completelyReceivedData.CopyTo(0, header, 0, PacketSpec.HeaderLength);
								packetPayloadLength = PacketSpec.ObtainPayloadLength(header);

								receiveState = DataReceiveState.PacketPayload;
								completelyReceivedData.RemoveRange(0, PacketSpec.HeaderLength);
								bytesToProcessRemaining -= PacketSpec.HeaderLength;
							}
							else
							{
								bytesToProcessRemaining = 0;
							}
							break;

						case DataReceiveState.PacketPayload:
							if (completelyReceivedData.Count >= packetPayloadLength)
							{
								packet = new byte[PacketSpec.HeaderLength + packetPayloadLength + PacketSpec.FooterLength];
								Array.Copy(header, packet, PacketSpec.HeaderLength);
								completelyReceivedData.CopyTo(0, packet, PacketSpec.HeaderLength, packetPayloadLength);

								receiveState = DataReceiveState.PacketFooter;
								completelyReceivedData.RemoveRange(0, packetPayloadLength);
								bytesToProcessRemaining -= packetPayloadLength;

								if (PacketSpec.FooterLength == 0)
								{
									OnDataReceived(new TcpDataReceivedEventArgs(this, packet));
									receiveState = DataReceiveState.PacketHeader;
								}
							}
							else
							{
								bytesToProcessRemaining = 0;
							}
							break;

						case DataReceiveState.PacketFooter:
							if (completelyReceivedData.Count >= PacketSpec.FooterLength)
							{
								completelyReceivedData.CopyTo(0, packet, PacketSpec.HeaderLength + packetPayloadLength, PacketSpec.FooterLength);

								OnDataReceived(new TcpDataReceivedEventArgs(this, packet));

								receiveState = DataReceiveState.PacketHeader;
								completelyReceivedData.RemoveRange(0, PacketSpec.FooterLength);
								bytesToProcessRemaining -= PacketSpec.FooterLength;
							}
							else
							{
								bytesToProcessRemaining = 0;
							}
							break;
						}
					}

					BeginReceive();
				}
				else
				{
					// ソケットがシャットダウンすると bytesReceived が 0 になるため、接続を削除します。
					completelyReceivedData.Clear();
					Close();
					return;
				}
			}
			catch
			{
				Close();
			}
		}

		/// <summary>
		/// DataReceived イベントを発行します。
		/// </summary>
		/// <param name="e">イベントのデータを格納する <see cref="TcpDataReceivedEventArgs"/> オブジェクト。</param>
		private void OnDataReceived(TcpDataReceivedEventArgs e)
		{
			if (DataReceived != null)
			{
				DataReceived(this, e);
			}
		}

		/// <summary>
		/// 切断イベントを発行します。
		/// </summary>
		/// <param name="e">イベントのデータを格納する <see cref="TcpConnectionEventArgs"/> オブジェクト。</param>
		private void OnDisconnected(TcpConnectionEventArgs e)
		{
			if (Disconnected != null)
			{
				Disconnected(this, e);
			}
		}
	}
}
