﻿// ChatMessage.cs
//
// Author:
// tsntsumi <tsntsumi at tsntsumi.com>
//
// Copyright (c) 2015 tsntsumi
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public License
//	along with this program.  If not, see <http://www.gnu.org/licenses/>.

/// @file
/// <summary>
/// チャットメッセージパケット。
/// </summary>
/// @since 2015.8.14
using System;
using System.Text;
using System.Net;
using SocketNet;

namespace SampleChatClient
{
	/// <summary>
	/// <see cref="SampleChatServer.ChatServer"/>  と <see cref="ChatClient"/>  間で送受信されるチャットメッセージを表します。
	/// </summary>
	public class ChatMessage: PacketSpec
	{
		/// <summary>
		/// メッセージを送受信する接続を取得または設定します。
		/// </summary>
		public TcpConnection Connection { get; set; }

		/// <summary>
		/// メッセージ文字列を取得します。
		/// </summary>
		public string Message { get; private set; }

		/// <summary>
		/// コンストラクタ。
		/// </summary>
		public ChatMessage() : base(2, 0)
		{
		}

		/// <summary>
		/// コンストラクタ。
		/// </summary>
		/// <param name="message">メッセージ文字列。</param>
		/// <param name="connection">接続。</param>
		public ChatMessage(string message, TcpConnection connection = null)
			: base(2, 0)
		{
			Message = message;
			Connection = connection;
		}

		/// <summary>
		/// 指定されたヘッダからペイロード長を取り出します。
		/// </summary>
		/// <returns>ペイロード長。</returns>
		/// <param name="header">ヘッダ。</param>
		/// <remarks>
		/// ベースクラスの Packet.ObtainPayloadLength() と同じことをしているため、
		/// 本来ならオーバライドする必要はないが、サンプルとして提供するため実装した。
		/// </remarks>
		public override int ObtainPayloadLength(byte[] header)
		{
			int payloadLength = header[0] << 8 | header[1];

			return payloadLength;
		}

		/// <summary>
		/// 指定されたバイト配列からチャットメッセージオブジェクトを生成します。
		/// </summary>
		/// <returns>チャットメッセージオブジェクト。</returns>
		/// <param name="data">バイト配列データ。</param>
		public static ChatMessage FromByteArray(byte[] data)
		{
			ChatMessage chatMessage = new ChatMessage();
			Encoding encoding = new UTF8Encoding(false);
			// int length = (data[0] << 8) | data[1];
			byte[] messageData = new byte[data.Length - chatMessage.HeaderLength];

			Array.Copy(data, chatMessage.HeaderLength, messageData, 0, messageData.Length);
			chatMessage.Message = encoding.GetString(messageData);

			return chatMessage;
		}

		/// <summary>
		/// チャットメッセージオブジェクトを、バイト配列に変換します。
		/// </summary>
		/// <returns>バイト配列。</returns>
		public byte[] ToByteArray()
		{
			Encoding encoding = new UTF8Encoding(false);
			byte[] messageData = encoding.GetBytes(Message);
			byte[] header = new byte[] {
				(byte)((messageData.Length >> 8) & 0xFF), (byte)(messageData.Length & 0xFF)
			};
			byte[] data = new byte[header.Length + messageData.Length];

			header.CopyTo(data, 0);
			Array.Copy(messageData, 0, data, header.Length, messageData.Length);
			return data;
		}
	}
}

