﻿//  MeCab -- Yet Another Part-of-Speech and Morphological Analyzer
//
//  Copyright(C) 2001-2006 Taku Kudo <taku@chasen.org>
//  Copyright(C) 2004-2006 Nippon Telegraph and Telephone Corporation
using System;
using System.Collections.Generic;
using System.Text;
using System.IO;

namespace NMeCab.Core
{
    public class MeCabDictionary : IDisposable
    {
        #region  Const/Field/Property

        private const uint DictionaryMagicID = 0xEF718F77u;
        private const uint DicVersion = 102u;

        private Token[] tokens;
        private byte[] features;

        private DoubleArray da = new DoubleArray();

        private Encoding encoding;

        /// <summary>
        /// 辞書の文字コード
        /// </summary>
        public string CharSet
        {
            get { return this.encoding.WebName; }
        }

        /// <summary>
        /// バージョン
        /// </summary>
        public uint Version { get; private set; }

        /// <summary>
        /// 辞書のタイプ
        /// </summary>
        public DictionaryType Type { get; private set; }

        public uint LexSize { get; private set; }

        /// <summary>
        /// 左文脈 ID のサイズ
        /// </summary>
        public uint LSize { get; private set; }

        /// <summary>
        /// 右文脈 ID のサイズ
        /// </summary>
        public uint RSize { get; private set; }

        #endregion

        #region Open

        public void Open(BinaryReader reader)
        {
            uint magic = reader.ReadUInt32();
            //CanSeekの時のみストリーム長のチェック
            if (reader.BaseStream.CanSeek && reader.BaseStream.Length != (magic ^ DictionaryMagicID))
                throw new MeCabException("dictionary file is broken");

            this.Version = reader.ReadUInt32();
            if (this.Version != DicVersion)
                throw new MeCabException("incompatible version");

            this.Type = (DictionaryType)reader.ReadUInt32();
            this.LexSize = reader.ReadUInt32();
            this.LSize = reader.ReadUInt32();
            this.RSize = reader.ReadUInt32();
            uint dSize = reader.ReadUInt32();
            uint tSize = reader.ReadUInt32();
            uint fSize = reader.ReadUInt32();
            reader.ReadUInt32(); //dummy

            string charSet = StrUtils.GetString(reader.ReadBytes(32), Encoding.ASCII);
            this.encoding = Encoding.GetEncoding(charSet);

            this.da.Open(reader, dSize);

            this.tokens = new Token[tSize / 16];
            for (int i = 0; i < this.tokens.Length; i++)
                this.tokens [i] = Token.Create(reader);

            this.features = reader.ReadBytes((int)fSize);

            if (reader.BaseStream.ReadByte() != -1)
                throw new MeCabException("dictionary file is broken");
        }

        #endregion

        #region Search

        public DoubleArray.ResultPair ExactMatchSearch(string key)
        {
            return this.ExactMatchSearch(key, 0);
        }

        public DoubleArray.ResultPair ExactMatchSearch(string key, int nodePos)
        {
            DoubleArray.ResultPair result = this.da.ExactMatchSearch(this.encoding.GetBytes(key), nodePos);

            //文字数をデコードしたものに変換
            result.Length = key.Length;

            return result;
        }

        public int CommonPrefixSearch(string key, DoubleArray.ResultPair[] result, int rLen)
        {
            var bytes = this.encoding.GetBytes(key);
            int n = this.da.CommonPrefixSearch(bytes, result);

            //文字数をデコードしたものに変換
            for (int i = 0; i < n; i++)
            {
                result [i].Length = this.encoding.GetCharCount(bytes, 0, result [i].Length);
            }

            return n;
        }

        #endregion

        #region Get Infomation

        public Token[] GetToken(DoubleArray.ResultPair n)
        {
            Token[] dist = new Token[this.GetTokenSize(n)];
            int tokenPos = n.Value >> 8;
            Array.Copy(this.tokens, tokenPos, dist, 0, dist.Length);
            return dist;
        }

        public int GetTokenSize(DoubleArray.ResultPair n)
        {
            return 0xFF & n.Value;
        }

        public string GetFeature(uint featurePos)
        {
            return StrUtils.GetString(this.features, (int)featurePos, this.encoding);
        }

        #endregion

        #region etc.

        public bool IsCompatible(MeCabDictionary d)
        {
            return (this.Version == d.Version &&
                this.LSize == d.LSize &&
                this.RSize == d.RSize &&
                this.CharSet == d.CharSet);
        }

        #endregion

        #region Dispose

        private bool disposed;

        /// <summary>
        /// 使用されているリソースを開放する
        /// </summary>
        public void Dispose()
        {
            this.Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(bool disposing)
        {
            if (disposed)
                return;

            if (disposing)
            {
                if (this.da != null)
                    this.da.Dispose();
            }

            this.disposed = true;
        }

        ~MeCabDictionary ()
        {
            this.Dispose(false);
        }

        #endregion
    }
}
