__all__ = ("IMETextInput", )

from kivy.uix.textinput import TextInput
from kivy.base import EventLoop
from kivy.core.text.markup import MarkupLabel
from kivy.core.text import Label


class IMETextInput(TextInput):
    def __init__(self, **kwargs):
        super(IMETextInput, self).__init__(**kwargs)

        EventLoop.window.bind(on_textedit=self._on_textedit)
        self.editing = False
        self._spacehack = False

    def _on_textedit(self, window, text):
        self.editing = True if len(text) else False
        self.suggestion_text = text

        # hack: see below for the reason
        if not self.editing and self._spacehack:
            self.text = self.text.rsplit(" ", 1)[0]

    def keyboard_on_key_down(self, window, keycode, text, modifiers):
        if self.editing:
            return True  # IME consumes it all

        super(IMETextInput, self).keyboard_on_key_down(
                window, keycode, text, modifiers)

    def on_suggestion_text(self, window, value):
        # hack: suggestion can be at the beginning of a new textinput
        #       but _update_graphics() just looks at self._lines
        #       and won't show the suggestion if self._lines==[]
        #       even when _lines_labels is given
        if not len(self._lines) or (
                len(self._lines) == 1 and not self._lines[0]):
            self._set_text(" ")
            self.cursor = [0, 0]
            self._spacehack = True

        cursor_row = self.cursor_row
        if cursor_row >= len(self._lines) or self.canvas is None:
            return

        cursor_col = self.cursor_col
        txt = self._lines[cursor_row]
        pre = txt[:cursor_col]
        suf = txt[cursor_col:]

        kw = self._get_line_options()
        rct = self._lines_rects[cursor_row]

        lbl = None
        if value:
            lbl = MarkupLabel(
                text=pre + "[u]{}[/u]".format(value) + suf, **kw)
        else:
            lbl = Label(text=txt, **kw)

        lbl.refresh()

        self._lines_labels[cursor_row] = lbl.texture
        rct.size = lbl.size
        self._update_graphics()


if __name__ == '__main__':
    from kivy.uix.boxlayout import BoxLayout

    class RootWidget(BoxLayout):
        def __init__(self, *args, **kwargs):
            super().__init__(*args, **kwargs)

    from kivy.lang import Builder

    Builder.load_string("""
<RootWidget>:
    IMETextInput:
        text_language: "ja"
""")

    from kivy.app import App
    import japanize_kivy

    class IMEtest(App):
        def build(self):
            return RootWidget()

        def license(self):
            japanize_kivy.show_license()

    IMEtest().run()
