#!/usr/bin/env python
# -*- encoding: utf-8 -*-

# Copyright (c) 2014, tamanegi (tamanegi@users.sourceforge.jp)
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to
# deal in the Software without restriction, including without limitation the
# rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
# sell copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
# IN THE SOFTWARE.

# utilities (sub classes) for wm xml editor

import os
import re
import xml.etree.ElementTree as ET
import copy

"""utility classes for WMXmlEditor."""

# private xml element class for convenience
# But, some useful functions such as find might be sacrificed
class WMXmlElement:
  """XML element for WMXmlElement (no children)."""
  # this class does not store child elements
  def __init__( self, xml = None ):
    """constructor."""
    self.tag = ""
    self.text = None
    self.tail = None
    self.attrib = {}
    if not xml is None:
      self.copyFrom( xml )

  def copyFrom( self, xml ):
    """something like copy constructor."""
    self.tag = xml.tag
    self.text = copy.deepcopy( xml.text )
    self.tail = copy.deepcopy( xml.tail )
    self.attrib = copy.deepcopy( xml.attrib )

  # generator for attribute and its value... is it correct?
  def items( self ):
    """generator for attribute and its value."""
    for attr, val in self.attrib.items():
      yield copy.deepcopy(attr), copy.deepcopy(val)

class WMXmlQuery:
  """Xml query."""
  def __init__( self ):
    """constructor"""
    self.element = None
    self.attrName = None
    self.attrVal = None
    self.innerText = None

  def setElementQuery( self, string = None, ignorecase = False ):
    """element query setting."""
    if not string is None:
      if len(string.strip()) > 0:
        if ignorecase:
          self.element = re.compile( string.strip(), re.IGNORECASE )
        else:
          self.element = re.compile( string.strip() )

  def setAttrQuery( self, name = None, name_ic = False,
                          val = None, val_ic = False ):
    """attribute (name or/and value) query setting."""
    if not name is None:
      if len(name.strip()) > 0:
        if name_ic:
          self.attrName = re.compile( name.strip(), re.IGNORECASE )
        else:
          self.attrName = re.compile( name.strip() )
    if not val is None:
      if len(val.strip()) > 0:
        if val_ic:
          self.attrVal = re.compile( val.strip(), re.IGNORECASE )
        else:
          self.attrVal = re.compile( val.strip() )

  def setInnerTextQuery( self, string = None, ignorecase = False ):
    """inner text query setting."""
    if not string is None:
      if len(string.strip()) > 0:
        if ignorecase:
          self.innerText = re.compile( string.strip(), re.IGNORECASE )
        else:
          self.innerText = re.compile( string.strip() )

  def match( self, wmxmlelem ):
    """return whether current query matches with given element."""
    if not self.element is None:
      # use search, not match
      if not self.element.search( wmxmlelem.tag ):
        return False
    if not self.attrName is None or not self.attrVal is None:
      found = False
      for name, val in wmxmlelem.attrib.items():
        if self._matchAttrib( name, val ):
          found = True
          break
      if not found:
        return False
    if not self.innerText is None:
      if wmxmlelem.text is None:
        return False
      if len(wmxmlelem.text) == 0:
        return False
      if not self.innerText.search( wmxmlelem.text ):
        return False
    return True

  def _matchAttrib( self, name, val ):
    if not self.attrName is None:
      if not self.attrName.search( name ):
        return False
    if not self.attrVal is None:
      if not self.attrVal.search( val ):
        return False
    return True

### container for static parameter for WMXmlEditor
class WMXmlEditorParams:
  """WMXmlEditor static optional variables."""
  # ListCtrl: color for elements having text inside
  LC_emphasis_color = "blue"
  # ListCtrl: indent size
  LC_indent_size = 2

  # TreeCtrl: whether attribute names are shown
  TC_showattr_name = False
  TC_showattr_value = False # active only when showattr_name is True

  # undo/redo settings
  UNDO_queue_size = 15

  # find settings
  FIND_clear_wo_query = True

  # This function should not be called.
  # Instances of this class do not have variables, functions
  # This class has only class variables and class methods
  def __init__( self ):
    pass

  @classmethod
  def readConfigFromFile( cls, filename ):
    if len( filename ) == 0 or not os.path.exists( filename ):
      print _("Info: skip reading config file.")
      return
    xmlroot = ET.parse( filename ).getroot()
    if xmlroot is None:
      return
    # root element must be "wmxconf"
    if xmlroot.tag.lower() !=  "wmconf":
      return
    for child in xmlroot:
      if child.tag.lower() == "undo": # Undo related
        cls.parseUndoParams( child )
      elif child.tag.lower() == "find": # Find related
        cls.parseFindParams( child )
      elif child.tag.lower() == "lc": # ListCtrl
        cls.parseListCtrlParams( child )
      elif child.tag.lower() == "tc": # TreeCtrl
        cls.parseTreeCtrlParams( child )

  @classmethod
  def parseListCtrlParams( cls, element ):
    if element.attrib.has_key( "emphasis" ):
      cls.LC_emphasis_color = element.attrib["emphasis"]
    if element.attrib.has_key( "indent" ):
      value_save = cls.LC_indent_size
      try:
        cls.LC_indent_size = int(element.attrib["indent"])
      except ValueError:
        print _("Error: invalid indent size detected. please input integer value")
        cls.LC_indent_size = value_save

  @classmethod
  def parseTreeCtrlParams( cls, element ):
    if element.attrib.has_key( "showattr_name" ):
      try:
        value = int(element.attrib["showattr_name"])
        cls.TC_showattr_name = ( value > 0 )
      except ValueError:
        print _("Error: value of \"showattr_name\" must be integer")
    if element.attrib.has_key( "showattr_value" ):
      try:
        value = int(element.attrib["showattr_value"])
        cls.TC_showattr_value = ( value > 0 )
      except ValueError:
        print _("Error: value of \"showattr_value\" must be integer")

  @classmethod
  def parseUndoParams( cls, element ):
    if element.attrib.has_key( "no_query_trunc" ):
      try:
        value = int(element.attrib["no_query_trunc"])
        cls.UNDO_truncate_queue_wo_query = ( value > 0 )
      except ValueError:
        print _("Error: value of \"no_query_trunc\" must be integer")
    if element.attrib.has_key( "qsize" ):
      try:
        value = int(element.attrib["qsize"])
        cls.UNDO_queue_size = value
      except ValueError:
        print _("Error: value of \"qsize\" must be integer")

  @classmethod
  def parseFindParams( cls, element ):
    if element.attrib.has_key( "clear_wo_query" ):
      try:
        value = int(element.attrib["clear_wo_query"])
        cls.FIND_clear_wo_query = ( value > 0 )
      except ValueError:
        print _("Error: value of \"clear_wo_query\" must be integer")

# abstract class
# Action such as edit, add, remove
# Used to undo (and redo?) actions
class Action:
  """abstract class for actions such as add, delete, edit elements."""
  def __init__( self ):
    pass

  def undo( self, treectrl ):
    return True

# edit action; reversed action is also edit
class ActionEdit( Action ):
  """edit action."""
  def __init__( self, itemid = None, myxml = None ):
    Action.__init__( self )
    # "itemid" is the editted TreeItemId of TreeCtrl
    # "myxml" is a xml data before modification
    if not itemid is None and not myxml is None:
      self.initialize( itemid, myxml )

  def initialize( self, itemid, myxml ):
    ## copy and deepcopy does not work for TreeItemId
    self.itemid = itemid # is it safe?
    self.myxml = copy.deepcopy( myxml )

  # replace ItemPyData of the specified itemid by myxml
  def undo( self, treectrl, wmxmled ):
    # delete existing data
    olddata = treectrl.GetItemPyData( self.itemid )
    newxml = copy.deepcopy(olddata[1])
    if not olddata is None: # not necessary?
      del olddata
    treectrl.SetItemText( self.itemid, wmxmled.createTreeCtrlLabel( self.myxml ) )
    treectrl.SetItemPyData( self.itemid, [ 0, self.myxml ] )
    self.myxml = newxml # replace for redo
    # ListCtrl is not updated here
    return True

  def reverse( self ):
    return ActionEdit( self.itemid, self.myxml )

# edit multipe elements at once
class ActionEdits( Action ):
  """edit multiple elements action."""
  def __init__( self, itemids = None, myxmls = None ):
    Action.__init__( self )
    if not itemids is None and not myxmls is None:
      self.initialize( itemids, myxmls )

  def initialize( self, itemids, myxmls ):
    self.itemids = itemids
    self.myxmls = copy.deepcopy( myxmls )

  def undo( self, treectrl, wmxmled ):
    olddata = []
    newxmls = []
    for itemid in self.itemids:
      olddata.append( treectrl.GetItemPyData( itemid ) )
      newxmls.append( olddata[-1][1] )
    if not olddata:
      del olddata
    for i in range( 0, len(self.itemids) ):
      itemid = self.itemids[i]
      myxml = self.myxmls[i]
      treectrl.SetItemText( itemid, wmxmled.createTreeCtrlLabel( myxml ) )
      treectrl.SetItemPyData( itemid, [ 0, myxml ] )
    self.myxmls = newxmls
    return True

  def reverse( self ):
    return( ActionEdits( self.itemids, self.myxmls ) )

# add action; reversed action is delete
class ActionAdd( Action ):
  """add element action."""
  def __init__( self, uniqids = None, pos = None, itemid = None, myxml = None ):
    Action.__init__( self )
    # itemid is an TreeItemId of added object
    # myxml is optional argument now
    if not itemid is None:
      self.initialize( uniqids, pos, itemid, myxml )

  def initialize( self, uniqids, pos, itemid, myxml = None ):
    self.uniqids = copy.deepcopy( uniqids )
    self.pos = copy.deepcopy( pos )
    self.itemid = itemid
    self.myxml = copy.deepcopy( myxml )

  # simply delete specified data
  def undo( self, treectrl, wmxmled ):
    self.itemid = wmxmled.findItemIdByUniqId( self.uniqids[1] )
    self.pos = wmxmled.calculateItemPosition( self.itemid )
    # do not update uniqids here
    self.myxml, uniqids = wmxmled.createElementTreeFromTreeCtrl( self.itemid )
    self.myxml = self.myxml.getroot()
    # ListCtrl is not updated here
    return wmxmled.deleteTreeCtrlItem( self.itemid )

  def reverse( self ):
    return ActionDel( self.uniqids, self.pos, self.itemid, self.myxml )

class ActionAdds( Action ):
  """add multiple elements action."""
  def __init__( self, uniqids = None, pos = None, itemids = None, myxmls = None ):
    Action.__init__( self )
    if len(myxmls) > 0:
      self.initialize( uniqids, pos, itemids, myxmls )

  def initialize( self, uniqids, pos, itemids, myxmls ):
    mylen = len( uniqids )
    self.myadds = []
    # length shall be same
    #if len(pos) != mylen or len(itemids) != mylen or len(myxmls) != mylen:
    if len(itemids) != mylen or len(myxmls) != mylen:
      return
    for i in range( 0, mylen ):
      self.myadds.append( ActionAdd( uniqids[i], 0, itemids[i], myxmls[i] ) )

  def undo( self, treectrl, wmxmled ):
    self.myadds.reverse()
    for myadd in self.myadds:
      myadd.undo( treectrl, wmxmled )
    return True

  def reverse( self ):
    uniqids = []
    pos = []
    itemids = []
    myxmls = []
    for myadd in self.myadds:
      uniqids.append( myadd.uniqids )
      pos.append( myadd.pos )
      myxmls.append( myadd.myxml )
    return ActionDels( uniqids, pos, itemids, myxmls )

# delete action; reversed action is add
class ActionDel( Action ):
  """delete element action."""
  def __init__( self, uniqids = None, pos = None, itemid = None, myxml = None ):
    Action.__init__( self )
    # note: myxml is not WMXmlElement, but is Element object of ET
    #       itemid is optional, currently unused.
    if not myxml is None:
      self.initialize( uniqids, pos, itemid, myxml )

  def initialize( self, uniqids, pos, itemid, myxml ):
    self.uniqids = copy.deepcopy( uniqids )
    self.pos = copy.deepcopy( pos )
    self.itemid = itemid
    self.myxml = copy.deepcopy( myxml )

  # wmxmled is the main class; WMXmlEditor in wmxmled.py
  def undo( self, treectrl, wmxmled ):
    if treectrl.IsEmpty(): # maybe root element
      wmxmled.createXmlTree( self.myxml )
    else: # add as a child element
      tag = wmxmled.createTreeCtrlLabel( self.myxml )
      parentid = wmxmled.findItemIdByUniqId(self.uniqids[0])
      # find parent from saved uniqid
      if self.pos < 0:
        # if the original item is the last one
        # i.e. GetNextSibling(itemid) returns false
        self.itemid = treectrl.AppendItem( parentid, tag, wmxmled.imageFolder, wmxmled.imageFolderOpen )
      else:
        self.itemid = treectrl.InsertItemBefore( parentid, self.pos, tag, wmxmled.imageFolder, wmxmled.imageFolderOpen )
      treectrl.SetItemPyData( self.itemid, [ 0, WMXmlElement( self.myxml ), self.uniqids[1] ] )
      # save current uniqid
      uniqid_tmp = wmxmled.uniqid
      # add to tree
      wmxmled.createXmlTreeRecursive( self.myxml, self.itemid )
      # reassign uniq ids
      wmxmled.reassignUniqIDsRecursive( self.itemid, self.uniqids )
      # reset uniqid
      wmxmled.uniqid = uniqid_tmp
    # ListCtrl is not updated here
    return True

  def reverse( self ):
    return ActionAdd( self.uniqids, self.pos, self.itemid, self.myxml )

# multiple deletes
# registered actions should not be nested
class ActionDels( Action ):
  """delete multiple elements action."""
  def __init__( self, uniqids = None, pos = None, itemids = None, myxmls = None ):
    Action.__init__( self )
    if len(myxmls) > 0:
      self.initialize( uniqids, pos, itemids, myxmls )

  def initialize( self, uniqids, pos, itemids, myxmls ):
    mylen = len( uniqids )
    self.mydels = []
    # lengths shall be same
    #if len(pos) != mylen or len(itemids) != mylen or len(myxmls) != mylen:
    if len(pos) != mylen or len(myxmls) != mylen:
      return
    for i in range( 0, mylen ):
      self.mydels.append( ActionDel( uniqids[i], pos[i], None, myxmls[i] ) )

  def undo( self, treectrl, wmxmled ):
    self.mydels.reverse()
    for mydel in self.mydels:
      mydel.undo( treectrl, wmxmled )
    return True

  def reverse( self ):
    uniqids = []
    pos = []
    itemids = []
    myxmls = []
    for mydel in self.mydels:
      uniqids.append( mydel.uniqids )
      pos.append( mydel.pos )
      itemids.append( mydel.itemid )
      myxmls.append( mydel.myxml )
    return ActionAdds( uniqids, pos, itemids, myxmls )
