<?php
/**
 * GenerateUtil.
 * @package magic.core
 * @subpackage util
 */
/**
 * 簡易テンプレートエンジンです.
 * <p>
 * デザインとロジックの分離を簡易的に提供します。<br/>
 * 絶対にHTMLファイルにロジック(foreachなど)を書きたくない場合に使用してください。<br/>
 * 使い方は少し難しいですが、原理は単純です。<br/>
 * データベースから取得した値の配列が、{@link HttpResponse}に、
 * 'nameRow'というキーで格納されていると仮定します。<br/>
 * 内容は単純に、<br/>
 * Array<br/>
 * (<br/>
 *    [name] => リンゴ<br/>
 *    [name] => ミカン<br/>
 *    [name] => バナナ<br/>
 * )<br/>
 * です。<br/>
 * 例えば、このようなテンプレートファイルを準備します。<br/>
 * ＜tr＞<br/>
 *     ＜td＞${name}＜/td＞<br/>
 * ＜/tr＞<br/>
 * HTMLファイル側はこのようになります。<br/>
 * ＜table＞<br/>
 *     ＜tr＞名前＜/tr＞<br/>
 *     ＜?=$template＞<br/>
 * ＜/table＞<br/>
 * Viewクラスの{@link View::preRender()}ファンクションで、<br/>
 * $contents = file_get_contents(テンプレートファイル);<br/>
 * foreach ($this->response->get('nameRow') as $row) {<br/>
 *     $template .= GenerateUtil::staticCreate($contents, $row);<br/>
 * }<br/>
 * $this->response->set('template', $template);<br/>
 * のように書く事で、HTML側には＜tr＞部分が3行生成された文字列が返されます。
 * 出力結果は以下のようになります。<br/>
 * ----------<br/>
 * |  名前  |<br/>
 * ----------<br/>
 * | リンゴ |<br/>
 * ----------<br/>
 * | ミカン |<br/>
 * ----------<br/>
 * | バナナ |<br/>
 * ----------<br/>
 * 配列のキー名が、テンプレートファイルの${xxx}に対応している事に注意してください。<br/>
 * また、レスポンスにセットしたキー名が、HTML側では変数名になる事にも注意してください。
 * </p>
 * @package magic.core
 * @subpackage util
 * @author T.Okumura
 * @version 1.0.0
 * @final
 */
final class GenerateUtil {
    /**
     * 変換パターンを保持します.
     * @var string
     */
    private static $pattern = '/\$\{(\w+)\}/';
    /**
     * 置換対象の文字列配列を保持します.
     * @var array
     */
    private static $row = array();
    /**
     * コンストラクタ.
     */
    private function __construct() {
    }
    /**
     * テンプレートの対象文字列を置換します.
     * <p>
     * {@link GenerateUtil::staticCreate()}との違いは、ファイルの内容が渡されるか、
     * ファイルのパスが渡されるかだけです。
     * </p>
     * @param string $path ファイルのパス
     * @param array $parameters 置換対象文字列の配列
     * @return string 置換結果の文字列全体
     * @static
     */
    public static function dynamicCreate($path, array $parameters) {
        self::$row = $parameters;
        return preg_replace_callback(self::$pattern, array('self', '_createCallback'), file_get_contents($path));
    }
    /**
     * テンプレートの対象文字列を置換します.
     * <p>
     * {@link GenerateUtil::dynamicCreate()}との違いは、ファイルの内容が渡されるか、
     * ファイルのパスが渡されるかだけです。
     * </p>
     * @param string $contents 置換対象のファイル内容
     * @param array $parameters 置換対象文字列の配列
     * @return string 置換結果の文字列全体
     * @static
     */
    public static function staticCreate($contents, array $parameters) {
        self::$row = $parameters;
        return preg_replace_callback(self::$pattern, array('self', '_createCallback'), $contents);
    }
    /**
     * 置換コールバックファンクション.
     * @param string $key 置換対象のキー
     * @return string キーを置換対象の文字列に変換した文字列
     * @static
     */
    private static function _createCallback($key) {
        return self::$row[$key[1]];
    }
}
// EOF.