<?php
/**
 * HttpResponse.
 * @package magic.core
 * @subpackage helper.http
 */
/**
 * レスポンスを管理するクラスです.
 * <p>
 * HTMLファイルのvalue属性に指定された変数名と、Dtoクラスのゲッターメソッドの名前は、
 * フレームワークによって解決され、同一名称が存在すれば自動的にextractされます。<br/>
 * また、ユーザーが個別にレスポンスを設定する事もできます。
 * </p>
 * @package magic.core
 * @subpackage helper.http
 * @author T.Okumura
 * @version 1.0.0
 * @final
 */
final class HttpResponse {
    /**
     * レスポンスの情報を保持します.
     * @var array
     */
    private $_response = array();
    /**
     * コンストラクタ.
     */
    public function __construct() {
    }
    /**
     * レスポンスから値を取得します.
     * @param string $key 取得する値のキー
     * @return mixed 取得できた場合はその値、できなければFALSE
     */
    public function get($key) {
        return array_key_exists($key, $this->_response) ? $this->_response[$key] : FALSE;
    }
    /**
     * すべてのレスポンスを取得します.
     * <p>
     * 返されるのはレスポンスのコピーです。<br/>
     * これによって返された配列に変更を加えても、元のレスポンスの内容は変更されない事に注意してください。
     * </p>
     * @return array コピーされたレスポンスの配列
     */
    public function getAll() {
        $copy = $this->_response;
        return $copy;
    }
    /**
     * レスポンスに値を設定します.
     * <p>
     * 同じキーがあれば、値が上書きされます。
     * </p>
     * @param string $key 設定する値のキー
     * @param mixed $value 設定する値
     */
    public function set($key, $value) {
        $this->_response[$key] = $value;
    }
    /**
     * 渡された配列をレスポンスにマージして設定します.
     * <p>
     * 同じキーがあれば、値が上書きされます。
     * </p>
     * @param array $list レスポンスに設定する値の配列
     */
    public function setAll(array $list) {
        $this->_response = array_merge($this->_response, $list);
    }
    /**
     * レスポンスから値を削除します.
     * @param string $key 削除する値のキー
     */
    public function remove($key) {
        if (array_key_exists($key, $this->_response)) {
            unset($this->_response[$key]);
        }
    }
    /**
     * レスポンスを置換します.
     * <p>
     * レスポンスの内容がすべて置き換わる事に注意してください。<br/>
     * フレームワークによるextractの前に呼び出された場合、
     * HTMLファイルの変数に正しくセットされないかも知れません。
     * </p>
     * @param array $list 置換する配列
     */
    public function replace(array $list) {
        $this->_response = $list;
    }
    /**
     * レスポンスで保持している情報をすべて削除します.
     */
    public function clear() {
        $this->_response = array();
    }
    /**
     * ブラウザのキャッシュを抑制します.
     * <p>
     * このファンクションは、Viewクラスによるレンダリングが開始される前にコールする必要があります。
     * </p>
     */
    public function noCache() {
        header('Expires: Mon, 26 Jul 1997 05:00:00 GMT');
        header('Last-Modified: ' . gmdate("D, d M Y H:i:s") . ' GMT');
        header('Cache-Control: no-store, no-cache, must-revalidate');
        header('Cache-Control: post-check=0, pre-check=0', FALSE);
        header('Pragma: no-cache');
    }
}
// EOF.