<?php
/**
 * HttpRequest.
 * @package magic.core
 * @subpackage helper.http
 */
/**
 * リクエストを管理するクラスです.
 * <p>
 * <var>$_GET</var>や<var>$_POST</var>を操作するための機能を提供します。<br/>
 * GETリクエストとPOSTリクエストはマージされ、同一に扱われます。<br/>
 * 常にPOSTリクエストがGETリクエストを上書きする事に注意してください。
 * </p>
 * <p>
 * HTMLファイルのname属性に指定された名前と、Dtoクラスのセッターメソッドの名前は、
 * フレームワークによって解決され、同一名称が存在すれば自動的にインジェクションされます。
 * </p>
 * @package magic.core
 * @subpackage helper.http
 * @author T.Okumura
 * @version 1.0.0
 * @final
 */
final class HttpRequest {
    /**
     * リクエストの情報を保持します.
     * @var array
     */
    private $_request = array();
    /**
     * コンストラクタ.
     * <p>
     * GETリクエストとPOSTリクエストに同一のキーが存在する場合は、POSTリクエストの情報を保持します。<br/>
     * すべてのリクエストされた値は、前後の半角スペースがトリミングされます。
     * </p>
     * @see
     */
    public function __construct() {
        $this->replace(StringUtil::trim(array_merge($_GET, $_POST)));
    }
    /**
     * リクエストから値を取得します.
     * @param string $key 取得する値のキー
     * @return mixed 取得できた場合はその値、できなければFALSE
     */
    public function get($key) {
        return array_key_exists($key, $this->_request) ? $this->_request[$key] : FALSE;
    }
    /**
     * すべてのリクエストを取得します.
     * <p>
     * 返されるのはリクエストのコピーです。<br/>
     * 元の<var>$_GET</var>や<var>$_POST</var>の内容は変更されませんが、<br/>
     * フレームワークによる自動的なinject処理に影響を与える可能性があります。
     * </p>
     * @return array リクエストの配列
     */
    public function getAll() {
        return $this->_request;
    }
    /**
     * リクエストに値を設定します.
     * <p>
     * 同じキーがあれば、値が上書きされます。
     * </p>
     * @param string $key 設定する値のキー
     * @param mixed $value 設定する値
     */
    public function set($key, $value) {
        $this->_request[$key] = $value;
    }
    /**
     * 渡された配列をリクエストにマージして設定します.
     * <p>
     * 同じキーがあれば、値が上書きされます。
     * </p>
     * @param array $list リクエストに設定する値の配列
     */
    public function setAll(array $list) {
        $this->_request = array_merge($this->_request, $list);
    }
    /**
     * リクエストから値を削除します.
     * @param string $key 削除する値のキー
     */
    public function remove($key) {
        if (array_key_exists($key, $this->_request)) {
            unset($this->_request[$key]);
        }
    }
    /**
     * リクエストを置換します.
     * <p>
     * リクエストの内容がすべて置き換わる事に注意してください。<br/>
     * フレームワークによるinjectの前に呼び出された場合、
     * Dtoクラスに正しくセットされないかも知れません。
     * </p>
     * @param array $list 置換する配列
     */
    public function replace(array $list) {
        $this->_request = $list;
    }
    /**
     * リクエストで保持している情報をすべて削除します.
     */
    public function clear() {
        $this->_request = array();
    }
}
// EOF.