/**
 * \file nco.c
 * \brief Numerically Controlled Oscillator
 * \details
 * Sine/Cosine algorithm by table lookup and approximation.
 *
 * \date 2013/04/21
 * \author takemasa
 */

#ifndef NCO_H_
#define NCO_H_

#ifdef __i386__
#include "fract_x86.h"
#else
#include <fract.h>
#endif

#include <limits.h>

/**
 * \brief NCO object type
 * \details
 * An object type of NCO. All NCO initialize parameter and internal state are packed into this object type.
 * Programmer shuld not read/write these values directory.
 *
 * Use fract32_nco_init to initialize this object.
 */
typedef struct {
	fract32 frequency;
	fract32 phase;
} TNCO;

/**
 * \brief NCO object initialization
 * \param nco a pointer to the NCO management object variable.
 * \param frequency the frequency
 * \details
 * initialize the given NCO object with parameter frequency.
 * This function have to be called once, before calling fract32_nco.
 * The internal phase is initialized as zero.
 *
 * The frequency parameter specify the incremental value for each sample output.
 * This normalized frequency can be obtained by following formula.
 *
 * \code
 * frequency_parameter = desired_frequency * 2^32 / Fs
 * where Fs is the sampling frequency.
 * \/code
 *
 * For example, if the desired frequency is 1kHz and the Fs is 48kHz, the parameter is
 * 89478485 ( = 1000 * 2^32 / 48000 ).
 *
 * Note : The frequency parameter can be negative value. In this case, the
 * output signal represents the negative frequency.
 *
 *
 */
extern void fract32_nco_init ( TNCO * nco, fract32 frequency);

extern void fract32_nco_set_freq( TNCO * nco, fract32 frequency);
extern fract32 fract32_nco_get_freq( TNCO * nco);
extern void fract32_nco_set_phase( TNCO * nco, fract32 phase);
extern fract32 fract32_nco_get_phase( TNCO * nco);
extern void fract32_nco( TNCO *nco, fract32 c[], fract32 s[], int count );


#endif /* NCO_H_ */
