
#ifndef INC_CM_THREAD_SPECIFIC_DATA_H_
#define INC_CM_THREAD_SPECIFIC_DATA_H_

#include <assert.h>
#include <pthread.h>

#include "cm_thread_is_main_thread.h"
#include "mt_auto_ptr.h"

namespace cm {

template <typename T>
class ThreadSpecificData;

template <typename T>
class ThreadKey
{
friend class ThreadSpecificData<T>;
public:
    static ThreadKey<T>& instance()
    {
        static ThreadKey<T> singleton;
        return singleton;
    }

private:
    ThreadKey()
        : key_()
    {
        if (0 != pthread_key_create(&key_, &ThreadKey<T>::performDestruction)) {
            assert(false);
        }
    }

    static mt::AutoPtr<T>& getMainThreadObjectHolder()
    {
        static mt::AutoPtr<T> holder;
        return holder;
    }

    // Non-copyable
    ThreadKey(const ThreadKey& rhs);
    ThreadKey& operator=(const ThreadKey& rhs);

    static void performDestruction(void* ptr)
    {
        delete static_cast<T*>(ptr);
    }

    pthread_key_t key_;
};

template <typename T>
class ThreadSpecificData
{
public:
    static bool isEmpty()
    {
        if (!isMainThread()) {
            ThreadKey<T>& key = ThreadKey<T>::instance();
            return static_cast<T*>(pthread_getspecific(key.key_)) == 0;
        } 
        return ThreadKey<T>::getMainThreadObjectHolder().get() == 0;
    }

    static void set(T* ptr)
    {
        ThreadKey<T>& key = ThreadKey<T>::instance();

        if (!isMainThread()) {
            T* current = static_cast<T*>(pthread_getspecific(key.key_));

            if (current) {
                delete current;
            }
            if (0 != pthread_setspecific(key.key_, ptr)) {
                assert(false);
            }
        } else {
            mt::AutoPtr<T>& main_auto_ptr = ThreadKey<T>::getMainThreadObjectHolder();
            main_auto_ptr.reset();
            main_auto_ptr.set(ptr);
        }
        return;
    }

    static T& get()
    {
        ThreadKey<T>& key = ThreadKey<T>::instance();

        if (!isMainThread()) {
            return *static_cast<T*>(pthread_getspecific(key.key_));
        } else {
            return *ThreadKey<T>::getMainThreadObjectHolder().get();
        }
    }

private:
    // Non-copyable
    ThreadSpecificData(const ThreadSpecificData<T>& rhs);
    ThreadSpecificData<T>& operator=(const ThreadSpecificData<T>& rhs);
};

} // namespace cm

#endif // INC_CM_THREAD_SPECIFIC_DATA_H_

