
#ifndef INC_CM_THREAD_EVENT_DRIVEN_H_
#define INC_CM_THREAD_EVENT_DRIVEN_H_

#include <unistd.h>

#include "types.h"
#include "mt_auto_ptr.h"
#include "mt_shim_clear_memory.h"
#include "mt_container_foreach.h"
#include "mt_array_length.h"
#include "cm_thread.h"
#include "cm_socket.h"
#include "cm_event.h"
#include "cm_mutex.h"
#include "cm_condition.h"

namespace cm {

template <typename Arg>
struct ThreadArgEventDriven
{
    ThreadArgEventDriven(Arg* arg_ptr)
        : arg(arg_ptr),
          mutex(), cond(),
          pipe_socket(),
          is_thread_created(false)
    {
        int pipe_fds[2];
        int ret = pipe(pipe_fds);
        assert(ret == 0);

        for (uint i = 0; i < ARRAY_LENGTH(pipe_socket); ++i) {
            pipe_socket[i].reset();
            pipe_socket[i].set(new cm::Socket(pipe_fds[i]));
        }
    }

    Arg* arg;
    cm::Mutex mutex;
    cm::Condition cond;
    mt::AutoPtr<cm::Socket> pipe_socket[2];
    bool is_thread_created;
};

template <>
struct ThreadArgEventDriven<void>
{
    ThreadArgEventDriven()
        : mutex(),
          pipe_socket(),
          is_thread_created(false)
    {
        int pipe_fds[2];
        int ret = pipe(pipe_fds);
        assert(ret == 0);

        for (uint i = 0; i < ARRAY_LENGTH(pipe_socket); ++i) {
            pipe_socket[i].reset();
            pipe_socket[i].set(new cm::Socket(pipe_fds[i]));
        }
    }

    cm::Mutex mutex;
    cm::Condition cond;
    mt::AutoPtr<cm::Socket> pipe_socket[2];
    bool is_thread_created;
};


template <typename Handler, typename Arg = void>
class ThreadEventDriven
{
public:
    ThreadEventDriven(Arg* arg)
        : arg_(arg), thread_(Handler::getName())
    {
        Mutex::Lock lock(arg_.mutex);
        thread_.create(&arg_);
        if (!arg_.is_thread_created) {
            arg_.cond.wait(arg_.mutex);
        }
    }

    ThreadEventDriven()
        : arg_(), thread_(Handler::getName())
    {
        Mutex::Lock lock(arg_.mutex);
        thread_.create(&arg_);
        if (!arg_.is_thread_created) {
            arg_.cond.wait(arg_.mutex);
        }
    }

    ~ThreadEventDriven()
    {
        size_t bytes_written = 0;
        (arg_.pipe_socket[1])->write(bytes_written, "quit", sizeof("quit") - 1);

        char_t terminate_response_buffer[128];
        mt::clearMemory(terminate_response_buffer);
        thread_.join();
    }

private:
    // Non-copyable
    ThreadEventDriven(const ThreadEventDriven& rhs);
    ThreadEventDriven& operator=(const ThreadEventDriven& rhs);

    template <typename HandlerType, typename ArgType>
    struct ThreadObjectEventDriven
    {
        ThreadObjectEventDriven(ThreadArgEventDriven<ArgType>* arg)
            : event_(cm::Event::tsdInstance()),
              handler_(arg->arg),
              arg_ptr_(arg),
              loop_(true)
        {
            event_.addHandlerRead(*this, &ThreadObjectEventDriven<HandlerType, ArgType>::handleTerminate, *(arg_ptr_->pipe_socket[0]));
        }

        ~ThreadObjectEventDriven()
        {
            event_.delHandlerRead(*(arg_ptr_->pipe_socket[0]));
        }

        bool handleTerminate(cm::Socket& event)
        {
            assert(getFD(event) == getFD(*(arg_ptr_->pipe_socket[0])));

            char_t terminate_request_buffer[128];
            mt::clearMemory(terminate_request_buffer);
            size_t bytes_read = 0;
            bool ret = (arg_ptr_->pipe_socket[0])->read(bytes_read, terminate_request_buffer, sizeof(terminate_request_buffer) - 1);
            assert(ret == true);

            loop_ = false;
            return false;
        }

        void run()
        {
            {
                cm::Mutex::Lock lock(arg_ptr_->mutex);
                arg_ptr_->is_thread_created = true;
                arg_ptr_->cond.broadcast();
            }

            while (loop_) {
                event_.pend();
            }
        }

        cm::Event& event_;
        HandlerType handler_;
        ThreadArgEventDriven<ArgType>* arg_ptr_;
        bool loop_;

    private:
        // Non-copyable
        ThreadObjectEventDriven(const ThreadObjectEventDriven<HandlerType, ArgType>& rhs);
        ThreadObjectEventDriven<HandlerType, ArgType>& operator=(const ThreadObjectEventDriven<HandlerType, ArgType>& rhs);
    };

    template <typename HandlerType>
    struct ThreadObjectEventDriven<HandlerType, void>
    {
        ThreadObjectEventDriven(ThreadArgEventDriven<void>* arg)
            : event_(cm::Event::tsdInstance()),
              handler_(),
              arg_ptr_(arg),
              loop_(true)
        {
            event_.addHandlerRead(*this, &ThreadObjectEventDriven<HandlerType, void>::handleTerminate, *(arg_ptr_->pipe_socket[0]));
        }

        ~ThreadObjectEventDriven()
        {
            event_.delHandlerRead(*(arg_ptr_->pipe_socket[0]));
        }

        bool handleTerminate(cm::Socket& event)
        {
            assert(getFD(event) == getFD(*(arg_ptr_->pipe_socket[0])));

            char_t terminate_request_buffer[128];
            mt::clearMemory(terminate_request_buffer);
            size_t bytes_read = 0;
            bool ret = (arg_ptr_->pipe_socket[0])->read(bytes_read, terminate_request_buffer, sizeof(terminate_request_buffer) - 1);
            assert(ret == true);

            loop_ = false;
            return false;
        }

        void run()
        {
            {
                cm::Mutex::Lock lock(arg_ptr_->mutex);
                arg_ptr_->is_thread_created = true;
                arg_ptr_->cond.broadcast();
            }

            while (loop_) {
                event_.pend();
            }
        }

        cm::Event& event_;
        HandlerType handler_;
        ThreadArgEventDriven<void>* arg_ptr_;
        bool loop_;

    private:
        // Non-copyable
        ThreadObjectEventDriven(const ThreadObjectEventDriven<HandlerType, void>& rhs);
        ThreadObjectEventDriven<HandlerType, void>& operator=(const ThreadObjectEventDriven<HandlerType, void>& rhs);
    };

    ThreadArgEventDriven<Arg> arg_;
    cm::Thread<ThreadObjectEventDriven<Handler, Arg>, ThreadArgEventDriven<Arg> > thread_;
};

} // namespace cm

#endif // INC_CM_THREAD_EVENT_DRIVEN_H_

