
#ifndef INC_CM_IO_VECTOR_H_
#define INC_CM_IO_VECTOR_H_

#include "mt_static_assert.h"
#include "mt_member_detector.h"
#include "mt_typelist.h"
#include "mt_typelist_algo_utility.h"

namespace cm {

struct IOVectorBase
{
    virtual ~IOVectorBase()
    {}

    virtual size_t setToVector(struct iovec*, size_t skip_bytes = 0) const = 0;
    virtual void setRelease() const = 0;
    virtual bool getRelease() const = 0;
};

template <typename TL, typename TLList>
struct IOVectorChain : IOVectorBase
{
public:
    typedef typename TL::Head T;
    typedef typename TL::Tail RF;

    virtual void setRelease() const
    {
        should_release_ = false;
    }

    virtual bool getRelease() const
    {
        return should_release_;
    }

    IOVectorChain(const IOVectorChain<TL, TLList>& rhs)
        : ptr_(rhs.ptr_), size_(rhs.size_), release_functor_(rhs.release_functor_), tail_(rhs.tail_), should_release_(rhs.getRelease())
    {
        rhs.setRelease();
    }

    IOVectorChain(const T* ptr, size_t size, RF release_functor, const IOVectorChain<typename TLList::Head, typename TLList::Tail>& obj)
        : ptr_(ptr), size_(size), release_functor_(release_functor), tail_(obj), should_release_(true)
    {}

    template <typename Functor>
    void performReleaseFunctor(Functor& func, typename mt::EnableIf<mt::HasFunctorOperator<Functor>::value>::Result* = 0)
    {
        release_functor_();
    }

    template <typename Functor>
    void performReleaseFunctor(Functor&, typename mt::EnableIf<!mt::HasFunctorOperator<Functor>::value>::Result* = 0)
    {
    }

    virtual ~IOVectorChain()
    {
        if (should_release_) {
            performReleaseFunctor(release_functor_);
        }
    }

    virtual size_t setToVector(struct iovec* iovec_ptr, size_t skip_bytes = 0u) const
    {
        size_t offset = 0;
        size_t skipped_bytes = doSetToVector(iovec_ptr, offset, skip_bytes);
        assert(skipped_bytes == 0);
        return offset;
    }

    size_t doSetToVector(struct iovec* iovec_ptr, size_t& offset, size_t skip_bytes) const
    {
        size_t skipped_bytes = tail_.doSetToVector(iovec_ptr, offset, skip_bytes);
        assert(skip_bytes >= skipped_bytes);

        if (size_ >= skipped_bytes) {
            char* ptr = const_cast<char*>(reinterpret_cast<const char*>(ptr_));
            iovec_ptr[offset].iov_base = reinterpret_cast<void*>(ptr + skipped_bytes);
            iovec_ptr[offset].iov_len = size_ - skipped_bytes;
            ++offset;
            return 0u;
        }
        return (skipped_bytes - size_);
    }

    template <typename X, typename XRF>
    IOVectorChain< mt::Typelist<X, XRF>, mt::Typelist<TL, TLList> >
    operator()(const X* ptr, size_t size, XRF x_release_functor)
    {
        return IOVectorChain< mt::Typelist<X, XRF>, mt::Typelist<TL, TLList> >(ptr, size, x_release_functor, *this);
    }

    template <typename X>
    IOVectorChain< mt::Typelist<X, mt::NullType>, mt::Typelist<TL, TLList> >
    operator()(const X* ptr, size_t size)
    {
        return IOVectorChain< mt::Typelist<X, mt::NullType>, mt::Typelist<TL, TLList> >(ptr, size, mt::getNullType(), *this);
    }

    IOVectorChain< IOVectorBase, mt::Typelist<TL, TLList> >
    operator()(const IOVectorBase& rhs)
    {
        return IOVectorChain< IOVectorBase, mt::Typelist<TL, TLList> >(rhs, *this);
    }

private:
    const T* ptr_;
    size_t size_;
    RF release_functor_;
    const IOVectorChain<typename TLList::Head, typename TLList::Tail> tail_;
    mutable bool should_release_;
};

template <typename TL>
struct IOVectorChain<IOVectorBase, TL> : public IOVectorBase
{
    IOVectorChain(const IOVectorChain<IOVectorBase, TL>& rhs)
        : head_(rhs.head_), tail_(rhs.tail_), should_release_(rhs.getRelease())
    {
        rhs.setRelease();
    }

    IOVectorChain(const IOVectorBase& rhs, const IOVectorChain<typename TL::Head, typename TL::Tail>& obj)
        : head_(rhs), tail_(obj), should_release_(obj.getRelease())
    {
        obj.setRelease();
    }

    virtual void setRelease() const
    {
        should_release_ = false;
    }

    virtual bool getRelease() const
    {
        return should_release_;
    }

    virtual size_t setToVector(struct iovec* iovec_ptr, size_t skip_bytes = 0u) const
    {
        size_t offset = tail_.setToVector(iovec_ptr, skip_bytes);
        offset += head_.setToVector(iovec_ptr + offset, skip_bytes);
        return offset;
    }

    virtual ~IOVectorChain()
    {
    }

    template <typename X, typename XRF>
    IOVectorChain< mt::Typelist<X, XRF>, mt::Typelist<IOVectorBase, TL> >
    operator()(const X* ptr, size_t size, XRF x_release_functor)
    {
        return IOVectorChain< mt::Typelist<X, XRF>, mt::Typelist<IOVectorBase, TL> >(ptr, size, x_release_functor, *this);
    }

    template <typename X>
    IOVectorChain< mt::Typelist<X, mt::NullType>, mt::Typelist<IOVectorBase, TL> >
    operator()(const X* ptr, size_t size)
    {
        return IOVectorChain< mt::Typelist<X, mt::NullType>, mt::Typelist<IOVectorBase, TL> >(ptr, size, mt::getNullType(), *this);
    }

    IOVectorChain< IOVectorBase, mt::Typelist<IOVectorBase, TL> >
    operator()(const IOVectorBase& rhs)
    {
        return IOVectorChain< IOVectorBase, mt::Typelist<IOVectorBase, TL> >(rhs, *this);
    }

private:
    const IOVectorBase& head_;
    const IOVectorChain<typename TL::Head, typename TL::Tail> tail_;
    mutable bool should_release_;
};

template <>
struct IOVectorChain<mt::NullType, mt::NullType>
{
public:
    IOVectorChain()
        : should_release_(true)
    {}

    IOVectorChain(const IOVectorChain<mt::NullType, mt::NullType>& rhs)
        : should_release_(rhs.getRelease())
    {
        rhs.setRelease();
    }

    void setRelease() const
    {
        should_release_ = false;
    }

    bool getRelease() const
    {
        return should_release_;
    }

    virtual ~IOVectorChain()
    {}

    size_t doSetToVector(struct iovec*, size_t&, size_t skip_bytes) const
    {
        return skip_bytes;
    }

    template <typename T, typename RF>
    IOVectorChain< mt::Typelist<T, RF>, mt::Typelist<mt::NullType, mt::NullType> >
    operator()(const T* ptr, size_t size, RF release_functor)
    {
        return IOVectorChain< mt::Typelist<T, RF>, mt::Typelist<mt::NullType, mt::NullType> >(ptr, size, release_functor, *this);
    }

    template <typename T>
    IOVectorChain< mt::Typelist<T, mt::NullType>, mt::Typelist<mt::NullType, mt::NullType> >
    operator()(const T* ptr, size_t size)
    {
        return IOVectorChain< mt::Typelist<T, mt::NullType>, mt::Typelist<mt::NullType, mt::NullType> >(ptr, size, mt::getNullType(), *this);
    }

    template <typename X, typename Y>
    IOVectorChain< IOVectorBase, mt::Typelist<mt::NullType, mt::NullType> >
    operator()(const IOVectorBase& rhs)
    {
        return IOVectorChain< IOVectorBase, mt::Typelist<mt::NullType, mt::NullType> >(rhs, *this);
    }

    mutable bool should_release_;
};

typedef IOVectorChain<mt::NullType, mt::NullType> IOVec;


} // namespace cm
#endif // INC_CM_IO_VECTOR_H_
