
#include <unistd.h>

#include "gmock/gmock.h"

#include "mt_auto_ptr.h"
#include "mt_typelist_algo_utility.h"
#include "mt_shim_clear_memory.h"
#include "cm_vector_socket.h"
#include "cm_socket.h"

namespace {

#pragma GCC diagnostic ignored "-Weffc++"
struct StringBufferMock
{
    MOCK_CONST_METHOD0(getString, const char*());
};
#pragma GCC diagnostic warning "-Weffc++"

struct RFRelease
{
    RFRelease(StringBufferMock* mock_ptr)
        : mock_ptr_(mock_ptr)
    {}

    void operator()()
    {
        delete mock_ptr_;
    }

    StringBufferMock* mock_ptr_;
};

static const char string1[] = "hello";
static const char string2[] = " ";
static const char string3[] = "world ";
static const char string4[] = "!!!!";

static const char string_for_mock[] = "[good by cruel world --]";

TEST(CmVectorSocketTest, simple)
{
    StringBufferMock* mock = new StringBufferMock;

    EXPECT_CALL(*mock, getString())
        .WillOnce(::testing::Return(string_for_mock));

    const cm::IOVectorBase& iov
        = cm::IOVec()(string1, sizeof(string1) - 1)
                     (string2, sizeof(string2) - 1)
                     (string3, sizeof(string3) - 1)
                     (string4, sizeof(string4) - 1)
                     (mock->getString(), sizeof(string_for_mock) - 1, RFRelease(mock));

    int pipe_fds[2];
    int retval = pipe(pipe_fds);
    EXPECT_EQ(retval, 0);

    cm::VectorSocket write_socket(pipe_fds[1]);
    cm::Socket read_socket(pipe_fds[0]);

    size_t bytes_written = 0u;
    bool ret = write_socket.writev(bytes_written, iov);

    EXPECT_EQ(ret, true);

    size_t total_bytes_written = sizeof(string1) - 1 + sizeof(string2) - 1 + sizeof(string3) - 1 + sizeof(string4) - 1 + sizeof(string_for_mock) - 1;
    EXPECT_EQ(bytes_written, total_bytes_written);

    size_t bytes_read = 0u;
    char recv_buffer[1024];
    mt::clearMemory(recv_buffer);
    ret = read_socket.read(bytes_read, recv_buffer, sizeof(recv_buffer));

    EXPECT_EQ(ret, true);
    EXPECT_EQ(bytes_read, total_bytes_written);
}

TEST(CmVectorSocketTest, ownership_move_from_socket)
{
    cm::Socket original_sock(STDOUT_FILENO);

    cm::VectorSocket vec_sock(original_sock);

    const cm::IOVectorBase& iov = cm::IOVec()(string1, sizeof(string1) - 1)
                     (string2, sizeof(string2) - 1)
                     (string3, sizeof(string3) - 1);

    size_t bytes_written = 0u;
    vec_sock.writev(bytes_written, iov);
}

TEST(CmVectorSocketTest, ownership_move_from_socket_via_auto_ptr)
{
    mt::AutoPtr<cm::SocketIf> original_socket(new cm::Socket(STDOUT_FILENO));

    mt::AutoPtr<cm::VectorSocket> vec_sock(new cm::VectorSocket(*original_socket));

    const cm::IOVectorBase& iov = cm::IOVec()(string1, sizeof(string1) - 1)
                     (string2, sizeof(string2) - 1)
                     (string3, sizeof(string3) - 1);

    size_t bytes_written = 0u;
    vec_sock->writev(bytes_written, iov);

    mt::AutoPtr<cm::Socket> renewed_socket(new cm::Socket(*vec_sock));

    std::cout << std::endl;
    renewed_socket->write(bytes_written, "hello world(2)!!!", sizeof("hello world(2)!!!") - 1);
    renewed_socket->release();
}

TEST(CmVectorSocketTest, nested_iovector_handling)
{
    StringBufferMock* mock = new StringBufferMock;

    EXPECT_CALL(*mock, getString())
        .WillOnce(::testing::Return(string_for_mock));

    const cm::IOVectorBase& iov1
        = cm::IOVec()(string1, sizeof(string1) - 1)
                     (string2, sizeof(string2) - 1)
                     (string3, sizeof(string3) - 1)
                     (string4, sizeof(string4) - 1)
                     (mock->getString(), sizeof(string_for_mock) - 1, RFRelease(mock));

    const cm::IOVectorBase& iov2
        = cm::IOVec()(string1, sizeof(string1) - 1)
                     (string2, sizeof(string2) - 1)
                     (string3, sizeof(string3) - 1)
                     (string4, sizeof(string4) - 1);

    const cm::IOVectorBase& iov3
        = cm::IOVec()(string1, sizeof(string1) - 1)
                     (string2, sizeof(string2) - 1)
                     (iov1)(iov2);

    int pipe_fds[2];
    int retval = pipe(pipe_fds);
    EXPECT_EQ(retval, 0);

    cm::VectorSocket write_socket(pipe_fds[1]);
    cm::Socket read_socket(pipe_fds[0]);

    size_t bytes_written = 0u;
    bool ret = write_socket.writev(bytes_written, iov3);

    EXPECT_EQ(ret, true);

    size_t total_bytes_written = (sizeof(string1) - 1 + sizeof(string2) - 1) 
            + (sizeof(string1) - 1 + sizeof(string2) - 1 + sizeof(string3) - 1 + sizeof(string4) - 1 + sizeof(string_for_mock) - 1)
            + (sizeof(string1) - 1 + sizeof(string2) - 1 + sizeof(string3) - 1 + sizeof(string4) - 1);
    EXPECT_EQ(bytes_written, total_bytes_written);

    size_t bytes_read = 0u;
    char recv_buffer[1024];
    mt::clearMemory(recv_buffer);
    ret = read_socket.read(bytes_read, recv_buffer, sizeof(recv_buffer));

    EXPECT_EQ(ret, true);
    EXPECT_EQ(bytes_read, total_bytes_written);
}

} // namespace
