

#include <string>

#include "gmock/gmock.h"

#include "cm_thread_event_driven.h"
#include "cm_socket_mock.h"

namespace {

struct ThreadHandler;

struct ThreadHandlerArg
{
    ThreadHandlerArg(cm::SocketIf* sock)
        : obj_ptr(0), sock_ptr(sock)
    {}

    ThreadHandlerArg()
        : obj_ptr(0), sock_ptr(0)
    {}

    ThreadHandler* obj_ptr;
    cm::SocketIf* sock_ptr;

private:
    // Non coopyable
    ThreadHandlerArg(const ThreadHandlerArg& rhs);
    ThreadHandlerArg& operator=(const ThreadHandlerArg& rhs);
};

ThreadHandlerArg arg_for_noarg_thread;
cm::SocketMock mock_for_noarg_thread;

struct ThreadHandler
{
    ThreadHandler(ThreadHandlerArg* arg_ptr)
        : mock_(*(arg_ptr->sock_ptr))
    {
        ON_CALL(*this, handleIt(::testing::_))
            .WillByDefault(::testing::Invoke(this, &ThreadHandler::handleSocket));

        cm::Event::tsdInstance().addHandlerRead(*this, &ThreadHandler::handleIt, mock_);

        arg_ptr->obj_ptr = this;
    }

    ThreadHandler()
        : mock_(mock_for_noarg_thread)
    {
        ON_CALL(*this, handleIt(::testing::_))
            .WillByDefault(::testing::Invoke(this, &ThreadHandler::handleSocket));

        cm::Event::tsdInstance().addHandlerRead(*this, &ThreadHandler::handleIt, mock_);

        arg_for_noarg_thread.obj_ptr = this;
    }

    ~ThreadHandler()
    {
        cm::Event::tsdInstance().delHandlerRead(mock_);
    }

    static const char_t* getName() { return "test_thread"; }

    MOCK_METHOD1(handleIt, bool(cm::SocketIf& event_type));

    bool handleSocket(cm::SocketIf& event_type)
    {
        std::string recv_data;
        recv_data.resize(128u, '\0');
        size_t bytes_read = 0;
        bool is_success = event_type.read(bytes_read, &recv_data[0], recv_data.size());
        if (is_success) {
            std::cout << "received data = " << recv_data.c_str() << std::endl;
        }
        return true;
    }

private:
    cm::SocketIf& mock_;
};

TEST(CmThreadEventDriven, simple)
{
    cm::SocketMock mock;
    ThreadHandlerArg arg(&mock);

    {
        cm::ThreadEventDriven<ThreadHandler, ThreadHandlerArg> thread(&arg);

        {
            ::testing::InSequence dummy;
            EXPECT_CALL(*(arg.obj_ptr), handleIt(::testing::_))
                .Times(2);
        }

        std::string data_to_put("hello world");
        mock.invokeRead(&data_to_put[0], data_to_put.size());

        data_to_put = "good bye world";
        mock.invokeRead(&data_to_put[0], data_to_put.size());

        sleep(1);
    }
}

TEST(CmThreadEventDriven, simple_noarg)
{
    {
        cm::ThreadEventDriven<ThreadHandler> thread;

        {
            ::testing::InSequence dummy;
            EXPECT_CALL(*(arg_for_noarg_thread.obj_ptr), handleIt(::testing::_))
                .Times(2);
        }

        std::string data_to_put("hello world");
        mock_for_noarg_thread.invokeRead(&data_to_put[0], data_to_put.size());

        data_to_put = "good bye world";
        mock_for_noarg_thread.invokeRead(&data_to_put[0], data_to_put.size());

        sleep(1);
    }
}


} // namespace

