
#include <vector>
#include <sstream>

#include "gtest/gtest.h"
#include "mt_array_length.h"
#include "mt_option.h"
#include "mt_container_foreach.h"

namespace mt {

namespace {

#pragma GCC diagnostic ignored "-Weffc++"
template <typename TL>
class TestOptions : public Options<TL>
{
public:
    bool isSameArguments(const std::vector<std::string>& args)
    {
        EXPECT_EQ(this->arguments_.size(), args.size());

        uint arg_index = 0;
        CONTAINER_FOREACH(const std::string& item, this->arguments_) {
            EXPECT_STREQ(item.c_str(), args[arg_index].c_str());
            ++arg_index;
        }

        return true;
    }

    virtual ~TestOptions()
    {}
};
#pragma GCC diagnostic warning "-Weffc++"

struct OptionTest
{
/*  Sample option element definition
    Usage of this option
    ./sample_program -t hello 10     -> Indicates multiple inputs are splitted by the shell by space character ' '.
    ./sample_program --test hello 10 -> Some more verbose way to specify option

    ./sample_program -t "hello, world" 10
*/
    static const char* getOptionKey() { return "t"; }
    static const char* getVOptionKey() { return "test"; }

    typedef Uint2Type<2u> ArgNumber;
    typedef std::string ArgType1;
    typedef size_t ArgType2; 

    bool operator()(ArgType1& arg1, ArgType2& arg2, const std::string& in1, const std::string& in2)
    {
        arg1 = in1;
        std::istringstream ss(in2);
        ss >> arg2;
        return true;
    }
};

struct OptionTest2
{
    static const char* getOptionKey() { return "t2"; }
    static const char* getVOptionKey() { return "test2"; }

    typedef Uint2Type<1u> ArgNumber;
    typedef sint ArgType1;

    bool operator()(ArgType1& arg1, const std::string& in1)
    {
        std::istringstream ss(in1);
        ss >> arg1;
        return true;
    }
};

typedef mt::MakeTypelist<OptionTest>::Result OptionElements;

}

TEST(OptionTest, instantiation)
{
    typedef mt::MakeTypelist<OptionTest>::Result OptionElements;

    TestOptions<OptionElements> option;

    const char* arguments_table[] = {
        "-t", "hello", "10", "--help"
    };

    int argc = static_cast<int>(ARRAY_LENGTH(arguments_table));
    option.setCommandLineArgs(argc, arguments_table);

    std::vector<std::string> args;
    CONTAINER_FOREACH_CONST(const char* const& item, arguments_table) {
        args.push_back(item);
    }

    EXPECT_EQ(true, option.isSameArguments(args));
}

TEST(OptionTest, parse)
{
    typedef mt::MakeTypelist<OptionTest, OptionTest2>::Result OptionElements;

    TestOptions<OptionElements> option;

    const char* arguments_table[] = {
        "-t", "hello", "10", "--help", "--test2", "-20"
    };

    int argc = static_cast<int>(ARRAY_LENGTH(arguments_table));
    option.setCommandLineArgs(argc, arguments_table);

    std::string arg1;
    size_t arg2;
    option.getParam<OptionTest>(arg1, arg2);

    EXPECT_STREQ(arg1.c_str(), "hello");
    EXPECT_EQ(arg2, 10);

    sint arg;
    option.getParam<OptionTest2>(arg);
    EXPECT_EQ(arg, -20);
}

} // namespace mt


