
#ifndef INC_CM_IO_VECTOR_H_
#define INC_CM_IO_VECTOR_H_

#include "mt_static_assert.h"
#include "mt_member_detector.h"
#include "mt_typelist.h"
#include "mt_typelist_algo_utility.h"

namespace cm {

struct IOVectorBase
{
    virtual ~IOVectorBase()
    {}

    virtual size_t setToVector(struct iovec*, size_t skip_bytes = 0) const = 0;
};

template <typename TL, typename TLList>
struct IOVectorChain : IOVectorBase
{
public:
    typedef typename TL::Head T;
    typedef typename TL::Tail RF;

    static const size_t LAYERED_COUNT = IOVectorChain<typename TLList::Head, typename TLList::Tail>::LAYERED_COUNT + 1;

    IOVectorChain(const IOVectorChain<TL, TLList>& rhs)
        : ptr_(rhs.ptr_), size_(rhs.size_), release_functor_(rhs.release_functor_), tail_(rhs.tail_), should_release_(rhs.should_release_)
    {
        rhs.should_release_ = false;
    }

    IOVectorChain(const T* ptr, size_t size, RF release_functor, const IOVectorChain<typename TLList::Head, typename TLList::Tail>& obj)
        : ptr_(ptr), size_(size), release_functor_(release_functor), tail_(obj), should_release_(true)
    {}

    template <typename Functor>
    void performReleaseFunctor(Functor& func, typename mt::EnableIf<mt::HasFunctorOperator<Functor>::value>::Result* = 0)
    {
        release_functor_();
    }

    template <typename Functor>
    void performReleaseFunctor(Functor&, typename mt::EnableIf<!mt::HasFunctorOperator<Functor>::value>::Result* = 0)
    {
    }

    virtual ~IOVectorChain()
    {
        if (should_release_) {
            performReleaseFunctor(release_functor_);
        }
    }

    size_t setToVector(struct iovec* iovec_ptr, size_t skip_bytes = 0u) const
    {
        STATIC_ASSERT(LAYERED_COUNT > 0);
        size_t offset = 0;
        size_t skipped_bytes = doSetToVector(iovec_ptr, offset, skip_bytes);
        assert(skipped_bytes == 0);
        return offset;
    }

    size_t doSetToVector(struct iovec* iovec_ptr, size_t& offset, size_t skip_bytes) const
    {
        size_t skipped_bytes = tail_.doSetToVector(iovec_ptr, offset, skip_bytes);
        assert(skip_bytes >= skipped_bytes);

        if (size_ >= skipped_bytes) {
            char* ptr = const_cast<char*>(reinterpret_cast<const char*>(ptr_));
            iovec_ptr[offset].iov_base = reinterpret_cast<void*>(ptr + skipped_bytes);
            iovec_ptr[offset].iov_len = size_ - skipped_bytes;
            ++offset;
            return 0u;
        }
        return (skipped_bytes - size_);
    }

    template <typename X, typename XRF>
    IOVectorChain< mt::Typelist<X, XRF>, mt::Typelist<TL, TLList> >
    operator()(const X* ptr, size_t size, XRF x_release_functor)
    {
        return IOVectorChain< mt::Typelist<X, XRF>, mt::Typelist<TL, TLList> >(ptr, size, x_release_functor, *this);
    }

    template <typename X>
    IOVectorChain< mt::Typelist<X, mt::NullType>, mt::Typelist<TL, TLList> > operator()(const X* ptr, size_t size)
    {
        return IOVectorChain< mt::Typelist<X, mt::NullType>, mt::Typelist<TL, TLList> >(ptr, size, mt::getNullType(), *this);
    }

private:
    const T* ptr_;
    size_t size_;
    RF release_functor_;
    const IOVectorChain<typename TLList::Head, typename TLList::Tail> tail_;
    mutable bool should_release_;
};

template <>
struct IOVectorChain<mt::NullType, mt::NullType>
{
public:
    static const size_t LAYERED_COUNT = 0;

    IOVectorChain()
        : should_release_(true)
    {}

    IOVectorChain(const IOVectorChain<mt::NullType, mt::NullType>& rhs)
        : should_release_(rhs.should_release_)
    {
        rhs.should_release_ = false;
    }

    virtual ~IOVectorChain()
    {}

    size_t doSetToVector(struct iovec*, size_t&, size_t skip_bytes) const
    {
        return skip_bytes;
    }

    template <typename T, typename RF>
    IOVectorChain< mt::Typelist<T, RF>, mt::Typelist<mt::NullType, mt::NullType> >
    operator()(const T* ptr, size_t size, RF release_functor)
    {
        return IOVectorChain< mt::Typelist<T, RF>, mt::Typelist<mt::NullType, mt::NullType> >(ptr, size, release_functor, *this);
    }

    template <typename T>
    IOVectorChain< mt::Typelist<T, mt::NullType>, mt::Typelist<mt::NullType, mt::NullType> >
    operator()(const T* ptr, size_t size)
    {
        return IOVectorChain< mt::Typelist<T, mt::NullType>, mt::Typelist<mt::NullType, mt::NullType> >(ptr, size, mt::getNullType(), *this);
    }

    mutable bool should_release_;
};

typedef IOVectorChain<mt::NullType, mt::NullType> IOVec;


} // namespace cm
#endif // INC_CM_IO_VECTOR_H_
