
#include <string>

#include "gmock/gmock.h"

#include "cm_event.h"
#include "cm_thread.h"
#include "cm_socket_mock.h"

namespace {

struct FakeHandler
{
    FakeHandler(bool ret = true)
        : retvalue_(ret)
    {}

    bool handleIt(cm::SocketIf& event_type)
    {
        std::string recv_data;
        recv_data.resize(128u, '\0');
        size_t bytes_read = 0;
        bool is_success = event_type.read(bytes_read, &recv_data[0], recv_data.size());
        if (is_success) {
            std::cout << "received data = " << recv_data.c_str() << std::endl;
        }
        return retvalue_;
    }

    bool handleItConst(cm::SocketIf& event_type) const
    {
        std::string recv_data;
        recv_data.resize(128u, '\0');
        size_t bytes_read = 0;
        bool is_success = event_type.read(bytes_read, &recv_data[0], recv_data.size());
        if (is_success) {
            std::cout << "received data = " << recv_data.c_str() << std::endl;
        }
        return retvalue_;
    }

    bool retvalue_;
};

class MessageHandler
{
public:
    MessageHandler()
    {
        ON_CALL(*this, handleEvent(::testing::_))
            .WillByDefault(::testing::Invoke(&fake_, &FakeHandler::handleIt));
    }

    MOCK_METHOD1(handleEvent, bool(cm::SocketIf& event_type));

    FakeHandler fake_;
};

class MessageHandlerConst
{
public:
    MessageHandlerConst()
    {
        ON_CALL(*this, handleEventConst(::testing::_))
            .WillByDefault(::testing::Invoke(&fake_, &FakeHandler::handleItConst));
    }

    MOCK_CONST_METHOD1(handleEventConst, bool(cm::SocketIf& event_type));

    FakeHandler fake_;
};

TEST(CmEventTest, simplePendOperation)
{
    cm::Event& event = cm::Event::tsdInstance();

    cm::SocketMock event_type(10);
    MessageHandler handler;
    event.addHandlerRead(handler, &MessageHandler::handleEvent, event_type);
    
    std::string data_to_put("hello world");
    event_type.invokeRead(&data_to_put[0], data_to_put.size());

    data_to_put = "good bye world";
    event_type.invokeRead(&data_to_put[0], data_to_put.size());

    {
        ::testing::InSequence dummy;
        EXPECT_CALL(handler, handleEvent(::testing::_))
            .Times(1);

        FakeHandler false_fake_handler(false);
        EXPECT_CALL(handler, handleEvent(::testing::_))
            .WillOnce(::testing::Invoke(&false_fake_handler, &FakeHandler::handleIt));
    }

    event.pend();
    event.pend();

    event.delHandlerRead(event_type);
}

TEST(CmEventTest, simplePendOperationTowardConstMethod)
{
    cm::Event& event = cm::Event::tsdInstance();

    cm::SocketMock event_type(10);
    MessageHandlerConst handler;
    event.addHandlerRead(handler, &MessageHandlerConst::handleEventConst, event_type);
    
    std::string data_to_put("hello world");
    event_type.invokeRead(&data_to_put[0], data_to_put.size());

    data_to_put = "good bye world";
    event_type.invokeRead(&data_to_put[0], data_to_put.size());

    {
        ::testing::InSequence dummy;
        EXPECT_CALL(handler, handleEventConst(::testing::_))
            .Times(1);

        FakeHandler false_fake_handler(false);
        EXPECT_CALL(handler, handleEventConst(::testing::_))
            .WillOnce(::testing::Invoke(&false_fake_handler, &FakeHandler::handleItConst));
    }

    event.pend();
    event.pend();

    event.delHandlerRead(event_type);
}

struct ThreadArg
{
    ThreadArg(cm::SocketMock (&etype)[4], MessageHandler& mhandler)
        : event_type(etype), message_handler(mhandler)
    {}

    cm::SocketMock (&event_type)[4];
    MessageHandler& message_handler;
};

class ThreadFunc
{
public:
    ThreadFunc(ThreadArg* arg)
        : event_(cm::Event::tsdInstance()), event_type_(arg->event_type), handler(arg->message_handler)
    {
        for (unsigned int i = 0; i != sizeof(event_type_)/sizeof(0[event_type_]); ++i) {
            event_.addHandlerRead(handler, &MessageHandler::handleEvent, event_type_[i]);
        }
    }

    ~ThreadFunc()
    {
        for (unsigned int i = 0; i != sizeof(event_type_)/sizeof(0[event_type_]); ++i) {
            event_.delHandlerRead(event_type_[i]);
        }
    }

    void run()
    {
        for (unsigned int count = 0; count < 2; ++count) {
            event_.pend();
        }
    }

private:
    cm::Event& event_;
    cm::SocketMock (&event_type_)[4];
    MessageHandler& handler;
};

TEST(CmEventTest, simplePendOperationAmongDifferentThreads)
{
    cm::Thread<ThreadFunc, ThreadArg> thread("cm_event_thread");

    cm::SocketMock event_type[4];
    MessageHandler handler;
    ThreadArg arg(event_type, handler);

    {
        {
            ::testing::InSequence dummy;
            EXPECT_CALL(handler, handleEvent(::testing::_))
                .Times(1);

            FakeHandler false_fake_handler(false);
            EXPECT_CALL(handler, handleEvent(::testing::_))
                .WillOnce(::testing::Invoke(&false_fake_handler, &FakeHandler::handleIt));
        }

        thread.create(&arg);

        std::string data_to_put1("hello world");
        std::string data_to_put2("good bye world");

        event_type[0].invokeRead(&data_to_put1[0], data_to_put1.size());
        event_type[0].invokeRead(&data_to_put2[0], data_to_put2.size());

        thread.join();
    }
}

TEST(CmEventTest, multipleEventTypesPendOperation)
{
    cm::Thread<ThreadFunc, ThreadArg> thread("cm_event_thread");

    cm::SocketMock event_type[4];
    MessageHandler handler;
    ThreadArg arg(event_type, handler);

    {
        {
            ::testing::InSequence dummy;
            EXPECT_CALL(handler, handleEvent(::testing::_))
                .Times(1);

            EXPECT_CALL(handler, handleEvent(::testing::_))
                .Times(1);

            FakeHandler false_fake_handler(false);
            EXPECT_CALL(handler, handleEvent(::testing::_))
                .WillOnce(::testing::Invoke(&false_fake_handler, &FakeHandler::handleIt));
        }

        thread.create(&arg);

        std::string data_to_put1("hello world");
        std::string data_to_put2("good bye world");

        event_type[0].invokeRead(&data_to_put1[0], data_to_put1.size());
        event_type[1].invokeRead(&data_to_put2[0], data_to_put2.size());
        event_type[0].invokeRead(&data_to_put2[0], data_to_put2.size());

        thread.join();
    }
}

} // namespace

