/***************************************************************************/
/** @file       Font_Texture_Raw.inl
    @brief      
    @author     shom
    @internal
****************************************************************************/

#include "Font_Texture_Raw.h"

#include "core/graphic/2d/image_board.h"


using namespace font_def;


/***************************************************************************
	CFontCharacter_Base
****************************************************************************/

CFontCharacter_Base::CFontCharacter_Base()
:
m_p_iboard_font( CREATE_SPTR( cimage_board ) )//,
{
}

void	CFontCharacter_Base::Initialize( const TCHAR* sz_path_texture )
{
	m_p_iboard_font->Initialize( sz_path_texture );
}

void	CFontCharacter_Base::Finalize()
{
	m_p_iboard_font->Finalize();
}

void	CFontCharacter_Base::SetContext( const FontContext_t& in_context )
{
	m_p_iboard_font->SetSize( in_context.v2Size );
	m_p_iboard_font->SetColor( in_context.uColor );
}

void	CFontCharacter_Base::Draw( u32 in_character, const vec2f& in_v2_pos )
{
	///--
	vec2f v2_uv_start, v2_uv_end;
	calc_texture_uv( in_character, v2_uv_start, v2_uv_end );
	m_p_iboard_font->SetUv( v2_uv_start, v2_uv_end );

	m_p_iboard_font->SetPos( in_v2_pos );

	m_p_iboard_font->Update();
	///--

	///--
	m_p_iboard_font->Draw();
	///--
}


/***************************************************************************
	CFontCharacter_Ascii
****************************************************************************/

CFontCharacter_Ascii::CFontCharacter_Ascii()
{
}

void	CFontCharacter_Ascii::Initialize()
{
	CFontCharacter_Base::Initialize( _T( "texture/font/font_ascii.png" ) );
}

/*------------------------------------------------------------------------*/
namespace
{
	const u32 sc_nof_chara_on_column_in_texture_ascii = 16;
	const u32 sc_nof_chara_on_row_in_texture_ascii = 16;
} 
/*------------------------------------------------------------------------*/
void	CFontCharacter_Ascii::calc_texture_uv(
	u32 in_character,
	vec2f& out_v2_start, vec2f& out_v2_end
	)
{
	const u32 idx_u = in_character % sc_nof_chara_on_column_in_texture_ascii;
	const u32 idx_v = in_character / sc_nof_chara_on_column_in_texture_ascii;

	const f32 u_start = SCAST<f32>( idx_u     ) / SCAST<f32>( sc_nof_chara_on_column_in_texture_ascii );
	const f32 v_start = SCAST<f32>( idx_v     ) / SCAST<f32>( sc_nof_chara_on_row_in_texture_ascii );
	const f32 u_end   = SCAST<f32>( idx_u + 1 ) / SCAST<f32>( sc_nof_chara_on_column_in_texture_ascii );
	const f32 v_end   = SCAST<f32>( idx_v + 1 ) / SCAST<f32>( sc_nof_chara_on_row_in_texture_ascii );

	out_v2_start.x = u_start;
	out_v2_start.y = v_start;
	out_v2_end.x = u_end;
	out_v2_end.y = v_end;
}


/***************************************************************************
	CFontCharacter_Sjis
****************************************************************************/

CFontCharacter_Sjis::CFontCharacter_Sjis()
{
}

void	CFontCharacter_Sjis::Initialize()
{
	CFontCharacter_Base::Initialize( _T( "texture/font/hiragana_katakana.png" ) );
}

/*------------------------------------------------------------------------*/
namespace
{
	const u32 sc_character_a_hira = 0x829f/**/;
	const u32 sc_character_a_kata = 0x8340/*@*/;	

	const u32 sc_nof_chara_on_column_in_texture_sjis = 16;
	const u32 sc_nof_chara_on_row_in_texture_sjis = 16;

	const u32 sc_character_onbiki = 0x815b/*[*/;
	const u32 idx_onbiki =
		sc_nof_chara_on_column_in_texture_sjis * sc_nof_chara_on_row_in_texture_sjis - 1;
} 
/*------------------------------------------------------------------------*/
void	CFontCharacter_Sjis::calc_texture_uv(
	u32 in_character,
	vec2f& out_v2_start, vec2f& out_v2_end
	)
{
	const b32 b_onbiki = ( in_character == sc_character_onbiki );
	const b32 b_hiragana = in_character < sc_character_a_kata;
	const u32 idx_offset =
		( b_onbiki ) ? idx_onbiki :
		  ( in_character -
		    ( b_hiragana ? sc_character_a_hira : sc_character_a_kata )
			);

	const u32 idx_u = idx_offset % sc_nof_chara_on_column_in_texture_sjis;
	const u32 idx_v = idx_offset / sc_nof_chara_on_column_in_texture_sjis + ( b_hiragana ? 0 : 6 );

	const f32 u_start = SCAST<f32>( idx_u     ) / SCAST<f32>( sc_nof_chara_on_column_in_texture_sjis );
	const f32 v_start = SCAST<f32>( idx_v     ) / SCAST<f32>( sc_nof_chara_on_row_in_texture_sjis );
	const f32 u_end   = SCAST<f32>( idx_u + 1 ) / SCAST<f32>( sc_nof_chara_on_column_in_texture_sjis );
	const f32 v_end   = SCAST<f32>( idx_v + 1 ) / SCAST<f32>( sc_nof_chara_on_row_in_texture_sjis );

	out_v2_start.x = u_start;
	out_v2_start.y = v_start;
	out_v2_end.x = u_end;
	out_v2_end.y = v_end;
}


/***************************************************************************
	CFont_Texture_Raw
****************************************************************************/

CFont_Texture_Raw::CFont_Texture_Raw()
:
m_p_character_ascii( CREATE_SPTR( CFontCharacter_Ascii ) ),
m_p_character_sjis( CREATE_SPTR( CFontCharacter_Sjis ) )//,
{
}

void	CFont_Texture_Raw::Initialize()
{
	m_p_character_ascii->Initialize();
	m_p_character_sjis->Initialize();
}

void	CFont_Texture_Raw::Finalize()
{
	m_p_character_ascii->Finalize();
	m_p_character_sjis->Finalize();
}

void	CFont_Texture_Raw::Draw( const char* const in_sz_msg, const FontContext_t& in_context )
{
	m_p_character_ascii->SetContext( in_context );
	m_p_character_sjis->SetContext( in_context );

	u32 idx = 0;
	u32 nof_char = 0;
	while( in_sz_msg[ idx ] != '\0' )
	{
		//--
		const f32 pos_x_offset = in_context.v2Size.x * SCAST<f32>( nof_char );
		const f32 pos_x = SCAST<f32>( in_context.uPosX ) + pos_x_offset;

		const vec2f	v2_pos( pos_x, SCAST<f32>( in_context.uPosY ) );
		//--

		if( ( in_sz_msg[ idx ] & 0xff ) < 0x80 )
			//ascii
		{
			m_p_character_ascii->Draw( ( in_sz_msg[ idx ] & 0xff ), v2_pos );

			idx += 1;
		}
		else
			//sjis
		{			
			const u32 character =
				( ( in_sz_msg[ idx ] & 0xff ) << 8 ) + ( in_sz_msg[ idx + 1 ] & 0xff );
	
			m_p_character_sjis->Draw( character, v2_pos );
	
			idx += 2;
		}

		++nof_char;
	}
}
