/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package textkeymatcher.entity;

import java.util.ArrayList;
import java.util.List;

/**
 * データソースごとの値のリスト.<br>
 *
 * @author seraphy
 */
public class RowValues {
    
    /**
     * データとしてNULLが登録されたことを示す.<br>
     */
    public static final Object NULL_VALUE = new Object() {
        @Override
        public String toString() {
            return "<NULL>";
        }
    };

    /**
     * データソースごとのデータを入れる配列.<br>
     * データソースごとに添字を指定する.<br>
     * 単一データであれば文字列が入り、複数データであれば文字列のリストオブジェクトが入る.<br>
     * 該当がなければ、nullとなる.<br>
     */
    private Object[] datas;

    /**
     * データソース数を指定して構築する.
     * @param dataWidth データソース数
     */
    public RowValues(int dataWidth) {
        this.datas = new Object[dataWidth];
    }

    /**
     * データソース数を取得する
     * @return データソース数
     */
    public int getDataWidth() {
        return datas.length;
    }

    /**
     * データソースと、値を指定してデータをセットする.<br>
     * @param dataIndex データソース添字
     * @param value データ、null可
     */
    public void add(int dataIndex, String value) {
        Object val;
        if (value == null) {
            val = NULL_VALUE;
        } else {
            val = value;
        }
        
        if (datas[dataIndex] == null) {
            // 新規の場合
            datas[dataIndex] = val;
        
        } else if (datas[dataIndex] instanceof List) {
            // すでに2要素以上ある場合
            @SuppressWarnings(value = "unchecked")
            List<Object> lst = (List<Object>) datas[dataIndex];
            lst.add(val);

        } else {
            // すでに存在する場合はリストにつめ直す.
            assert datas[dataIndex] instanceof String;
            List<Object> tmp = new ArrayList<Object>();
            tmp.add(datas[dataIndex]);
            tmp.add(val);
            datas[dataIndex] = tmp;
        }
    }

    /**
     * 各データソースの中の最大のデータ数.<br>
     * すべてのデータソースにひとつもデータがなければ0となる.<br>
     * @return 最大のデータ数
     */
    public int getMaxRowCount() {
        int mx = 0;
        int dataWidth = datas.length;
        for (int dataIndex = 0; dataIndex < dataWidth; dataIndex++) {
            int count = getRowCountAt(dataIndex);
            if (count > mx) {
                mx = count;
            }
        }
        return mx;
    }

    /**
     * 指定したデータソースのデータの個数を返す.
     * @param dataIndex 添字
     * @return データの個数、未登録ならば0
     */
    public int getRowCountAt(int dataIndex) {
        if (dataIndex < 0 || dataIndex >= datas.length) {
            // 範囲外、データがないので0を返す.
            return 0;
        }
        Object data = datas[dataIndex];
        if (data != null && data instanceof List) {
            // リスト格納であれば、要素数を返す
            @SuppressWarnings(value = "unchecked")
            List<Object> lst = (List<Object>) datas[dataIndex];
            return lst.size();
        }
        // 非nullで、リストでなければ単一要素なので1を返す
        if (data != null) {
            return 1;
        }
        // nullであればデータは未設定なので0
        return 0;
    }

    /**
     * 指定したデータソースの指定した番号のデータを取り出す.<br>
     * データ未登録である場合、および範囲外を指定した場合はnullが返される.<br>
     * データがnull値の場合は{@link #NULL_VALUE}が返される.<br>
     * @param column データソース番号
     * @param rowIndex データのインデックス、getCountで得られる個数分
     * @return データ、{@link #NULL_VALUE}、もしくはデータなしの場合および範囲外の場合はnull
     */
    public Object getRaw(int column, int rowIndex) {
        if (column < 0 || column >= datas.length) {
            return null;
        }
        Object data = datas[column];
        if (data == null) {
            // このカラムでは、データ未登録
            return null;
        }
        
        if (data instanceof List) {
            @SuppressWarnings(value = "unchecked")
            List<Object> lst = (List<Object>) data;
            if (rowIndex >= 0 && rowIndex < lst.size()) {
                // 複数データならば存在する行分だけ
                return lst.get(rowIndex);
            }
            // 範囲外であればデータなし
            return null;
        }

        if (rowIndex == 0) {
            // 単一データは先頭行のみ (データ未登録の場合はnull)
            return data;
        }
        // 単一データで次行以降はデータなし。
        return null;
    }

    /**
     * 指定したデータソースの指定した番号のデータを取り出す.<br>
     * データ未登録である場合、および範囲外を指定した場合はnullが返される.<br>
     * 登録データがNULL値である場合は空文字が返される.<bf>
     * @param column データソース番号
     * @param rowIndex データのインデックス、getCountで得られる個数分
     * @return データ、もしくは空白、データなしの場合および範囲外の場合はnull
     */
    public String get(int column, int rowIndex) {
        Object value = getRaw(column, rowIndex);
        assert value == null || value instanceof String || NULL_VALUE.equals(value);
        if (value != null) {
            if (NULL_VALUE.equals(value)) {
                return "";
            }
            return (String) value;
        }
        return null;
    }
    
}
