package kisscelltopng.kiss.io;

import java.io.BufferedInputStream;
import java.io.DataInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Arrays;

import kisscelltopng.kiss.types.KiSSCell;


/**
 * KiSSセルデータを読み込むためのローダー.<br>
 * 仕様は、 http://www2s.biglobe.ne.jp/~yav/kiss/kissgsj.html を参照した。<br>
 * (ckissアルファありのセル形式は実データから推定.)<br>
 * @author seraphy
 */
public class KiSSCellLoader {

	/**
	 * 入力ストリームからKiSSセルデータを取得する.
	 * @param is 入力ストリーム
	 * @return KiSSセルデータ
	 * @throws IOException 入力に失敗した場合
	 */
	public KiSSCell load(InputStream is) throws IOException {
		return load(is, true);
	}
	
	/**
	 * 入力ストリームからKiSSセルデータを取得する.<br>
	 * pixelRequiredがfalseの場合、pixelはロードされずnullとして設定されます.<br>
	 * @param is 入力ストリーム
	 * @param pixelRequired ピクセルをロードする場合はtrue
	 * @return KiSSセルデータ
	 * @throws IOException 入力に失敗した場合
	 */
	public KiSSCell load(InputStream is, boolean pixelRequired) throws IOException {
		if (is == null) {
			throw new IllegalArgumentException();
		}

		DataInputStream dis = new DataInputStream(new BufferedInputStream(is));
		
		byte[] lead = new byte[4];
		dis.read(lead);

		boolean indexedColor;
		int bitsPerPixel;
		int width;
		int height;
		int offsetX;
		int offsetY;
		
		if ( !Arrays.equals(lead, "KiSS".getBytes("UTF-8"))) {
			// KiSSシグネチャがない場合は旧形式
			indexedColor = true;
			bitsPerPixel = 4;
			offsetX = offsetY = 0;

			ByteBuffer bbuf = ByteBuffer.wrap(lead);
			bbuf.order(ByteOrder.LITTLE_ENDIAN);
			width = bbuf.getShort();
			height = bbuf.getShort();
			
		} else {
			// KiSSシグネチャあり
			byte typ = dis.readByte();
			if (typ == 0x20) {
				// kiss形式
				indexedColor = true;

			} else if (typ == 0x21) {
				// ckiss形式
				indexedColor = false;
			
			} else {
				throw new KiSSDataException("unknown cell-type: " + typ);
			}
			
			bitsPerPixel = dis.readByte();
			
			dis.skipBytes(2);
			
			byte[] buf = new byte[8];
			if (dis.read(buf) != buf.length) {
				throw new KiSSDataException("invalid cell size");
			}
			
			ByteBuffer bbuf = ByteBuffer.wrap(buf);
			bbuf.order(ByteOrder.LITTLE_ENDIAN);
			width = bbuf.getShort();
			height = bbuf.getShort();
			offsetX = bbuf.getShort();
			offsetY = bbuf.getShort();
			
			dis.skipBytes(16);
		}
		
		if (indexedColor && bitsPerPixel == 4) {
			if (width % 2 != 0) {
				width++; // 偶数にする.(奇数幅の場合、実データには1ピクセル分余分にデータが存在するため)
			}
		}
		
		int length = width * height;
		int[] pixels;
		if (pixelRequired) {
			pixels = new int[length];
			int index = 0;

			while (index < length) {
				// 4bit IndexedColor
				if (bitsPerPixel == 4) {
					byte c = dis.readByte();
					int a = (c >> 4) & 0x0f;
					int b = c & 0x0f;
					pixels[index++] = a;
					pixels[index++] = b; // 常に偶数幅にしているので範囲チェックは不要
					
				// 8bit IndexedColor
				} else if (bitsPerPixel == 8) {
					byte c = dis.readByte();
					pixels[index++] = c & 0xff;
					
				// alpha付き32bit TrueColor
				} else if (bitsPerPixel == 32) {
					byte b = dis.readByte();
					byte g = dis.readByte();
					byte r = dis.readByte();
					byte a = dis.readByte();
					
					int argb = ((a & 0xff) << 24) | ((r & 0xff) << 16) | ((g & 0xff) << 8) | (b & 0xff);
					pixels[index++] = argb;
					
				} else {
					throw new KiSSDataException("invalid bitsPerPixel: " + bitsPerPixel);
				}
			}
		} else {
			pixels = null;
		}
		
		KiSSCell cell = new KiSSCell();
		cell.setIndexedColor(indexedColor);
		cell.setBitsPerPixel(bitsPerPixel);
		cell.setWidth(width);
		cell.setHeight(height);
		cell.setOffsetX(offsetX);
		cell.setOffsetY(offsetY);
		cell.setPixels(pixels);
		
		return cell;
	}
	
}
