package charactermanaj.model.io;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.AbstractCollection;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;

import org.apache.tools.zip.ZipEntry;
import org.apache.tools.zip.ZipFile;

/**
 * 拡張子zipに対するZipアーカイブファクトリ.<br>
 * アーカイブの実装は、Apache Antによる実装を用いる.<br>
 * @author seraphy
 */
public class ZipArchivedFileFactory extends ArchivedFileFactory {

	@Override
	public ZipArchivedFile openArchive(File file) throws IOException {
		return new ZipArchivedFile(file);
	}
}

class ZipArchivedFile extends AbstractCollection<ZipArchivedEntry> implements ArchivedFile<ZipArchivedEntry> {

	protected String ZIP_ENCODING = System.getProperty("zip.encoding", "csWindows31J");
	
	protected File archivedFile;
	
	protected ZipFile zipFile;
	
	protected HashMap<String, ZipArchivedEntry> entries = new HashMap<String, ZipArchivedEntry>();
	
	public ZipArchivedFile(File archivedFile) throws IOException {
		if (archivedFile == null) {
			throw new IllegalArgumentException();
		}
		this.archivedFile = archivedFile;
		this.zipFile = new ZipFile(archivedFile, ZIP_ENCODING);
		
		enumFiles();
	}
	
	@SuppressWarnings("unchecked")
	protected void enumFiles() throws IOException {
		Enumeration<? extends ZipEntry> enm = zipFile.getEntries();
		while (enm.hasMoreElements()) {
			ZipEntry header = (ZipEntry) enm.nextElement();
			ZipArchivedEntry entry = new ZipArchivedEntry(header);
			entries.put(entry.getName(), entry);
		}
	}
	
	public String getName() {
		return archivedFile.getPath().replace('\\', '/');
	}
	
	public ArchivedEntry getEntry(String name) {
		return entries.get(name);
	}
	
	public InputStream getInputStream(ArchivedEntry entry) throws IOException {
		if (entry == null) {
			throw new IllegalArgumentException();
		}
		ZipArchivedEntry lhaEntry = entries.get(entry.getName());
		if (lhaEntry == null) {
			return null;
		}
		return zipFile.getInputStream(lhaEntry.getZipEntry());
	}
	
	public void close() throws IOException {
		zipFile.close();
	}
	
	@Override
	public Iterator<ZipArchivedEntry> iterator() {
		return entries.values().iterator();
	}
	
	@Override
	public int size() {
		return entries.size();
	}
}

class ZipArchivedEntry implements ArchivedEntry {
	
	protected ZipEntry zipEntry;
	
	protected String name;
	
	public ZipArchivedEntry(ZipEntry zipEntry) {
		if (zipEntry == null) {
			throw new IllegalArgumentException();
		}
		this.zipEntry = zipEntry;
		this.name = zipEntry.getName().replace('\\', '/');
	}
	
	public ZipEntry getZipEntry() {
		return zipEntry;
	}

	public String getName() {
		return name;
	}
	
	public long getTime() {
		return zipEntry.getTime();
	}
	
	public boolean isDirectory() {
		return getName().endsWith("/");
	}
}
