package charactermanaj.model.io;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.AbstractCollection;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;

import jp.gr.java_conf.dangan.util.lha.LhaFile;
import jp.gr.java_conf.dangan.util.lha.LhaHeader;

/**
 * lzh形式のアーカイブファクトリ.<br>
 * LHA解析には、jlha(石塚美珠瑠氏)の実装を使用する.<br>
 * jlhaがクラスパス上にない場合は実行時エラーとなる.<br>
 * @author seraphy
 */
public class LzhArchivedFileFactory extends ArchivedFileFactory {

	@Override
	public LzhArchivedFile openArchive(File file) throws IOException {
		return new LzhArchivedFile(file);
	}
}

class LzhArchivedFile extends AbstractCollection<LzhArchivedEntry> implements ArchivedFile<LzhArchivedEntry> {

	protected File archivedFile;
	
	protected LhaFile lhaFile;
	
	protected HashMap<String, LzhArchivedEntry> entries = new HashMap<String, LzhArchivedEntry>();
	
	public LzhArchivedFile(File archivedFile) throws IOException {
		if (archivedFile == null) {
			throw new IllegalArgumentException();
		}
		this.archivedFile = archivedFile;
		this.lhaFile = new LhaFile(archivedFile);
		
		enumFiles();
	}
	
	protected void enumFiles() throws IOException {
		Enumeration<LhaHeader> enm = lhaFile.entries();
		while (enm.hasMoreElements()) {
			LhaHeader header = enm.nextElement();
			LzhArchivedEntry entry = new LzhArchivedEntry(header);
			entries.put(entry.getName(), entry);
		}
	}
	
	public String getName() {
		return archivedFile.getPath().replace('\\', '/');
	}
	
	public ArchivedEntry getEntry(String name) {
		return entries.get(name);
	}
	
	public InputStream getInputStream(ArchivedEntry entry) throws IOException {
		if (entry == null) {
			throw new IllegalArgumentException();
		}
		LzhArchivedEntry lhaEntry = entries.get(entry.getName());
		if (lhaEntry == null) {
			return null;
		}
		return lhaFile.getInputStream(lhaEntry.getLhaHeader());
	}
	
	public void close() throws IOException {
		lhaFile.close();
	}
	
	@Override
	public Iterator<LzhArchivedEntry> iterator() {
		return entries.values().iterator();
	}
	
	@Override
	public int size() {
		return entries.size();
	}
}

class LzhArchivedEntry implements ArchivedEntry {
	
	protected LhaHeader lhaHeader;
	
	protected String name;
	
	public LzhArchivedEntry(LhaHeader lhaHeader) {
		if (lhaHeader == null) {
			throw new IllegalArgumentException();
		}
		this.lhaHeader = lhaHeader;
		this.name = lhaHeader.getPath().replace('\\', '/');
	}
	
	public LhaHeader getLhaHeader() {
		return lhaHeader;
	}

	public String getName() {
		return name;
	}
	
	public long getTime() {
		return lhaHeader.getLastModified().getTime();
	}
	
	public boolean isDirectory() {
		return getName().endsWith("/");
	}
}
