package charactermanaj.model.io;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Properties;

/**
 * アーカイブファィルへのアクセスを行うためのファクトリの抽象実装、およびアクセスメソッド.<br>
 * アーカイブファクトリは任意の拡張子をサポートするために拡張可能となっており、
 * 各jarファイルの「META-INF/archivedFileFactory.properties」ファイルにおいて、
 * 「拡張子=実装クラス名」の形式でサポート形式を記述する.<br>
 * 複数のjarがある場合、それらのプロパティはすべて連結して読み取られる.<br>
 * @author seraphy
 */
public abstract class ArchivedFileFactory {
	
	/**
	 * サポートしているアーカイブの定義.<br>
	 */
	public static final String RESOURCES = "META-INF/archivedFileFactory.properties";
	
	/**
	 * 「ディレクトリ」を示す定数.
	 */
	public static final String TYPEKEY_DIRECTORY = "*directory"; 
	
	/**
	 * サポートしているアーカイブの定義のマップ.<br>
	 * キーは拡張子、値はアーカイブファクトリの実装クラス名.<br>
	 */
	protected static LinkedHashMap<String, String> CONFIG;
	
	/**
	 * アーカイブファクトリのインスタンスのキャッシュ.<br>
	 * キーは実装クラス名、値はインスタンス.<br>
	 */
	protected static HashMap<String, ArchivedFileFactory> caches = new HashMap<String, ArchivedFileFactory>();

	/**
	 * アーカイブを開く.<br>
	 * @param file アーカイブへのパス
	 * @return アーカイブ
	 * @throws IOException 失敗
	 */
	public abstract ArchivedFile<? extends ArchivedEntry> openArchive(File file) throws IOException;
	
	/**
	 * ファイルを指定し、拡張子からアーカイブタイプを自動選択してアーカイブを開く.<br>
	 * @param file ファイル
	 * @return アーカイブ
	 * @throws IOException 失敗
	 */
	public static ArchivedFile<? extends ArchivedEntry> getArchivedFile(File file) throws IOException {
		ArchivedFileFactory factory = getFactory(file);
		return factory.openArchive(file);
	}

	/**
	 * 指定した拡張子に対するアーカイブファクトリを返す.<br>
	 * @param typ 拡張子
	 * @return アーカイブファクトリ
	 * @throws IOException 失敗
	 * @throws UnsupportedArchivedFileException サポートされていない拡張子
	 */
	public static ArchivedFileFactory getFactory(String typ) throws IOException {
		initConfig();
		ArchivedFileFactory factory = null;
		String cls = CONFIG.get(typ);
		if (cls != null) {
			factory = getFactoryForClass(cls);
		}
		if (factory == null) {
			throw new UnsupportedArchivedFileException(typ);
		}
		return factory;
	}
	
	/**
	 * 指定した拡張子をサポートするアーカイブファクトリがあるか?
	 * @param typ 拡張子
	 * @return サポートしている場合はtrue
	 */
	public static boolean isSupported(String typ) {
		if (typ == null) {
			return false;
		}
		try {
			ArchivedFileFactory factory = getFactory(typ);
			if (factory != null) {
				return true;
			}

		} catch (Exception ex) {
			// 無視する.
		}
		return false;
	}
	
	/**
	 * サポートしている拡張子を返す.<br>
	 * (「ディレクトリ」を示す定数は除く.)<br>
	 * @return サポートしている拡張子のコレクション
	 */
	public static List<String> getSupportedTypes() {
		ArrayList<String> result = new ArrayList<String>();
		for (String ext : CONFIG.keySet()) {
			if ( !ext.startsWith("*")) {
				result.add(ext);
			}
		}
		return result;
	}

	/**
	 * 指定したファイルをサポートするアーカイブファクトリがあるか?<br>
	 * 拡張子より自動判定される.<br>
	 * ディレクトリを指定した場合はディレクトリアーカイブファクトリとして判定される.<br>
	 * @param file ファイル
	 * @return サポートされる場合はtrue
	 */
	public static boolean isSupported(File file) {
		if (file == null) {
			throw new IllegalArgumentException();
		}
		if (file.isDirectory()) {
			return isSupported(TYPEKEY_DIRECTORY);
		}
		try {
			return isSupported(getExtention(file));

		} catch (IOException ex) {
			return false;
		}
	}

	/**
	 * 指定したファクトリクラスのインスタンスを返す.<br>
	 * まだ一度も作成されていない場合は作成され、それをキャッシュする.<br>
	 * 該当がない場合はnullが返される.<br>
	 * 初回、作成に失敗した場合はnullとしてキャッシュするため、二度目以降は試行せずにnullが返される.<br>
	 * @param className ファクトリのクラス名
	 * @return ファクトリのインスタンス
	 * @throws IOException 失敗
	 */
	protected static ArchivedFileFactory getFactoryForClass(String className) throws IOException {
		if (!caches.containsKey(className)) {
			ClassLoader cl = getClassLoader();
			try {
				Class<?> cls = cl.loadClass(className);
				ArchivedFileFactory factory = (ArchivedFileFactory) cls.newInstance();
				caches.put(className, factory);
				
			} catch (Exception ex) {
				caches.put(className, null);
				throw new UnsupportedArchivedFileException("configuration error: " + className, ex);
			}
		}
		return caches.get(className);
	}
	
	/**
	 * 指定したファイルの拡張子からファクトリクラスを自動認識し、そのファクトリのインスタンスを返す.<br>
	 * @param file ファイル
	 * @return アーカイブファクトリ
	 * @throws IOException 失敗
	 * @throws UnsupportedArchivedFileException サポートされていない拡張子
	 */
	public static ArchivedFileFactory getFactory(File file) throws IOException {
		if (file == null) {
			throw new IllegalArgumentException();
		}
		if (file.isDirectory()) {
			return getFactory(TYPEKEY_DIRECTORY);
		}
		return getFactory(getExtention(file));
	}
	
	/**
	 * ファイルから拡張子を取得する.
	 * @param file ファイル
	 * @return 拡張子
	 * @throws UnsupportedArchivedFileException 拡張子がない場合
	 */
	protected static String getExtention(File file) throws UnsupportedArchivedFileException {
		String lcName = file.getName().toLowerCase();
		int pos = lcName.indexOf('.');
		if (pos < 0) {
			throw new UnsupportedArchivedFileException(lcName);
		}
		String ext = lcName.substring(pos + 1);
		return ext;
	}

	/**
	 * 各jarファイルの「META-INF/archivedFileFactory.properties」ファイルをすべて連結して
	 * 読み取り、各拡張子ごとのアーカイブファクトリの実装クラスの定義を取得する.<br>
	 * @throws IOException 設定の読み取りに失敗した場合
	 */
	protected static void initConfig() throws IOException {
		if (CONFIG != null) {
			return;
		}
		LinkedHashMap<String, String> config = new LinkedHashMap<String, String>();

		ClassLoader cl = getClassLoader();
		Enumeration<URL> resources = cl.getResources(RESOURCES);
		while (resources != null && resources.hasMoreElements()) {
			URL resource = resources.nextElement();
			Properties prop = new Properties();
			InputStream is = resource.openStream();
			try {
				prop.load(is);
			} finally {
				is.close();
			}
			Enumeration<?> keys = prop.propertyNames();
			while (keys.hasMoreElements()) {
				String key = (String) keys.nextElement();
				config.put(key, prop.getProperty(key));
			}
		}
		ArchivedFileFactory.CONFIG = config;
	}
	
	/**
	 * クラスローダを取得する.<br>
	 * スレッドに関連づけられているクラスローダがあれば、それを用いる.<br>
	 * そうでなければ、このクラスのクラスローダを返す.<br>
	 * @return クラスローダ
	 */
	protected static ClassLoader getClassLoader() {
		return AccessController.doPrivileged(new PrivilegedAction<ClassLoader>() {
			public ClassLoader run() {
				ClassLoader cl = Thread.currentThread().getContextClassLoader();
				if (cl == null) {
					cl = ArchivedFileFactory.class.getClassLoader();
				}
				return cl;
			}
		});
	}
}
