package kisscelltopng.logic;

import java.awt.Dimension;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.imageio.ImageIO;

import kisscelltopng.kiss.converter.KiSSCellImageBuilder;
import kisscelltopng.kiss.io.KiSSFile;
import kisscelltopng.kiss.types.KiSSCell;
import kisscelltopng.kiss.types.KiSSPalette;
import kisscelltopng.ui.model.CellsTableItem;
import kisscelltopng.ui.model.CellsTablePaneModel;

import charactermanaj.model.io.ArchivedEntry;
import charactermanaj.model.io.ArchivedFile;
import charactermanaj.model.io.ArchivedFileFactory;

/**
 * KiSSセルから透過PNGに変換するためのロジック実装.<br>
 * @author seraphy
 */
public class KiSSDataConverter extends AbstractDataConverter {

	/**
	 * KiSSファイル
	 */
	private KiSSFile kissFile;
	
	/**
	 * セルのキャッシュ.
	 */
	private Caches<KiSSCell> cellCaches = new Caches<KiSSCell>();

	/**
	 * KiSSアーカイブファイルのロード.
	 */
	public void load(File file) throws IOException {
		if (kissFile != null) {
			throw new IllegalStateException("already opened.");
		}
		if (file == null) {
			throw new IllegalArgumentException();
		}
		ArchivedFile<? extends ArchivedEntry> archivedFile = ArchivedFileFactory.getArchivedFile(file);
		kissFile = new KiSSFile(archivedFile);
		cellCaches.clear();
	}

	/**
	 * KiSSアーカイブファイルのリリース.
	 */
	public void release(CellsTablePaneModel model) throws IOException {
		if (kissFile != null) {
			kissFile.close();
			kissFile = null;
		}
		if (model != null) {
			model.clear();
		}
		cellCaches.clear();
	}

	/**
	 * KiSSアーカイブファイルにあるパレットの一覧を取得する.
	 * @return パレットのリスト
	 * @throws IOException
	 */
	protected List<KiSSPalette> getPalettes() throws IOException {
		ArrayList<KiSSPalette> palettes = new ArrayList<KiSSPalette>();
		for (String paletteName : kissFile.getPaletteNames()) {
			KiSSPalette palette = kissFile.getPalette(paletteName);
			palette.setName(paletteName);
			palettes.add(palette);
		}
		return palettes;
	}
	
	/**
	 * KiSSファイルの内容を、セルテーブルの表示用モデルに設定する.<br>
	 * @param model セルテーブルのモデル
	 * @throws IOException 失敗
	 */
	public void updateCellsTablePaneModel(CellsTablePaneModel model) throws IOException {
		if (model == null) {
			throw new IllegalArgumentException();
		}
		if (kissFile == null) {
			throw new IllegalStateException("archive is not opened.");
		}
		
		// バレットの一覧をモデルに設定する.
		List<KiSSPalette> palettes = getPalettes();
		model.setPalettes(palettes);
		
		// セルの一覧を作成しモデルに設定する.
		ArrayList<CellsTableItem> items = new ArrayList<CellsTableItem>();

		for (String fileName : kissFile.getCellNames()) {
			// セルデータを読み取る。(ただしサイズなどの情報のみ。ピクセル情報は取得しない)
			KiSSCell cell = kissFile.getCell(fileName, false);
			CellsTableItem item = new CellsTableItem();
			item.setFileName(fileName);

			// ファイル名から拡張子を取り除いたものをデフォルトの名前とする.
			String cellName;
			int st = fileName.lastIndexOf('/');
			if (st < 0) {
				cellName = fileName;
			} else {
				cellName = fileName.substring(st + 1);
			}
			int en = cellName.lastIndexOf('.');
			if (en > 0) {
				cellName = cellName.substring(0, en);
			}
			
			item.setCellName(cellName);
			item.setSelected(true);
			
			if (cell.isIndexedColor()) {
				if (palettes.isEmpty()) {
					// セルファイルがあるがパレットが定義されていなければ
					// 表示できないのでスキップする.
					// TODO: WARNING表示するべきか。
					continue;
				}

				item.setIndexedColor(true);
				item.setPalette(palettes.get(0));
				item.setPaletteGroup(0);

			} else {
				item.setIndexedColor(false);
			}

			item.setSize(new Dimension(cell.getWidth(), cell.getHeight()));
			
			items.add(item);
		}
		
		Collections.sort(items);
		
		model.clear();
		model.add(items);
	}
	
	/**
	 * セルテーブルのアイテムを指定し、そのアイテムの表示画像を取得する.<br>
	 * @param item セルテーブルのアイテム
	 * @return 表示画像
	 */
	public BufferedImage getImage(CellsTableItem item) throws IOException {
		if (item == null) {
			return null;
		}
		String fileName = item.getFileName();
		KiSSCell cell = cellCaches.get(fileName);
		if (cell == null) {
			cell = kissFile.getCell(fileName, true);
			cellCaches.put(fileName, cell);
		}
		KiSSCellImageBuilder builder = new KiSSCellImageBuilder();
		builder.setPalette(item.getPalette());
		builder.setPaletteGroup(item.getPaletteGroup());
		return builder.build(cell);
	}
	
	@Override
	protected void writeImage(File outDir, String cellName, BufferedImage img)
			throws IOException {
		File outFile = new File(outDir, cellName + ".png");
		ImageIO.write(img, "png", outFile);
	}
	
}
