package kisscelltopng.kiss.converter;

import java.awt.image.BufferedImage;
import java.awt.image.DataBufferInt;
import java.awt.image.WritableRaster;

import kisscelltopng.kiss.types.KiSSCell;
import kisscelltopng.kiss.types.KiSSPalette;

/**
 * KiSSセルデータをBufferedImageに変換する.<br>
 * セルデータがパレットを必要とする場合、指定したパレットで描画する.<br>
 * @author seraphy
 */
public class KiSSCellImageBuilder {

	private KiSSPalette palette;
	
	private int paletteGroup;
	
	public void setPalette(KiSSPalette palette) {
		this.palette = palette;
	}
	
	public KiSSPalette getPalette() {
		return palette;
	}
	
	public void setPaletteGroup(int pageOfPalette) {
		this.paletteGroup = pageOfPalette;
	}
	
	public int getPaletteGroup() {
		return paletteGroup;
	}
	
	/**
	 * 指定したセルを現在のパレットに従ってBufferedImageとして構築する.<br>
	 * パレットを必要とするセルでパレットが指定されていないか、セル内のパレットインデックスが
	 * 実際のパレットと一致しない場合は実行時例外が発生する.<br>
	 * @param cell セル
	 * @return 構築されたBufferedImage
	 */
	public BufferedImage build(KiSSCell cell) {
		if (cell == null) {
			throw new IllegalArgumentException();
		}
		
		if (cell.isIndexedColor() && palette == null) {
			throw new IllegalStateException("palette required.");
		}

		int width = cell.getWidth();
		int height = cell.getHeight();
		
		BufferedImage img = new BufferedImage(width, height, BufferedImage.TYPE_INT_ARGB);
		WritableRaster raster = img.getRaster();
		DataBufferInt db = (DataBufferInt) raster.getDataBuffer();
		int[] data = db.getData();
		int[] pixels = cell.getPixels();
		
		if (data.length != pixels.length) {
			throw new RuntimeException("unmatched pixel-size");
		}

		if ( !cell.isIndexedColor()) {
			// インデックスカラーでなければ、そのまま転送
			System.arraycopy(pixels, 0, data, 0, pixels.length);

		} else {
			// インデックスカラーの適用.
			int[] pl = getPalette().getPaletteGroup(getPaletteGroup());
			final int mx = data.length;
			for (int idx = 0; idx < mx; idx++) {
				int c = pixels[idx];
				int argb;
				if (c <= 0) {
					argb = 0;

				} else {
					if (c >= pl.length) {
						// 範囲外の色インデックスの場合は他の色で代用する.
						c = c % pl.length;
					}
					argb = 0xff000000 | (pl[c] & 0xffffff);
				}
				data[idx] = argb;
			}
		}
		
		return img;
	}
	
}
