package charactermanaj.model.io;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.AbstractCollection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;

public class DirArchivedFileFactory extends ArchivedFileFactory {

	@Override
	public DirArchivedFile openArchive(File file) throws IOException {
		return new DirArchivedFile(file);
	}
	
}

class DirArchivedFile extends AbstractCollection<FileArchivedEntry> implements ArchivedFile<FileArchivedEntry> {
	
	private File dirSt;

	protected HashMap<String, FileArchivedEntry> entries = new HashMap<String, FileArchivedEntry>();
	
	public DirArchivedFile(File dir) throws IOException {
		if (dir == null) {
			throw new IllegalArgumentException("dir must none null.");
		}
		enumFiles(dir);
	}
	
	protected void enumFiles(File dirSt) throws IOException {
		LinkedList<File> stack = new LinkedList<File>();
		stack.push(dirSt);
		while ( !stack.isEmpty()) {
			File dir = stack.pop();
			if (dir.isDirectory()) {
				for (File file : dir.listFiles()) {
					if (file.isDirectory()) {
						stack.push(file);
					} else {
						addArchivedEntry(file);
					}
				}
			} else {
				addArchivedEntry(dir);
			}
		}
	}
	
	protected void addArchivedEntry(File file) {
		FileArchivedEntry entry = new FileArchivedEntry(file);
		entries.put(entry.getName(), entry);
	}

	public ArchivedEntry getEntry(String name) {
		return entries.get(name);
	}

	public InputStream getInputStream(ArchivedEntry entry) throws IOException {
		if (entry == null) {
			throw new IllegalArgumentException();
		}
		String name = entry.getName();
		FileArchivedEntry fileEntry = entries.get(name);
		if (fileEntry == null) {
			throw new IllegalArgumentException("no entry. " + entry);
		}
		return new FileInputStream(fileEntry.getFile());
	}

	public String getName() {
		return dirSt.getPath().replace('\\', '/');
	}

	public void close() throws IOException {
		// なにもしない.
	}

	@Override
	public Iterator<FileArchivedEntry> iterator() {
		return entries.values().iterator();
	}

	@Override
	public int size() {
		return entries.size();
	}
}

class FileArchivedEntry implements ArchivedEntry {
	
	protected File file;
	
	protected String name;
	
	public FileArchivedEntry(File file) {
		if (file == null) {
			throw new IllegalArgumentException();
		}
		this.file = file;
		this.name = file.getPath().replace('\\', '/');
	}

	public File getFile() {
		return file;
	}
	
	public String getName() {
		return name;
	}

	public long getTime() {
		return file.lastModified();
	}

	public boolean isDirectory() {
		return file.isDirectory();
	}
	
	@Override
	public String toString() {
		return getName();
	}
}
